package ru.yandex.chemodan.app.psbilling.core.promos;

import java.io.IOException;
import java.util.List;
import java.util.Map;

import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.node.ObjectNode;
import com.fasterxml.jackson.databind.node.TextNode;
import lombok.AllArgsConstructor;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.psbilling.core.texts.TextsManager;
import ru.yandex.commune.json.serialize.JsonParserException;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;

@AllArgsConstructor
public class PromoPayloadParser {
    private static final Logger logger = LoggerFactory.getLogger(PromoPayloadParser.class);
    private final TextsManager textsManager;

    public String processPayload(String payload, String language,
                                 boolean onlyParseAndAddMissingTankerKeys) {
        try {
            JsonNode jsonNode = new ObjectMapper().readTree(payload);
            processNode(jsonNode, language, onlyParseAndAddMissingTankerKeys);
            return jsonNode.toString();
        } catch (IOException ex) {
            throw new JsonParserException("unable to parse payload" + ex.getMessage(), ex);
        }
    }

    private void processNode(JsonNode jsonNode, String language,
                             boolean onlyParseAndAddMissingTankerKeys) {
        if (jsonNode.isValueNode()) {
            return;
        }

        if (jsonNode.isArray()) {
            for (JsonNode arrayItem : jsonNode) {
                processObject(arrayItem, language, onlyParseAndAddMissingTankerKeys);
            }
        } else if (jsonNode.isObject()) {
            processObject(jsonNode, language, onlyParseAndAddMissingTankerKeys);
        }
    }

    private void processObject(JsonNode node, String language,
                               boolean onlyParseAndAddMissingTankerKeys) {
        List<Map.Entry<String, JsonNode>> fields = Cf.wrap(node.fields()).toList();
        ObjectNode nodeObject = (ObjectNode) node;

        for (Map.Entry<String, JsonNode> jsonField : fields) {
            if (jsonField.getKey().endsWith("_tanker")) {
                JsonNode fieldValue = jsonField.getValue();
                if (!fieldValue.isObject()) {
                    throw new JsonParserException("unable to parse promo payload: tanker node value is not object");
                }
                String tankerProject = fieldValue.get("tanker_project").textValue();
                String tankerKeySet = fieldValue.get("tanker_key_set").textValue();
                String tankerKey = fieldValue.get("tanker_key").textValue();

                if (onlyParseAndAddMissingTankerKeys) {
                    textsManager.addTankerKeyIfNotExist(tankerProject, tankerKeySet, tankerKey);
                    continue;
                }

                String translationText = getTankerTranslation(language, tankerProject, tankerKeySet, tankerKey);
                nodeObject.set(jsonField.getKey().replace("_tanker", ""), new TextNode(translationText));
                nodeObject.remove(jsonField.getKey());
            }
            processNode(jsonField.getValue(), language, onlyParseAndAddMissingTankerKeys);
        }
    }

    private String getTankerTranslation(String language, String tankerProject,
                                        String tankerKeySet, String tankerKey) {
        try {
            Option<String> translation = textsManager.findTranslation(tankerProject, tankerKeySet, tankerKey)
                    .findByLangOrDefault(language);
            if (!translation.isPresent()) {
                logger.warn("unable to find translations for key {}", tankerKey);
            }
            return translation.orElse("");
        } catch (Exception ex) {
            throw new JsonParserException("unable to parse promo payload", ex);
        }
    }
}
