package ru.yandex.chemodan.app.psbilling.core.promos;

import java.util.UUID;

import lombok.AllArgsConstructor;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.SetF;
import ru.yandex.chemodan.app.psbilling.core.dao.products.ProductLineDao;
import ru.yandex.chemodan.app.psbilling.core.dao.promos.PromoPayloadDao;
import ru.yandex.chemodan.app.psbilling.core.dao.promos.PromoTemplateDao;
import ru.yandex.chemodan.app.psbilling.core.dao.promos.UserPromoDao;
import ru.yandex.chemodan.app.psbilling.core.entities.products.ProductLineEntity;
import ru.yandex.chemodan.app.psbilling.core.entities.promos.PromoApplicationType;
import ru.yandex.chemodan.app.psbilling.core.entities.promos.PromoTemplateEntity;
import ru.yandex.chemodan.app.psbilling.core.entities.promos.UserPromoEntity;
import ru.yandex.chemodan.app.psbilling.core.products.UserProductManager;
import ru.yandex.chemodan.app.psbilling.core.products.UserProductPrice;
import ru.yandex.chemodan.app.psbilling.core.tasks.execution.TaskScheduler;
import ru.yandex.chemodan.app.psbilling.core.texts.TextsManager;
import ru.yandex.chemodan.util.blackbox.UserTimezoneHelper;
import ru.yandex.chemodan.util.exception.NotFoundException;
import ru.yandex.inside.passport.PassportUid;
import ru.yandex.inside.utils.Language;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;

@AllArgsConstructor
public class PromoService {
    private static final Logger logger = LoggerFactory.getLogger(PromoService.class);

    private final PromoTemplateDao promoTemplateDao;
    private final UserPromoDao userPromoDao;
    private final TextsManager textsManager;
    private final TaskScheduler taskScheduler;
    private final UserTimezoneHelper userTimezoneHelper;
    private final PromoPayloadParser promoPayloadParser;
    private final ProductLineDao productLineDao;
    private final PromoPayloadDao promoPayloadDao;

    public PromoTemplate findByCode(String code) {
        return promoTemplateDao.findByCode(code).map(this::create)
                .orElseThrow(() -> new NotFoundException("promo with key " + code + " not found"));
    }

    public PromoTemplate findById(UUID promoId) {
        return create(promoTemplateDao.findByIdO(promoId)
                .orElseThrow(() -> new NotFoundException("promo with id " + promoId + " not found")));
    }

    public ListF<PromoTemplate> getReadyForUsingPromos(Option<PassportUid> uid) {
        return findAllWithUidUsageCached(uid).filter(p -> p.canBeUsedForUid(uid));
    }

    public ListF<PromoTemplate> findAllWithUidUsageCached(Option<PassportUid> uid) {
        MapF<PromoTemplateEntity, SetF<UUID>> allWithProductLines = promoTemplateDao.findOnlyB2cWithProductLines();
        return mapPromosWithUidUsageCached(allWithProductLines, uid);
    }

    private PromoTemplate create(PromoTemplateEntity entity) {
        switch (entity.getApplicationArea()) {
            case GLOBAL:
            case GLOBAL_B2C:
                return new GlobalPromoTemplate(entity, userPromoDao, textsManager, promoTemplateDao, userTimezoneHelper,
                        promoPayloadParser, productLineDao);
            case PER_USER:
                return new PerUserPromoTemplate(entity, promoTemplateDao, taskScheduler, userPromoDao, textsManager,
                        userTimezoneHelper, promoPayloadParser, productLineDao);
            default:
                throw new IllegalStateException(entity.getApplicationArea().toString());
        }
    }

    private PromoTemplate create(PromoTemplateEntity entity, SetF<UUID> lineIds,
                                 Option<UserPromoEntity> userPromoEntity) {
        switch (entity.getApplicationArea()) {
            case GLOBAL:
            case GLOBAL_B2C:
                return new GlobalPromoTemplate(entity, userPromoDao, textsManager, lineIds, userPromoEntity,
                        userTimezoneHelper, promoPayloadParser, productLineDao);
            case PER_USER:
                return new PerUserPromoTemplate(entity, lineIds, userPromoEntity, taskScheduler, userPromoDao,
                        textsManager, userTimezoneHelper, promoPayloadParser, productLineDao);
            default:
                throw new IllegalStateException(entity.getApplicationArea().toString());
        }
    }


    private ListF<PromoTemplate> findPromoTemplatesByProductLinesWithUserUsageCached(
            ListF<UUID> productLines, PassportUid uid) {
        MapF<PromoTemplateEntity, SetF<UUID>> promosWithLines = promoTemplateDao.findOnlyB2cByProductLines(productLines);

        return mapPromosWithUidUsageCached(promosWithLines, Option.of(uid));
    }

    private ListF<PromoTemplate> mapPromosWithUidUsageCached(
            MapF<PromoTemplateEntity, SetF<UUID>> entitiesWithLineIds, Option<PassportUid> uid) {

        MapF<UUID, UserPromoEntity> userPromosByPromoTemplateId = uid.map(u -> {
            ListF<UserPromoEntity> userPromos = userPromoDao.findUserPromos(u);
            return userPromos.toMapMappingToKey(UserPromoEntity::getPromoTemplateId);
        }).orElse(Cf.map());

        return entitiesWithLineIds.mapEntries((p, lines) -> create(p, lines,
                userPromosByPromoTemplateId.getO(p.getId())));
    }


    public boolean activatePromoForUser(PassportUid uid, UUID promoId) {
        return findById(promoId).activatePromoForUser(uid, false, false);
    }

    public boolean activatePromoForUser(PassportUid uid, String promoCode, boolean activateIfUsed, boolean sendEmail) {
        return findByCode(promoCode)
                .activatePromoForUser(uid, activateIfUsed, sendEmail);
    }

    //returns code of promo template to be used
    public Option<String> setPromoUsedState(
            UserProductManager userProductManager,
            PassportUid uid,
            UUID productPriceId
    ) {
        UserProductPrice price = userProductManager.findPrice(productPriceId);
        UUID productId = price.getPeriod().getUserProductId();

        ListF<ProductLineEntity> regularLines = userProductManager.getUserAvailableProductLinesByProduct(
                uid, productId, false);
        if (regularLines.isNotEmpty()) {
            logger.info("found regular promo line {} with required product {}. No promo will be used",
                    regularLines.first(), productId);
            return Option.empty();
        }
        ListF<UUID> productLines = userProductManager.getUserAvailableProductLinesByProduct(uid, productId, true)
                .map(ProductLineEntity::getId);

        ListF<PromoTemplate> canBeUsedPromos = findPromoTemplatesByProductLinesWithUserUsageCached(productLines, uid)
                .filter(p -> p.canBeUsedForUid(uid))
                .sorted(PromoTemplate.topMultipleTimeThenByUntilDate(Option.of(uid)));

        if (canBeUsedPromos.isEmpty()) {
            logger.info("no promo found for user {} to be used", uid);
            return Option.empty();
        }

        Option<PromoTemplate> canUseMultiplePromo = canBeUsedPromos
                .find(x -> x.getApplicationType() == PromoApplicationType.MULTIPLE_TIME);

        if (canUseMultiplePromo.isPresent()) {
            logger.info("skip setting promo used status for user {} " +
                    "cause got active promo which can be used multiple times", uid);
            return canUseMultiplePromo.map(PromoTemplate::getCode);
        }

        SetF<UUID> promoProductLines = canBeUsedPromos.flatMap(PromoTemplate::getProductLineIds).unique();
        Option<UUID> regularProductLine = productLines.find(lineId -> !promoProductLines.containsTs(lineId));
        if (regularProductLine.isPresent()) {
            logger.info("found regular promo line {} with required product {}. No promo will be used",
                    regularProductLine, productId);
            return Option.empty();
        }

        Option<PromoTemplate> toBeUsedPromo = canBeUsedPromos.firstO();

        toBeUsedPromo.ifPresent(p -> p.markUsedForUser(uid));
        return toBeUsedPromo.map(PromoTemplate::getCode);
    }

    public ListF<PromoTemplate> getFuturePromos(Option<PassportUid> uidO) {
        return findAllWithUidUsageCached(uidO).filter(p -> p.canBeUsedForUidInFuture(uidO));
    }

    public void setPayload(String promoKey, String promoPayloadType,
                           Integer promoPayloadVersion, String promoPayloadContent) {
        PromoTemplateEntity promoTemplate = promoTemplateDao.findByCode(promoKey)
                .orElseThrow(() -> new NotFoundException("Promo " + promoKey + " not found"));

        // парсим и добавляем несуществующие ключи в tanker_keys
        promoPayloadParser.processPayload(promoPayloadContent,
                Language.RUSSIAN.value(), true);

        promoPayloadDao.createOrUpdate(PromoPayloadDao.InsertData.builder()
                .promoId(promoTemplate.getId())
                .payloadType(promoPayloadType)
                .content(promoPayloadContent)
                .version(promoPayloadVersion).build());
    }
}
