package ru.yandex.chemodan.app.psbilling.core.promos.groups;

import java.util.UUID;

import lombok.extern.slf4j.Slf4j;
import org.joda.time.Instant;

import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.SetF;
import ru.yandex.bolts.collection.Tuple2;
import ru.yandex.bolts.function.Function0;
import ru.yandex.chemodan.app.psbilling.core.dao.CreatedOrExistResult;
import ru.yandex.chemodan.app.psbilling.core.dao.promos.group.GroupPromoDao;
import ru.yandex.chemodan.app.psbilling.core.entities.groups.Group;
import ru.yandex.chemodan.app.psbilling.core.entities.promos.PromoApplicationType;
import ru.yandex.chemodan.app.psbilling.core.entities.promos.PromoTemplateEntity;
import ru.yandex.chemodan.app.psbilling.core.entities.promos.group.GroupPromoEntity;
import ru.yandex.chemodan.app.psbilling.core.entities.promos.group.GroupPromoStatusType;
import ru.yandex.chemodan.util.blackbox.UserTimezoneHelper;
import ru.yandex.chemodan.util.exception.BadRequestException;

@Slf4j
public class GlobalB2bPromoTemplate extends AbstractGroupPromoTemplate {

    public GlobalB2bPromoTemplate(PromoTemplateEntity promoTemplate, UserTimezoneHelper userTimezoneHelper,
                                  Function0<SetF<UUID>> provideProductLine, GroupPromoDao groupPromoDao,
                                  Option<Tuple2<Group, GroupPromoEntity>> warmingCache) {
        super(promoTemplate, userTimezoneHelper, provideProductLine, groupPromoDao, warmingCache);
    }

    public GlobalB2bPromoTemplate(PromoTemplateEntity promoTemplate, UserTimezoneHelper userTimezoneHelper,
                                  Function0<SetF<UUID>> provideProductLine, GroupPromoDao groupPromoDao) {
        super(promoTemplate, userTimezoneHelper, provideProductLine, groupPromoDao);
    }

    @Override
    public boolean canBeUsed(Option<Group> group) {
        if (getApplicationType() == PromoApplicationType.MULTIPLE_TIME || !group.isPresent()) {
            // если акция мультиактивизационная или у нас нет group для проверки состояния по пользователю, то смотрим
            // доступна ли сама акция
            return isActive();
        }
        // у глобальных акций привязка к пользователю говорит только о том, что она уже использовалась
        // поэтому если привязки нет, то опять же смотрим на статус доступности самой акции
        return getPromoEntity(group.get()).map(GroupPromoEntity::isActive).orElseGet(this::isActive);
    }

    @Override
    public Option<Instant> canBeUsedUntilDate(Option<Group> group) {
        if (!canBeUsed(group)) {
            throw new IllegalStateException("cannot be used " + this);
        }

        return getToDate();
    }

    @Override
    public void markUsed(Group group) {
        log.info("set global promo {} used for group {}", getId(), group);
        Instant from = getPromoFromDate();
        Option<Instant> to = calcPromoToDate(group.getOwnerUid(), from);

        CreatedOrExistResult<GroupPromoEntity> result = groupPromoDao.createIfNotExist(
                GroupPromoDao.InsertData.builder()
                        .groupId(group.getId())
                        .promoTemplateId(getId())
                        .fromDate(from)
                        .toDate(to)
                        .status(GroupPromoStatusType.USED)
                        .build()
        );

        if (!result.isCreated()) {
            throw new IllegalStateException("Group promo for globalB2b promo " + getCode() + " for groupId " + group + " already exist");
        }
    }

    @Override
    public Option<GroupPromoEntity> activatePromo(Group group, boolean force, Function0<GroupPromoEntity> afterCallback) {
        throw new BadRequestException("Unable to activate globalB2b promo " + getCode() + " for groupId " + group.getId());
    }

    @Override
    public boolean isAvailableFor(Group groupPromoUsage, boolean force) {
        return false;
    }
}
