package ru.yandex.chemodan.app.psbilling.core.promos.groups;

import java.util.NoSuchElementException;
import java.util.UUID;

import lombok.extern.slf4j.Slf4j;
import org.joda.time.Instant;

import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.SetF;
import ru.yandex.bolts.collection.Tuple2;
import ru.yandex.bolts.function.Function0;
import ru.yandex.chemodan.app.psbilling.core.dao.CreatedOrExistResult;
import ru.yandex.chemodan.app.psbilling.core.dao.promos.group.GroupPromoDao;
import ru.yandex.chemodan.app.psbilling.core.entities.groups.Group;
import ru.yandex.chemodan.app.psbilling.core.entities.promos.PromoTemplateEntity;
import ru.yandex.chemodan.app.psbilling.core.entities.promos.group.GroupPromoEntity;
import ru.yandex.chemodan.app.psbilling.core.entities.promos.group.GroupPromoStatusType;
import ru.yandex.chemodan.util.blackbox.UserTimezoneHelper;
import ru.yandex.chemodan.util.exception.BadRequestException;
import ru.yandex.misc.lang.Validate;

@Slf4j
public class PerGroupPromoTemplate extends AbstractGroupPromoTemplate {

    public PerGroupPromoTemplate(PromoTemplateEntity promoTemplate, UserTimezoneHelper userTimezoneHelper,
                                 Function0<SetF<UUID>> provideProductLine, GroupPromoDao groupPromoDao,
                                 Option<Tuple2<Group, GroupPromoEntity>> warmingCache) {
        super(promoTemplate, userTimezoneHelper, provideProductLine, groupPromoDao, warmingCache);
    }

    public PerGroupPromoTemplate(PromoTemplateEntity promoTemplate, UserTimezoneHelper userTimezoneHelper,
                                 Function0<SetF<UUID>> provideProductLine, GroupPromoDao groupPromoDao) {
        super(promoTemplate, userTimezoneHelper, provideProductLine, groupPromoDao);
    }

    @Override
    public boolean canBeUsed(Option<Group> group) {
        if (!group.isPresent()) {
            // если акция для пользователя и у нас нет group для проверки состояния по группе,
            // то такую акцию нельзя использовать не активировав
            return false;
        }

        // group promo template can be used only if activated for user
        return getPromoEntity(group.get()).map(GroupPromoEntity::isActive).orElse(false);

    }

    @Override
    public Option<Instant> canBeUsedUntilDate(Option<Group> group) {
        if (!canBeUsed(group)) {
            throw new IllegalStateException("cannot be used " + this);
        }
        GroupPromoEntity entity = getPromoEntity(group.get()).orElseThrow(() -> new NoSuchElementException("if promo " +
                "template can be used by some user, then must " + "be activated user promo"));

        return entity.getToDate();
    }

    @Override
    public void markUsed(Group group) {
        Option<GroupPromoEntity> entityO = getPromoEntity(group);
        Validate.some(entityO);
        GroupPromoEntity entity = entityO.get();
        Validate.isTrue(entity.isActive());

        log.info("set group promo {} used for group {}", getId(), group);
        groupPromoDao.updateStatusById(entity.getId(), GroupPromoStatusType.USED);
    }

    @Override
    public Option<GroupPromoEntity> activatePromo(Group group, boolean force, Function0<GroupPromoEntity> afterCallback) {
        if (isExpired()) {
            throw new BadRequestException("promo with key " + getCode() + " is expired");
        }

        if (!isAvailableFor(group, force)) {
            return Option.empty();
        }

        Instant from = getPromoFromDate();
        Option<Instant> to = calcPromoToDate(group.getOwnerUid(), from);
        GroupPromoStatusType status = GroupPromoStatusType.ACTIVE;

        CreatedOrExistResult<GroupPromoEntity> result =
                groupPromoDao.createIfNotExist(
                        GroupPromoDao.InsertData.builder()
                                .groupId(group.getId())
                                .promoTemplateId(getId())
                                .fromDate(from)
                                .toDate(to)
                                .status(status)
                                .build()
                );

        if (!result.isCreated()) {
            log.info("GroupPromo exist. {}. New fields - status={}, fromDate={}, toDate={}", result.getEntity(),
                    status, from, to);

            GroupPromoEntity entity = groupPromoDao.updateById(
                    result.getEntity().getId(),
                    new GroupPromoDao.UpdateData(
                            from,
                            to,
                            GroupPromoStatusType.ACTIVE
                    )
            );

            return Option.of(entity);
        }

        return Option.of(result.getEntity());
    }

    @Override
    public boolean isAvailableFor(Group group, boolean force) {
        if (isExpired()) {
            log.info("can't activate {} promo for group {} cause it's expired", promoTemplate, group);
            return false;
        }

        return canActivatePromo(group, force);
    }

    private boolean canActivatePromo(Group group, boolean force) {
        Option<GroupPromoEntity> entityO = getPromoEntity(group);

        if (!entityO.isPresent()) {
            log.info("can activate = true for {} promo for group {}. Not found groupPromo", promoTemplate, group);
            return true;
        }

        GroupPromoEntity entity = entityO.get();

        //Not used expired promo
        if (!entity.isActive() && entity.getStatus() == GroupPromoStatusType.ACTIVE) {
            log.info("can activate = true for {} promo for group {}. found groupPromo: {}", promoTemplate, group,
                    entity);
            return true;
        }

        boolean result = entity.getStatus() == GroupPromoStatusType.USED && force;
        log.info("can activate = {} for {} promo for group {}. found groupPromo: {}", result, promoTemplate, group,
                entity);
        return result;
    }
}
