package ru.yandex.chemodan.app.psbilling.core.promos.v2;

import java.util.UUID;

import lombok.RequiredArgsConstructor;
import org.joda.time.Instant;

import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.SetF;
import ru.yandex.bolts.collection.Tuple2;
import ru.yandex.bolts.function.Function0;
import ru.yandex.chemodan.app.psbilling.core.dao.promos.PromoTemplateDao;
import ru.yandex.chemodan.app.psbilling.core.dao.promos.PromoTemplateWithUsedPromo;
import ru.yandex.chemodan.app.psbilling.core.dao.promos.group.GroupPromoDao;
import ru.yandex.chemodan.app.psbilling.core.dao.promos.group.GroupPromoTemplateDao;
import ru.yandex.chemodan.app.psbilling.core.entities.groups.Group;
import ru.yandex.chemodan.app.psbilling.core.entities.promos.PromoTemplateEntity;
import ru.yandex.chemodan.app.psbilling.core.entities.promos.group.GroupPromoEntity;
import ru.yandex.chemodan.app.psbilling.core.promos.groups.AbstractGroupPromoTemplate;
import ru.yandex.chemodan.app.psbilling.core.promos.groups.GlobalB2bPromoTemplate;
import ru.yandex.chemodan.app.psbilling.core.promos.groups.PerGroupPromoTemplate;
import ru.yandex.chemodan.util.blackbox.UserTimezoneHelper;
import ru.yandex.chemodan.util.exception.NotFoundException;

@RequiredArgsConstructor
public class GroupPromoService {
    private final PromoTemplateDao promoTemplateDao;
    private final GroupPromoDao groupPromoDao;
    private final GroupPromoTemplateDao groupPromoTemplateDao;
    private final UserTimezoneHelper userTimezoneHelper;

    public MapF<UUID, AbstractGroupPromoTemplate> readyForUsingGroupPromos(Option<Group> group, Option<UUID> promoId) {
        return groupPromoTemplateDao.findPromoTemplateForInTimePoint(
                        group.map(Group::getId),
                        promoId,
                        Option.of(Instant.now())
                )
                .mapValues(this::create);
    }

    public ListF<AbstractGroupPromoTemplate> findByLine(Group group, ListF<UUID> lineIds) {
        return groupPromoTemplateDao.findPromoTemplateByProductLine(group, lineIds)
                .mapEntries((promo, line) -> create(promo, Option.of(group), line::unique));
    }

    public AbstractGroupPromoTemplate findByCode(String code) {
        return promoTemplateDao.findByCode(code)
                .map(this::create)
                .orElseThrow(() -> new NotFoundException("promo with code " + code + " not found"));
    }

    public AbstractGroupPromoTemplate findById(UUID promoId) {
        return promoTemplateDao.findByIdO(promoId)
                .map(this::create)
                .orElseThrow(() -> new NotFoundException("promo with id " + promoId + " not found"));
    }

    private AbstractGroupPromoTemplate create(PromoTemplateEntity entity) {
        return create(
                new PromoTemplateWithUsedPromo<>(entity, Option.empty()),
                Option.empty(),
                provideProductLine(entity.getId())
        );
    }

    private AbstractGroupPromoTemplate create(
            PromoTemplateWithUsedPromo<GroupPromoEntity> entity,
            Option<Group> groupsO,
            Function0<SetF<UUID>> provideProductLine
    ) {
        PromoTemplateEntity promoTemplate = entity.getPromoTemplate();

        Option<Tuple2<Group, GroupPromoEntity>> warmingCache = Option.when(
                groupsO.isPresent() && entity.getUsed().isPresent(),
                () -> Tuple2.tuple(groupsO.get(), entity.getUsed().get())
        );

        switch (promoTemplate.getApplicationArea()) {
            case GLOBAL_B2B:
                return new GlobalB2bPromoTemplate(
                        promoTemplate,
                        userTimezoneHelper,
                        provideProductLine,
                        groupPromoDao,
                        warmingCache
                );
            case PER_GROUP:
                return new PerGroupPromoTemplate(
                        promoTemplate,
                        userTimezoneHelper,
                        provideProductLine,
                        groupPromoDao,
                        warmingCache
                );
            default:
                throw new IllegalStateException(promoTemplate.getApplicationArea().toString());
        }
    }

    private Function0<SetF<UUID>> provideProductLine(UUID promoId) {
        return () -> promoTemplateDao.getProductLines(promoId);
    }
}
