package ru.yandex.chemodan.app.psbilling.core.promos.v2;

import java.util.UUID;

import lombok.Data;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.jetbrains.annotations.NotNull;

import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.SetF;
import ru.yandex.bolts.function.Function1B;
import ru.yandex.chemodan.app.psbilling.core.dao.products.ProductLineDao;
import ru.yandex.chemodan.app.psbilling.core.entities.AbstractEntity;
import ru.yandex.chemodan.app.psbilling.core.entities.groups.Group;
import ru.yandex.chemodan.app.psbilling.core.entities.products.ProductLineEntity;
import ru.yandex.chemodan.app.psbilling.core.entities.promos.PromoApplicationType;
import ru.yandex.chemodan.app.psbilling.core.products.GroupProduct;
import ru.yandex.chemodan.app.psbilling.core.products.GroupProductManager;
import ru.yandex.chemodan.app.psbilling.core.products.LimitedTimeProductLineB2b;
import ru.yandex.chemodan.app.psbilling.core.promos.NewPromoTemplate;
import ru.yandex.chemodan.app.psbilling.core.promos.groups.AbstractGroupPromoTemplate;
import ru.yandex.inside.passport.PassportUid;

@Slf4j
@RequiredArgsConstructor
public class GroupPromoUsedService {

    private final GroupPromoService groupPromoService;
    private final GroupProductManager groupProductManager;
    private final ProductLineDao productLineDao;

    public Option<AbstractGroupPromoTemplate> useGroupPromoForProduct(
            PassportUid uid,
            Group group,
            GroupProduct product,
            ListF<String> productExperiments
    ) {
        ListF<ProductLineEntity> productLines = productLineDao.findByGroupProduct(product.getId());

        SetF<UUID> productSets = productLines
                .map(ProductLineEntity::getProductSetId)
                .unique();

        MapF<UUID, ListF<ProductLineEntity>> lineBySet = productLineDao.findByProductSetIds(productSets)
                .groupBy(ProductLineEntity::getProductSetId);

        LineBySetForBestData data = new LineBySetForBestData(lineBySet, uid, group, productExperiments);

        MapF<UUID, LimitedTimeProductLineB2b> bestLineInSetWithoutPromo = suitableLineInSet(data, true);
        Option<ProductLineEntity> bestLineWithoutPromo = productLines
                .find(productInSetPredicate(bestLineInSetWithoutPromo));

        if (bestLineWithoutPromo.isPresent()) {
            log.info("found regular product line {} with required product {}. No promo will be used",
                    bestLineWithoutPromo, product);
            return Option.empty();
        }

        MapF<UUID, LimitedTimeProductLineB2b> bestLineBySetWithPromo = suitableLineInSet(data, false);

        ListF<UUID> lineWithPromoIds = productLines
                .filter(productInSetPredicate(bestLineBySetWithPromo))
                .map(AbstractEntity::getId);

        ListF<AbstractGroupPromoTemplate> canBeUsePromo = groupPromoService.findByLine(group, lineWithPromoIds)
                .filter(p -> p.canBeUsed(group))
                .sorted(NewPromoTemplate.byMultipleTimeThenByUntilDate(Option.of(group)));

        if (canBeUsePromo.isEmpty()) {
            log.info("no promo found for group {} to be used", group);
            return Option.empty();
        }

        Option<AbstractGroupPromoTemplate> multipleTimePromo = canBeUsePromo.find(x -> x
                .getApplicationType() == PromoApplicationType.MULTIPLE_TIME
        );

        if (multipleTimePromo.isPresent()) {
            log.info("skip setting promo used status for group {} " +
                    "cause got active promo which can be used multiple times", group);
            return multipleTimePromo;
        }

        SetF<UUID> promoProductLines = canBeUsePromo.flatMap(NewPromoTemplate::getProductLineIds).unique();
        Option<ProductLineEntity> regularProductLine = productLines.find(l -> !promoProductLines.containsTs(l.getId()));
        if (regularProductLine.isPresent()) {
            log.info("found regular promo line {} with required product {}. No promo will be used",
                    regularProductLine, product);
            return Option.empty();
        }

        Option<AbstractGroupPromoTemplate> promoToUse = canBeUsePromo.firstO();
        promoToUse.ifPresent(p -> p.markUsed(group));
        return promoToUse;
    }

    /**
     * Ищем подходящие продукты во всех продуктсетах где они представлены
     * @param data
     * @param excludePromoLine - исключить линейки, которые участвуют в промо
     * @return
     */
    private MapF<UUID, LimitedTimeProductLineB2b> suitableLineInSet(LineBySetForBestData data,
                                                                    boolean excludePromoLine) {
        return data.lineBySet
                .mapValues(lines -> groupProductManager.selectProductLine(
                                lines,
                                excludePromoLine,
                                Option.of(data.uid),
                                Option.of(data.group),
                                data.productExperiments,
                                Option.empty()
                        )
                )
                .filterValues(Option::isPresent)
                .mapValues(Option::get);
    }

    @NotNull
    private Function1B<ProductLineEntity> productInSetPredicate(MapF<UUID, LimitedTimeProductLineB2b> bestLineBySetWithoutPromo) {
        return productLine -> bestLineBySetWithoutPromo.getO(productLine.getProductSetId())
                .map(LimitedTimeProductLineB2b::getProductLine)
                .equals(Option.of(productLine));
    }


    @Data
    private static class LineBySetForBestData {
        private final MapF<UUID, ListF<ProductLineEntity>> lineBySet;
        private final PassportUid uid;
        private final Group group;
        private final ListF<String> productExperiments;
    }
}
