package ru.yandex.chemodan.app.psbilling.core.staff;

import java.io.InputStream;
import java.nio.charset.StandardCharsets;

import com.fasterxml.jackson.annotation.JsonAnyGetter;
import com.fasterxml.jackson.annotation.JsonAnySetter;
import com.fasterxml.jackson.annotation.JsonProperty;
import lombok.Getter;
import lombok.RequiredArgsConstructor;
import lombok.SneakyThrows;
import org.apache.commons.io.IOUtils;
import org.springframework.web.client.RestTemplate;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.SetF;
import ru.yandex.commune.dynproperties.DynamicProperty;
import ru.yandex.inside.passport.PassportUid;
import ru.yandex.misc.env.EnvironmentType;
import ru.yandex.misc.io.http.UriBuilder;
import ru.yandex.misc.lang.DefaultObject;
import ru.yandex.misc.lang.StringUtils;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;

@RequiredArgsConstructor
public class YandexStaffService {
    private static final Logger logger = LoggerFactory.getLogger(YandexStaffService.class);
    private final String host;
    private final RestTemplate restTemplate;
    private final DynamicProperty<String> uidsToOverride = new DynamicProperty<String>("ps-billing.staff.uids", "");
    private final ListF<PassportUid> additionalUidsForSyncing;
    private final DynamicProperty<String> uidsToRemove =
            new DynamicProperty<String>("ps-billing.staff.uids-to-remove", "");


    public YandexStaffService(String host, RestTemplate restTemplate)
    {
        this.host = host;
        this.restTemplate = restTemplate;
        if (EnvironmentType.getActive() == EnvironmentType.PRODUCTION) {
            this.additionalUidsForSyncing = loadAdditionalUids();
        } else {
            this.additionalUidsForSyncing = Cf.list();
        }
    }

    @SneakyThrows
    private ListF<PassportUid> loadAdditionalUids() {
        InputStream file =
                YandexStaffService.class.getResourceAsStream("additional_staff_production_uids.txt");
        return Cf.x(IOUtils.readLines(file, StandardCharsets.UTF_8))
                .stableUnique()
                .map(Long::parseLong)
                .map(PassportUid::cons);
    }

    public ListF<PassportUid> getUidsForSyncing() {
        if (StringUtils.isNotEmpty(uidsToOverride.get())) {
            return Cf.list(StringUtils.split(uidsToOverride.get(), ","))
                    .map(String::trim)
                    .map(Long::parseLong)
                    .map(PassportUid::cons);
        }

        SetF<PassportUid> uids = getStaffUids().plus(additionalUidsForSyncing).unique();
        if (StringUtils.isNotEmpty(uidsToRemove.get())) {
            uids = uids.minus(Cf.list(StringUtils.split(uidsToRemove.get(), ","))
                    .map(String::trim)
                    .map(Long::parseLong)
                    .map(PassportUid::cons));
        }
        return uids.toList();
    }

    public ListF<PassportUid> getStaffUids() {
        StaffResponse response = restTemplate.getForObject(buildUrl(), StaffResponse.class);
        return response.getUsers().values().map(StaffRecord::getExternalUid).filterNotNull()
                .stableUnique().map(PassportUid::new);
    }

    private String buildUrl() {
        return UriBuilder.cons("https://" + host).appendPath("/api/login_links/export/").toUrl();
    }

    @Getter
    public static class StaffRecord extends DefaultObject {
        @JsonProperty("internal_uid")
        private long yaTeamUid;
        @JsonProperty("internal_login")
        private String yaTeamLogin;
        @JsonProperty("is_dismissed")
        private boolean isDismissed;
        @JsonProperty("is_homeworker")
        private boolean isHomeworker;
        @JsonProperty("chief_uid")
        private long chiefYaTeamUid;
        @JsonProperty("affiliation")
        private String affiliation;
        @JsonProperty("external_login")
        private String externalLogin;
        @JsonProperty("external_uid")
        private Long externalUid;
    }

    public static class StaffResponse extends DefaultObject {
        private MapF<String, StaffRecord> users = Cf.hashMap();

        @JsonAnyGetter
        public MapF<String, StaffRecord> getUsers() {
            return users;
        }

        @JsonAnySetter
        public void setUsers(String key, StaffRecord value) {
            this.users.put(key, value);
        }
    }

}
