package ru.yandex.chemodan.app.psbilling.core.synchronization.engine;

import java.util.UUID;

import lombok.AllArgsConstructor;
import lombok.Getter;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.transaction.support.TransactionTemplate;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.CollectionF;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.Tuple2;
import ru.yandex.bolts.function.Function;
import ru.yandex.chemodan.app.psbilling.core.dao.InsertingData;
import ru.yandex.commune.bazinga.BazingaTaskManager;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;

@AllArgsConstructor
public abstract class BaseTablesSynchronizer<P extends SynchronizableRecord> {
    protected static final int ANALYZE_BATCH_SIZE = 1000;
    private static final Logger logger = LoggerFactory.getLogger(BaseTablesSynchronizer.class);

    protected final ParentSynchronizableRecordDao<P> parentDao;
    protected final TransactionTemplate transactionTemplate;
    protected final BazingaTaskManager bazingaTaskManager;

    protected <C extends SynchronizableRecord, B extends InsertingData<C>> UpdateResult<C, B> updateDataInChildTableCore(
            P parentRecord,
            ChildSynchronizableRecordDao<C, B> childDao,
            Function<P, MapF<String, B>> activeActualChildrenProducer,
            Function<C, String> synchronizationKeyProvider) {

        //if target is disabled, we don't need create any child record, and have to disable all existing children
        MapF<String, B> actualActiveChildren = parentRecord.getTarget() == Target.ENABLED ?
                activeActualChildrenProducer.apply(parentRecord) : Cf.map();
        MapF<String, C> existingChildren = childDao.findEnabledByParentId(parentRecord.getId())
                .toMapMappingToKey(synchronizationKeyProvider);

        logger.info("actualActiveChildren: {}", actualActiveChildren.keySet());
        logger.info("existingChildren: {}", existingChildren.keySet());

        ListF<B> toInsert = Cf.arrayList();
        ListF<C> toUpdateTargetState = Cf.arrayList();
        for (Tuple2<String, B> actualActiveChildInsertData : actualActiveChildren.entries()) {
            Option<C> existingChildO = existingChildren.removeO(actualActiveChildInsertData.get1());
            if (!existingChildO.isPresent()) {
                toInsert.add(actualActiveChildInsertData.get2());
            } else {
                C existingChild = existingChildO.get();
                if (existingChild.getTarget() != parentRecord.getTarget()) {
                    toUpdateTargetState.add(existingChild);
                }
            }
        }

        CollectionF<C> toDeactivate = existingChildren.values();
        logger.info("Diff results: toInsert {}, toUpdateTargetState: {}, toDeactivate: {}",
                toInsert, toUpdateTargetState, toDeactivate);

        boolean updated = updateChildRecords(childDao, toInsert, toUpdateTargetState, toDeactivate,
                parentRecord.getTarget());
        return new UpdateResult<C, B>(updated, toInsert);
    }

    private <C extends SynchronizableRecord, B extends InsertingData<C>> boolean updateChildRecords(
            ChildSynchronizableRecordDao<C, B> childDao,
            ListF<B> toInsert, ListF<C> toUpdateTargetState, CollectionF<C> toDeactivate, Target target) {
        boolean wasUpdates = false;
        if (toInsert.isNotEmpty()) {
            childDao.batchInsert(toInsert, target);
            wasUpdates = true;
        }
        if (toUpdateTargetState.isNotEmpty()) {
            childDao.setTargetState(toUpdateTargetState.map(SynchronizableRecord::getId), target);
            wasUpdates = true;
        }
        if (toDeactivate.isNotEmpty()) {
            childDao.setTargetState(toDeactivate.map(SynchronizableRecord::getId), Target.DISABLED);
            wasUpdates = true;
        }

        return wasUpdates;
    }

    @Transactional
    public void updateStatusInParentTable(int batchSize, boolean propagateSnoozing) {
        Option<UUID> lastUpdated = Option.empty();
        do {
            lastUpdated = parentDao.analyzeChildrenStatus(lastUpdated.getOrNull(), batchSize, propagateSnoozing);
        } while (lastUpdated.isPresent());
    }

    @Transactional
    public void updateStatusInParentTable(boolean propagateSnoozing) {
        updateStatusInParentTable(ANALYZE_BATCH_SIZE, propagateSnoozing);
    }

    protected abstract void scheduleChildrenSynchronization(UUID parentId);

    @Getter
    @AllArgsConstructor
    protected static class UpdateResult<C extends SynchronizableRecord, B extends InsertingData<C>> {
        private final Boolean updated;
        private final ListF<B> toInsert;
    }
}
