package ru.yandex.chemodan.app.psbilling.core.synchronization.engine;

import java.sql.ResultSet;
import java.util.UUID;

import lombok.AllArgsConstructor;
import org.joda.time.Duration;
import org.joda.time.Instant;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.Tuple2;
import ru.yandex.bolts.function.Function;
import ru.yandex.bolts.function.Function2;
import ru.yandex.misc.spring.jdbc.JdbcTemplate3;

@SuppressWarnings("SqlResolve")
@AllArgsConstructor
public class ParentSynchronizableRecordJdbcHelper2<T extends SynchronizableRecord> implements IParentSynchronizableRecordJdbcHelper<T> {
    private JdbcTemplate3 jdbcTemplate;
    private String tableName;
    private Function<ResultSet, T> rowParser;
    private String childrenTableName1;
    private String childrenTableName2;
    private Function2<String, String, String> joinChildConditionProvider;

    public T lockRecord(UUID id) {
        return jdbcTemplate.queryForObject("select * from " + tableName + " where id = ? for update",
                (rs, num) -> rowParser.apply(rs), id);
    }

    public void updateStatusToSyncing(UUID id) {
        MapF<String, Object> params = Cf.hashMap();
        params.put("sync_status", SynchronizationStatus.SYNCING.value());
        params.put("now", Instant.now());
        params.put("id", id);

        jdbcTemplate.update("update " + tableName + " set status = :sync_status," +
                "status_updated_at = :now where id = :id", params);
    }


    public Option<UUID> analyzeChildrenStatus(UUID lastProcessedId, int limit, boolean propagateSnoozing) {
        String parentsFilterClause = lastProcessedId == null ? "" : " and ID > :lastProcessedId ";
        String joinCondition1 = joinChildConditionProvider.apply("p", "c1");
        String joinCondition2 = joinChildConditionProvider.apply("p", "c2");
        String countCondition1 = "count(c1.id) filter (where c1.status = :status_snoozing)";
        String countCondition2 = "count(c2.id) filter (where c2.status = :status_snoozing)";

        //use CREATE INDEX xxx ON tableName (status) WHERE status <> 'actual'
        String query = "with parent_ids as(" +
                "    select * from " + tableName +
                "    where status in (:status_syncing, :status_snoozing ) " + parentsFilterClause +
                "    order by id limit :limit " +
                ")," +
                "parents_synced as (" +
                "     select p.id," +
                " (count(c1.id) filter (where c1.status = :status_snoozing) " +
                "   + count(c2.id) filter (where c2.status = :status_snoozing))>0 as " + "has_snoozing_children " +
                "     from parent_ids p " +
                "       left join " + childrenTableName1 + " c1 on " + joinCondition1 +
                "       left join " + childrenTableName2 + " c2 on " + joinCondition2 +
                "     group by p.id " +
                "     having count(c1.id) = (count(c1.id) filter (where c1.status in(:status_actual, :status_snoozing" +
                " )))" +
                "        and count(c2.id) = (count(c2.id) filter (where c2.status in(:status_actual, :status_snoozing" +
                " )))" +
                "), " +
                "locked as (select t1.id, t2.has_snoozing_children" +
                "    from " + tableName + " as t1 " +
                "      join parents_synced t2 on t1.id = t2.id " +
                "    where t1.status in (:status_syncing, :status_snoozing ) " +
                //если нужно пробрасывать снузинг наверх, и он уже проброшен, то такие зиписи не нужны
                (propagateSnoozing ? " and not (t1.status = :status_snoozing and t2.has_snoozing_children) " : "") +
                "    for update of t1 skip locked), " +
                "locked_actual as (" +
                "     select p.id, (case when (:propagate_snoozing and (count(c1.id) filter (where c1.status = " +
                ":status_snoozing) + count(c2.id) filter (where c2.status = :status_snoozing))>0) then " +
                ":status_snoozing" +
                " else :status_actual end) as new_status" +
                "       from " + tableName + " as p " +
                "       left join " + childrenTableName1 + " c1 on " + joinCondition1 +
                "       left join " + childrenTableName2 + " c2 on " + joinCondition2 +
                "       where p.id in (select id from locked)" +
                "       group by p.id " +
                "       having count(c1.id) = (count(c1.id) filter (where c1.status in(:status_actual, " +
                ":status_snoozing )))" +
                "          and count(c2.id) = (count(c2.id) filter (where c2.status in(:status_actual, " +
                ":status_snoozing )))" +
                ")," +
                "update_st as (" +
                "     update " + tableName +
                "     set status = locked_actual.new_status, " +
                "         status_updated_at = :now," +
                "         actual_enabled_at = (case when target = :target_enabled and actual_enabled_at is null and " +
                "locked_actual.new_status = :status_actual then :now else actual_enabled_at end)," +
                "         actual_disabled_at = (case when target = :target_disabled and actual_disabled_at is null " +
                "and locked_actual.new_status = :status_actual then :now else actual_disabled_at end)" +
                "     from locked_actual " +
                "     where locked_actual.id = " + tableName + ".id" +
                ")" +
                "select max(id::text) as max_id, count(*) as cnt from parent_ids";

        MapF<String, Object> params = Cf.hashMap();
        params.put("status_actual", SynchronizationStatus.ACTUAL.value());
        params.put("status_syncing", SynchronizationStatus.SYNCING.value());
        params.put("status_snoozing", SynchronizationStatus.SNOOZING.value());
        params.put("propagate_snoozing", propagateSnoozing);
        params.put("target_disabled", Target.DISABLED.value());
        params.put("target_enabled", Target.ENABLED.value());
        params.put("now", Instant.now());
        params.put("limit", limit);
        if (lastProcessedId != null) {
            params.put("lastProcessedId", lastProcessedId);
        }

        Option<Tuple2<UUID, Integer>> updateResult = jdbcTemplate.queryForOption(query,
                (rs, rowNum) -> {
                    String maxId = rs.getString("max_id");
                    return new Tuple2<>(maxId == null ? null : UUID.fromString(maxId), rs.getInt("cnt"));
                },
                params);
        if (!updateResult.isPresent()) {
            return Option.empty();
        }

        Tuple2<UUID, Integer> tuple = updateResult.get();
        return tuple.get2() == limit ? Option.ofNullable(tuple.get1()) : Option.empty();
    }

    public void unlockRecord(T parentRecord) {
        //do noting, commit will free lock
    }

    public ListF<UUID> findRecordsInInitStatus() {
        //use CREATE INDEX xxx ON tableName (status) WHERE status <> 'actual'
        return jdbcTemplate.queryForList(
                "select id from " + tableName + " where status = ?",
                String.class,
                SynchronizationStatus.INIT.value()
        ).map(UUID::fromString);
    }

    public void setStatusActual(ListF<UUID> ids, Target withTarget) {
        String fieldName = withTarget == Target.ENABLED ? "actual_enabled_at" : "actual_disabled_at";
        MapF<String, Object> params = Cf.hashMap();
        params.put("ids", ids);
        params.put("now", Instant.now());
        params.put("status_actual", SynchronizationStatus.ACTUAL);

        jdbcTemplate.update("update " + tableName + " set " +
                "status = :status_actual, " +
                fieldName + " = :now, " +
                "status_updated_at = :now " +
                "where id in (:ids)", params);
    }

    public int countNotActualUpdatedBefore(Duration triggeredBefore, boolean countSnoozed) {
        ListF<String> statuses = Cf.arrayList(SynchronizationStatus.ACTUAL.value());
        if (!countSnoozed) {
            statuses.add(SynchronizationStatus.SNOOZING.value());
        }

        MapF<String, Object> params = Cf.hashMap();
        params.put("statuses", statuses);
        params.put("ts", Instant.now().minus(triggeredBefore));

        return jdbcTemplate.queryForOption("SELECT count(*) as cnt FROM " + tableName +
                        " WHERE status not in ( :statuses ) and status_updated_at < :ts",
                (rs, num) -> rs.getInt("cnt"), params).get();
    }

    public void resetSyncingStatus(ListF<UUID> ids) {
        if (ids.isEmpty()) {
            return;
        }

        MapF<String, Object> params = Cf.hashMap();
        params.put("ids", ids);
        params.put("now", Instant.now());
        params.put("status", SynchronizationStatus.INIT.value());

        jdbcTemplate.update("update " + tableName +
                " set status_updated_at = :now, status = :status where id in ( :ids )", params);
    }
}
