package ru.yandex.chemodan.app.psbilling.core.synchronization.feature;

import java.math.BigDecimal;
import java.util.List;
import java.util.Map;
import java.util.NoSuchElementException;
import java.util.UUID;

import com.google.common.annotations.VisibleForTesting;
import lombok.SneakyThrows;
import org.jetbrains.annotations.NotNull;
import org.joda.time.Duration;
import org.joda.time.Instant;
import org.springframework.expression.Expression;
import org.springframework.expression.ExpressionParser;
import org.springframework.expression.common.TemplateParserContext;
import org.springframework.expression.spel.standard.SpelExpressionParser;
import org.springframework.http.HttpEntity;
import org.springframework.http.HttpHeaders;
import org.springframework.http.ResponseEntity;
import org.springframework.transaction.support.TransactionTemplate;
import org.springframework.web.client.HttpStatusCodeException;
import org.springframework.web.client.RestTemplate;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.SetF;
import ru.yandex.bolts.function.Function;
import ru.yandex.bolts.function.Function1V;
import ru.yandex.bolts.function.Function2;
import ru.yandex.bolts.type.number.BigDecimalType;
import ru.yandex.chemodan.app.psbilling.core.dao.features.FeatureCallbackContextDao;
import ru.yandex.chemodan.app.psbilling.core.dao.features.ServiceDao;
import ru.yandex.chemodan.app.psbilling.core.dao.features.ServiceFeatureDao;
import ru.yandex.chemodan.app.psbilling.core.dao.products.FeatureDao;
import ru.yandex.chemodan.app.psbilling.core.dao.products.ProductFeatureDao;
import ru.yandex.chemodan.app.psbilling.core.entities.features.FeatureCallbackContext;
import ru.yandex.chemodan.app.psbilling.core.entities.features.FeatureWithOwner;
import ru.yandex.chemodan.app.psbilling.core.entities.features.IssuedFeature;
import ru.yandex.chemodan.app.psbilling.core.entities.products.FeatureEntity;
import ru.yandex.chemodan.app.psbilling.core.entities.products.ProductFeatureEntity;
import ru.yandex.chemodan.app.psbilling.core.synchronization.engine.SynchronizationStatus;
import ru.yandex.chemodan.app.psbilling.core.synchronization.engine.Target;
import ru.yandex.chemodan.app.psbilling.core.synchronization.feature.errorprocessors.DefaultErrorProcessor;
import ru.yandex.chemodan.app.psbilling.core.synchronization.feature.errorprocessors.ErrorProcessor;
import ru.yandex.chemodan.app.psbilling.core.synchronization.feature.templatecontexts.ActivationContext;
import ru.yandex.chemodan.app.psbilling.core.synchronization.feature.templatecontexts.DeactivationContext;
import ru.yandex.chemodan.app.psbilling.core.synchronization.feature.templatecontexts.SetAmountContext;
import ru.yandex.chemodan.app.psbilling.core.synchronization.feature.templatecontexts.TemplateContext;
import ru.yandex.chemodan.app.psbilling.core.synchronization.feature.templatecontexts.TemplateContextFactory;
import ru.yandex.chemodan.app.psbilling.core.util.JsonHelper;
import ru.yandex.chemodan.app.psbilling.core.util.RequestTemplate;
import ru.yandex.commune.bazinga.BazingaTaskManager;
import ru.yandex.commune.dynproperties.DynamicProperty;
import ru.yandex.commune.util.RetryUtils;
import ru.yandex.inside.passport.tvm2.Tvm2;
import ru.yandex.inside.passport.tvm2.TvmHeaders;
import ru.yandex.misc.digest.Sha256;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;

public abstract class AbstractFeatureActualizationService<TOwner, TParams extends FeatureActualizationTask.Parameters<TOwner>> {
    private static final Logger logger = LoggerFactory.getLogger(AbstractFeatureActualizationService.class);
    private static final Duration DEFAULT_SYNC_DURATION = Duration.standardMinutes(5);

    private final DynamicProperty<Integer> SNOOZE_DURATION_HOURS =
            new DynamicProperty<>("features.snooze.duration.hours", 4);

    private final FeatureDao featureDao;
    private final ProductFeatureDao productFeatureDao;
    private final RestTemplate externalSystems;
    private final ServiceFeatureDao serviceFeatureDao;
    private final ServiceDao serviceDao;
    private final Tvm2 tvm2;
    private final TransactionTemplate transactionTemplate;
    private final BazingaTaskManager bazingaTaskManager;
    private final ExpressionParser parser = new SpelExpressionParser();
    private final MapF<String, ErrorProcessor> errorProcessors;
    private final Function2<TOwner, UUID, FeatureActualizationTask<TOwner, TParams>> taskCreator;
    private final TemplateContextFactory<TOwner> templateContextFactory;
    private final ErrorProcessor defaultErrorProcessor;
    protected final FeatureCallbackContextDao featureCallbackContextDao;

    public AbstractFeatureActualizationService(FeatureDao featureDao,
                                               ProductFeatureDao productFeatureDao, RestTemplate externalSystems,
                                               ServiceFeatureDao serviceFeatureDao,
                                               FeatureCallbackContextDao featureCallbackContextDao,
                                               ServiceDao serviceDao, Tvm2 tvm2,
                                               TransactionTemplate transactionTemplate,
                                               BazingaTaskManager bazingaTaskManager,
                                               List<ErrorProcessor> errorProcessors,
                                               Function2<TOwner, UUID, FeatureActualizationTask<TOwner, TParams>> taskCreator,
                                               TemplateContextFactory<TOwner> templateContextFactory) {
        this.featureDao = featureDao;
        this.productFeatureDao = productFeatureDao;
        this.externalSystems = externalSystems;
        this.serviceFeatureDao = serviceFeatureDao;
        this.featureCallbackContextDao = featureCallbackContextDao;
        this.serviceDao = serviceDao;
        this.tvm2 = tvm2;
        this.transactionTemplate = transactionTemplate;
        this.bazingaTaskManager = bazingaTaskManager;
        this.errorProcessors = Cf.x(errorProcessors).toMapMappingToKey(ErrorProcessor::getName);
        this.taskCreator = taskCreator;
        this.templateContextFactory = templateContextFactory;
        this.defaultErrorProcessor = this.errorProcessors.getOrThrow(DefaultErrorProcessor.NAME);

        if (Cf.x(errorProcessors).map(ErrorProcessor::getName).unique().size() != errorProcessors.size()) {
            throw new IllegalStateException("there are duplicated names in error processors :" + errorProcessors);
        }
    }

    public void scheduleFeatureSynchronization() {
        ListF<FeatureWithOwner<TOwner>> notActualFeatures = serviceFeatureDao.findForSynchronization();
        for (FeatureWithOwner<TOwner> feature : notActualFeatures) {
            bazingaTaskManager.schedule(
                    taskCreator.apply(feature.getOwnerId(), feature.getFeatureId())
            );
        }
    }

    @VisibleForTesting
    public void synchronize(TOwner ownerId, UUID featureId) {
        synchronize(ownerId, featureId, DEFAULT_SYNC_DURATION);
    }

    @SneakyThrows
    public void synchronize(TOwner ownerId, UUID featureId, Duration maxExecutionTime) {
        withUnlockOnError(lockId -> {
            Option<FeatureState> featureStateO = calculateFeatureState(lockId, ownerId, featureId, maxExecutionTime);
            if (!featureStateO.isPresent()) {
                return;
            }
            FeatureState featureState = featureStateO.get();

            if (!featureState.isNothingToDo()) {
                Option<Map<String, Object>> contextO = activateIfNecessary(featureState, ownerId);
                if (!contextO.isPresent()) {
                    snoozeServices(featureState);
                    serviceFeatureDao.unlock(lockId);
                    return;
                }
                Map<String, Object> contextForCall = contextO.get();

                sendFeatureAmount(featureState, ownerId, contextForCall);

                if (featureState.isNeedToDeactivate()) {
                    deactivateFeature(featureState, ownerId, contextForCall);
                }
            }

            // без транзакции можем попасть в ситуацию, что удалили контекст, но не проставили actual,
            // таска перезапустится и упадет на походе по деактивации
            transactionTemplate.execute(status -> {
                ListF<IssuedFeature> serviceFeatures = featureState.getLockedServiceFeatures();
                if (featureState.isNeedToDeactivate()) {
                    delete(ownerId, featureId);
                }
                serviceFeatureDao.setStatusActualAndUnlock(lockId,
                        serviceFeatures.toMap(IssuedFeature::getId, IssuedFeature::getTarget));
                serviceDao.updateFirstFeatureDisabledAt(serviceFeatures
                        .filter(sf -> sf.getTarget() == Target.DISABLED)
                        .map(IssuedFeature::getParentServiceId));
                return null;
            });
        });
    }

    private Option<Map<String, Object>> activateIfNecessary(FeatureState featureState, TOwner ownerId) {
        final Option<Map<String, Object>> result;

        if (featureState.isNeedToActivate() &&
                featureState.getFeature().getActivationRequestTemplate().isPresent()) {
            result = activateFeature(featureState, ownerId);
        } else {
            result = Option.of(fetchContextFromDb(ownerId, featureState.getFeature().getId()));
        }
        if (featureState.isNeedToActivate() && result.isPresent()) {
            onToggleFeature(ownerId, featureState.getFeature(), true);
        }
        return result;
    }

    private void snoozeServices(FeatureState featureState) {
        serviceFeatureDao.snoozeSynchronization(featureState.getLockedServiceFeatures()
                        .filterNot(usf -> usf.getStatus() == SynchronizationStatus.ACTUAL)
                        .map(IssuedFeature::getId)
                , Instant.now().plus(Duration.standardHours(SNOOZE_DURATION_HOURS.get())));
    }

    private void deactivateFeature(FeatureState featureState, TOwner ownerId, Map<String, Object> contextForCall) {
        FeatureEntity feature = featureState.getFeature();
        if (!feature.getDeactivationRequestTemplate().isPresent()) {
            onToggleFeature(ownerId, feature, false);
            return;
        }

        DeactivationContext context = templateContextFactory.createDeactivationContext(ownerId, contextForCall);
        evaluateTemplateWithErrorsCatching(feature.getDeactivationRequestTemplate().get(),
                (e) -> getErrorProcessor(feature).skipDeactivationException(featureState, context, e),
                feature.getSystemTvmId(),
                context);
        onToggleFeature(ownerId, feature, false);
    }

    private ErrorProcessor getErrorProcessor(FeatureEntity feature) {
        return errorProcessors.getO(feature.getErrorProcessorName()).orElse(defaultErrorProcessor);
    }

    private void sendFeatureAmount(FeatureState featureState, TOwner ownerId, Map<String, Object> contextForCall) {
        FeatureEntity feature = featureState.getFeature();
        if (!feature.getSetAmountRequestTemplate().isPresent()) {
            return;
        }
        if (featureState.isNeedToActivate() && !feature.isCallSetAmountOnActivation()) {
            return;
        }
        if (featureState.isNeedToDeactivate() && !feature.isCallSetAmountOnDeactivation()) {
            return;
        }
        RequestTemplate requestTemplate = feature.getSetAmountRequestTemplate().get();
        SetAmountContext context = templateContextFactory.createSetAmountContext(ownerId, featureState.getAmount(),
                contextForCall);

        evaluateTemplateWithErrorsCatching(
                requestTemplate,
                (e) -> getErrorProcessor(feature).skipSetAmountException(featureState, context, e),
                feature.getSystemTvmId(),
                context
        );
    }

    @NotNull
    private Option<FeatureState> calculateFeatureState(UUID lockId, TOwner ownerId, UUID featureId,
                                                       Duration maxExecutionTime) {
        ListF<IssuedFeature> serviceFeatures =
                serviceFeatureDao.findAndLockEnabledOrNotActual(ownerId, featureId, lockId, maxExecutionTime);
        logger.info("fetched for ownerId {} and featureId {} serviceFeatures {}", ownerId, featureId, serviceFeatures);
        if (serviceFeatures.isEmpty()) {
            return Option.empty();
        }

        if (serviceFeatures.stream().allMatch(s -> s.getStatus() == SynchronizationStatus.ACTUAL)) {
            serviceFeatureDao.unlock(lockId);
            return Option.empty();
        }

        FeatureEntity feature = featureDao.findById(featureId);
        SetF<UUID> productFeatureIds = serviceFeatures.map(IssuedFeature::getProductFeatureId).unique();
        MapF<UUID, ProductFeatureEntity> productFeatures = productFeatureDao
                .findByIds(productFeatureIds)
                .toMapMappingToKey(ProductFeatureEntity::getId);

        ListF<IssuedFeature> wasActive = serviceFeatures.filter(AbstractFeatureActualizationService::isActivated);

        BigDecimal needAmount =
                sumAmount(productFeatures, serviceFeatures.filter(s -> s.getTarget() == Target.ENABLED));
        BigDecimal wasAmount = sumAmount(productFeatures, wasActive);

        boolean needToActivate = wasActive.isEmpty();
        boolean needToDeactivate = needAmount.compareTo(BigDecimal.ZERO) == 0;
        boolean nothingToDo = needAmount.compareTo(wasAmount) == 0;
        FeatureState featureState = new FeatureState(
                feature, serviceFeatures, needAmount, needToActivate, needToDeactivate, nothingToDo
        );
        logger.info("featureState {} wasAmount {}", featureState, wasAmount);
        return Option.of(featureState);
    }

    private static BigDecimal sumAmount(MapF<UUID, ProductFeatureEntity> productFeatures,
                                        ListF<IssuedFeature> issuedFeatures) {
        return issuedFeatures
                .map(s -> productFeatures.getTs(s.getProductFeatureId()).getAmount())
                .sum(new BigDecimalType());
    }

    private static boolean isActivated(IssuedFeature issuedFeature) {
        return issuedFeature.getActualEnabledAt().isPresent();
    }

    protected abstract Map<String, Object> fetchContextFromDb(TOwner ownerId, UUID featureId);

    protected abstract FeatureCallbackContext createFeatureCallbackContext(TOwner ownerId, UUID featureId);

    protected abstract void delete(TOwner ownerId, UUID featureId);

    protected abstract HttpHeaders buildHeaders(TemplateContext context);

    protected void onToggleFeature(TOwner ownerId, FeatureEntity feature, boolean activated) {
    }

    private Option<Map<String, Object>> activateFeature(FeatureState featureState, TOwner ownerId) {
        FeatureEntity feature = featureState.getFeature();
        FeatureCallbackContext callbackContext = createFeatureCallbackContext(ownerId, feature.getId());
        ActivationContext activationContext = templateContextFactory.createActivationContext(ownerId,
                generateUniq(callbackContext), featureState.getAmount());

        try {
            String response = evaluateTemplate(
                    feature.getActivationRequestTemplate().orElseThrow(() -> new NoSuchElementException(
                            "no activationRequestTemplate")),
                    feature.getSystemTvmId(), activationContext);

            Map<String, Object> responseAsMap = JsonHelper.getJsonToMap(response);

            featureCallbackContextDao.updateData(callbackContext.getId(), responseAsMap);
            return Option.of(responseAsMap);
        } catch (HttpStatusCodeException e) {
            if (getErrorProcessor(feature).activationErrorMustBeSnoozed(e)) {
                return Option.empty();
            }
            throw e;
        }
    }

    @NotNull
    private String generateUniq(FeatureCallbackContext callbackContext) {
        return Sha256.A.digest(callbackContext.getId().toString()).base64();
    }

    private String evaluateTemplateWithErrorsCatching(RequestTemplate requestTemplate,
                                                      Function<HttpStatusCodeException, Boolean> skipExceptionChecker,
                                                      Option<Integer> systemTvmId, TemplateContext context) {
        try {
            return evaluateTemplate(requestTemplate, systemTvmId, context);
        } catch (HttpStatusCodeException e) {
            if (!skipExceptionChecker.apply(e)) {
                throw e;
            }
            return e.getResponseBodyAsString();
        }
    }

    private String evaluateTemplate(RequestTemplate requestTemplate,
                                    Option<Integer> systemTvmId, TemplateContext context) {
        try {
            HttpHeaders headers = buildHeaders(context);
            requestTemplate.getContentType().ifPresent(headers::setContentType);
            if (systemTvmId.isPresent()) {
                headers.put(TvmHeaders.SERVICE_TICKET, tvm2.getServiceTicket(systemTvmId.get()));
            }

            Option<String> body = requestTemplate.getBodyTemplate().map(t -> fillTemplate(t, context));
            HttpEntity<?> httpEntity = new HttpEntity<>(body.orElse((String) null), headers);

            String url = fillTemplate(requestTemplate.getUrlTemplate(), context);
            logger.info("Calling {} '{}' with body entity: {}", requestTemplate.getHttpMethod(), url, body);

            ResponseEntity<String> response = externalSystems.exchange(
                    url, requestTemplate.getHttpMethod(),
                    httpEntity,
                    String.class
            );
            return response.getBody();
        } catch (HttpStatusCodeException e) {
            logger.error("response status {}, body {}", e.getStatusCode().value(), e.getResponseBodyAsString());
            throw e;
        }
    }


    private void withUnlockOnError(Function1V<UUID> callback) {
        UUID lockId = UUID.randomUUID();
        try {
            callback.apply(lockId);
        } catch (Exception e) {
            logger.info("Unlocking services {} due to error", lockId, e);
            RetryUtils.retry(logger, 3, () -> serviceFeatureDao.unlock(lockId));
            throw e;
        }
    }

    private String fillTemplate(String template, TemplateContext context) {
        Expression expression = parser.parseExpression(template, new TemplateParserContext());
        return expression.getValue(context, String.class);
    }
}

