package ru.yandex.chemodan.app.psbilling.core.synchronization.feature.errorprocessors;

import java.util.Map;
import java.util.Objects;

import org.springframework.http.HttpStatus;
import org.springframework.web.client.HttpStatusCodeException;

import ru.yandex.chemodan.app.psbilling.core.synchronization.feature.FeatureState;
import ru.yandex.chemodan.app.psbilling.core.synchronization.feature.templatecontexts.DeactivationContext;
import ru.yandex.chemodan.app.psbilling.core.synchronization.feature.templatecontexts.SetAmountContext;
import ru.yandex.chemodan.app.psbilling.core.util.JsonHelper;
import ru.yandex.inside.passport.blackbox2.Blackbox2;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;

public class MpfsErrorProcessor extends AbstractErrorProcessor {
    private static final Logger logger = LoggerFactory.getLogger(MpfsErrorProcessor.class);
    private static final String SERVICE_NOT_FOUND_ERROR_CODE = "133";
    private static final String NAME = "mpfs";
    private static final String ACCOUNT_HAS_NO_PASSWORD_CODE = "113";
    private static final String USER_BLOCKED_ERROR_CODE = "119";
    private static final String NOT_FOUND_IN_PASSPORT = "281";

    public MpfsErrorProcessor(Blackbox2 blackbox2) {
        super(blackbox2);
    }

    @Override
    public String getName() {
        return NAME;
    }

    @Override
    public boolean activationErrorMustBeSnoozed(HttpStatusCodeException e) {
        return isErrorWithCode(e, HttpStatus.CONFLICT, ACCOUNT_HAS_NO_PASSWORD_CODE)
                || isErrorWithCode(e, HttpStatus.FORBIDDEN, USER_BLOCKED_ERROR_CODE)
                || isErrorWithCode(e, HttpStatus.UNPROCESSABLE_ENTITY, NOT_FOUND_IN_PASSPORT);
    }

    @Override
    public boolean skipSetAmountException(
            FeatureState featureState,
            SetAmountContext context,
            HttpStatusCodeException e)
    {

        if (context.getUidO().isPresent() && featureState.isNeedToDeactivate() && isServiceNotFound(e)) {
            logger.warn(
                    "Service {} not found in mpfs for user {}, so we consider set amount error for feature {} as " +
                            "success",
                    context.getContext(), context.getUidO().get(), featureState.getFeature().getId());
            return true;
        }

        return super.skipSetAmountException(featureState, context, e);
    }

    @Override
    public boolean skipDeactivationException(
            FeatureState featureState,
            DeactivationContext context,
            HttpStatusCodeException e)
    {
        if (isErrorWithCode(e, HttpStatus.FORBIDDEN, USER_BLOCKED_ERROR_CODE) ||
                isErrorWithCode(e, HttpStatus.UNPROCESSABLE_ENTITY, NOT_FOUND_IN_PASSPORT)) {
            logger.warn("Valid error during deactivation (user blocked or not found in passport");
            return true;
        }
        if (context.getUidO().isPresent() && featureState.isNeedToDeactivate() && isServiceNotFound(e)) {
            logger.warn(
                    "Service {} not found in mpfs for user {}, so we consider set amount error for feature {} as " +
                            "success",
                    context.getContext(), context.getUidO().get(), featureState.getFeature().getId());
            return true;
        }

        return super.skipDeactivationException(featureState, context, e);
    }


    protected boolean isErrorWithCode(HttpStatusCodeException e, HttpStatus httpStatus, String code) {
        if (e.getStatusCode() != httpStatus) {
            return false;
        }

        Map<String, Object> json = JsonHelper.getJsonToMap(e.getResponseBodyAsString());
        return json.containsKey("code") && code.equals(Objects.toString(json.get("code")));
    }

    private boolean isServiceNotFound(HttpStatusCodeException e) {
        return isErrorWithCode(e, HttpStatus.NOT_FOUND, SERVICE_NOT_FOUND_ERROR_CODE);
    }
}
