package ru.yandex.chemodan.app.psbilling.core.synchronization.groupmember;

import java.util.UUID;

import lombok.AllArgsConstructor;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.psbilling.core.dao.groups.GroupServiceMemberDao;
import ru.yandex.chemodan.app.psbilling.core.dao.products.UserProductDao;
import ru.yandex.chemodan.app.psbilling.core.dao.users.UserServiceDao;
import ru.yandex.chemodan.app.psbilling.core.entities.groups.GroupServiceMember;
import ru.yandex.chemodan.app.psbilling.core.entities.products.BillingType;
import ru.yandex.chemodan.app.psbilling.core.entities.products.UserProductEntity;
import ru.yandex.chemodan.app.psbilling.core.entities.users.UserServiceBillingStatus;
import ru.yandex.chemodan.app.psbilling.core.synchronization.engine.SynchronizerConfig;
import ru.yandex.commune.bazinga.BazingaTaskManager;

@AllArgsConstructor
public class GroupServiceMemberActualizationService {

    private final GroupServiceMemberTableSynchronizer tablesSynchronizer;
    private final UserProductDao userProductDao;
    private final GroupServiceMemberDao groupServiceMemberDao;
    private final BazingaTaskManager bazingaTaskManager;

    public void scheduleGroupServiceMemberActualization() {
        ListF<UUID> recordsToSynchronize = groupServiceMemberDao.findRecordsInInitStatus();
        for (UUID userServiceMemberId : recordsToSynchronize) {
            bazingaTaskManager.schedule(new GroupServiceMemberActualizationTask(userServiceMemberId));
        }
    }

    public void updateGroupServiceMembersSyncState() {
        tablesSynchronizer.updateStatusInParentTable(true);
    }

    public void actualizeGroupServiceMember(UUID groupServiceMemberId) {
        tablesSynchronizer.updateDataInChildTable(groupServiceMemberId,
                new SynchronizerConfig<>(this::mapToActiveUserService,
                        userService -> userService.getId().toString())
        );
    }

    private MapF<String, UserServiceDao.InsertData> mapToActiveUserService(GroupServiceMember gsm) {
        UserProductEntity userProduct = userProductDao.findByGroupServiceId(gsm.getGroupServiceId());

        Option<UserServiceBillingStatus> status = userProduct.getBillingType() == BillingType.FREE ?
                Option.of(UserServiceBillingStatus.FREE_PERIOD) : Option.empty();

        //create new user service if reference not saved on GroupServiceMember
        UUID userServiceId = gsm.getUserServiceId().getOrElse(UUID::randomUUID);
        return Cf.map(userServiceId.toString(), UserServiceDao.InsertData.builder()
                .nextCheckDate(Option.empty())
                .uid(gsm.getUid())
                .userProductId(userProduct.getId())
                .billingStatus(status)
                .id(UUID.randomUUID())
                .build());
    }
}
