package ru.yandex.chemodan.app.psbilling.core.synchronization.groupservice;

import java.util.UUID;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Lazy;
import org.springframework.transaction.support.TransactionTemplate;

import ru.yandex.bolts.collection.ListF;
import ru.yandex.chemodan.app.psbilling.core.billing.groups.ClientBalanceCalculator;
import ru.yandex.chemodan.app.psbilling.core.config.featureflags.FeatureFlags;
import ru.yandex.chemodan.app.psbilling.core.dao.features.GroupServiceFeatureDao;
import ru.yandex.chemodan.app.psbilling.core.dao.groups.GroupProductDao;
import ru.yandex.chemodan.app.psbilling.core.dao.groups.GroupServiceDao;
import ru.yandex.chemodan.app.psbilling.core.dao.groups.GroupServiceMemberDao;
import ru.yandex.chemodan.app.psbilling.core.entities.features.GroupFeature;
import ru.yandex.chemodan.app.psbilling.core.entities.features.GroupServiceFeature;
import ru.yandex.chemodan.app.psbilling.core.entities.groups.GroupProductEntity;
import ru.yandex.chemodan.app.psbilling.core.entities.groups.GroupService;
import ru.yandex.chemodan.app.psbilling.core.entities.groups.GroupServiceMember;
import ru.yandex.chemodan.app.psbilling.core.synchronization.engine.AbstractTablesSynchronizer2;
import ru.yandex.chemodan.app.psbilling.core.synchronization.engine.Target;
import ru.yandex.chemodan.app.psbilling.core.synchronization.feature.GroupFeatureActualizationTask;
import ru.yandex.chemodan.app.psbilling.core.synchronization.groupmember.GroupServiceMemberActualizationTask;
import ru.yandex.chemodan.app.psbilling.core.tasks.execution.TaskScheduler;
import ru.yandex.commune.bazinga.BazingaTaskManager;
import ru.yandex.inside.passport.PassportUid;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;

public class GroupServiceTableSynchronizer extends
        AbstractTablesSynchronizer2<GroupService,
                GroupServiceMember, GroupServiceMemberDao.InsertData,
                GroupServiceFeature, GroupServiceFeatureDao.InsertData> {
    private static final Logger logger = LoggerFactory.getLogger(GroupServiceTableSynchronizer.class);

    private final GroupServiceFeatureDao groupServiceFeatureDao;

    @Lazy
    @Autowired
    private GroupProductDao groupProductDao;

    @Lazy
    @Autowired
    private TaskScheduler taskScheduler;

    @Lazy
    @Autowired
    private ClientBalanceCalculator clientBalanceCalculator;

    @Lazy
    @Autowired
    private FeatureFlags featureFlags;

    public GroupServiceTableSynchronizer(GroupServiceMemberDao childDao1, GroupServiceFeatureDao childDao2,
                                         GroupServiceDao parentDao,
                                         TransactionTemplate transactionTemplate,
                                         BazingaTaskManager bazingaTaskManager) {
        super(parentDao, childDao1, childDao2, transactionTemplate, bazingaTaskManager);
        this.groupServiceFeatureDao = childDao2;
    }

    @Override
    protected void scheduleChildrenSynchronization(UUID parentId) {
        ListF<GroupServiceMember> childrenInInit = childDao1.findInInitByParentId(parentId);
        childrenInInit.map(GroupServiceMember::getId).map(GroupServiceMemberActualizationTask::new)
                .forEach(bazingaTaskManager::schedule);

        ListF<GroupFeature> notActualChildren = groupServiceFeatureDao.findForSynchronization(parentId);
        notActualChildren.map(GroupFeatureActualizationTask::new)
                .forEach(bazingaTaskManager::schedule);
    }

    @Override
    protected void updateCallback1(GroupService parentRecord, UpdateResult<GroupServiceMember,
            GroupServiceMemberDao.InsertData> updateResult) {
        if (parentRecord.getTarget().equals(Target.ENABLED)) {
            clientBalanceCalculator.updateVoidDate(parentRecord);
        }
    }

    @Override
    protected void insertCallback1(GroupService parentRecord, ListF<GroupServiceMemberDao.InsertData> insertData) {
        GroupProductEntity groupProduct = groupProductDao.findByGroupServiceId(parentRecord.getId());
        for (GroupServiceMemberDao.InsertData insertDatum : insertData) {
            scheduleWelcomeEmail(insertDatum.getUid(), parentRecord, groupProduct);
        }
    }

    private void scheduleWelcomeEmail(String uidString, GroupService groupService, GroupProductEntity groupProduct) {
        try {
            long uid = Long.parseLong(uidString);
            if (featureFlags.getB2bWelcomeEmailEnabled().isEnabledForUid(PassportUid.cons(uid)) &&
                    !groupProduct.isHidden()) {
                taskScheduler.scheduleB2bWelcomeEmailTask(PassportUid.cons(uid), groupService);
            }
        } catch (NumberFormatException ex) {
            logger.error("Couldn't parse user %s as passport uid to send welcome email");
        }
    }
}
