package ru.yandex.chemodan.app.psbilling.core.synchronization.groupservice;

import java.util.UUID;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;

import lombok.RequiredArgsConstructor;

import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.psbilling.core.dao.groups.GroupDao;
import ru.yandex.chemodan.app.psbilling.core.dao.groups.GroupServiceDao;
import ru.yandex.chemodan.app.psbilling.core.entities.groups.Group;
import ru.yandex.chemodan.app.psbilling.core.entities.groups.GroupType;
import ru.yandex.chemodan.app.psbilling.core.synchronization.groupservice.actualizers.GroupServiceActualizerFactory;
import ru.yandex.commune.dynproperties.DynamicProperty;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;

@RequiredArgsConstructor
public class GroupServicesActualizationServiceImpl implements GroupServicesActualizationService {
    private static final Logger logger = LoggerFactory.getLogger(GroupServicesActualizationServiceImpl.class);

    private final GroupServiceTableSynchronizer tablesSynchronizer;
    private final GroupDao groupDao;
    private final GroupServiceDao groupServiceDao;
    private final GroupServiceActualizerFactory groupServiceActualizerFactory;


    private final DynamicProperty<Integer> groupsSynchronizationThreadsCount =
            new DynamicProperty<>("ps-billing.group-sync-threads-count", 10);

    @Override
    public void scheduleGroupActualizationTasks() {
        ListF<UUID> groupIdsWithActiveServices = groupServiceDao.groupIdsToSync();
        logger.info("got {} tasks to schedule", groupIdsWithActiveServices.size());
        ExecutorService exec = Executors.newFixedThreadPool(groupsSynchronizationThreadsCount.get());
        try {
            for (final UUID groupId : groupIdsWithActiveServices) {
                exec.submit(() -> groupServiceActualizerFactory.getActualizer(groupId).scheduleActualization());
            }
        } finally {
            exec.shutdown();
        }
    }

    @Override
    public void scheduleForceGroupActualization(GroupType groupType, String externalOrganizationId) {
        Option<Group> group = groupDao.findGroup(groupType, externalOrganizationId);
        if (group.isPresent()) {
            groupServiceActualizerFactory.getActualizer(group.get()).scheduleActualization();
        } else {
            logger.info("group for type {} and externalId {} not found", groupType, externalOrganizationId);
        }
    }

    @Override
    public void scheduleForceGroupActualization(UUID internalGroupId) {
        groupServiceActualizerFactory.getActualizer(internalGroupId).scheduleActualization();
    }

    @Override
    public void actualize(UUID internalGroupId) {
        groupServiceActualizerFactory.getActualizer(internalGroupId).actualize();
    }

    @Override
    public void updateGroupServicesSyncState() {
        tablesSynchronizer.updateStatusInParentTable(false);
    }

    @Override
    public void updateGroupServicesSyncState(int batchSize) {
        tablesSynchronizer.updateStatusInParentTable(batchSize, false);
    }
}
