package ru.yandex.chemodan.app.psbilling.core.synchronization.groupservice;

import java.util.UUID;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;

import lombok.Data;
import lombok.RequiredArgsConstructor;
import org.jetbrains.annotations.NotNull;
import org.joda.time.Duration;
import org.joda.time.Instant;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.CollectionF;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.SetF;
import ru.yandex.bolts.internal.NotImplementedException;
import ru.yandex.chemodan.app.psbilling.core.config.featureflags.FeatureFlags;
import ru.yandex.chemodan.app.psbilling.core.dao.features.GroupServiceFeatureDao;
import ru.yandex.chemodan.app.psbilling.core.dao.groups.GroupDao;
import ru.yandex.chemodan.app.psbilling.core.dao.groups.GroupProductDao;
import ru.yandex.chemodan.app.psbilling.core.dao.groups.GroupServiceDao;
import ru.yandex.chemodan.app.psbilling.core.dao.groups.GroupServiceMemberDao;
import ru.yandex.chemodan.app.psbilling.core.dao.products.ProductFeatureDao;
import ru.yandex.chemodan.app.psbilling.core.dao.products.ProductTemplateFeatureDao;
import ru.yandex.chemodan.app.psbilling.core.entities.AbstractEntity;
import ru.yandex.chemodan.app.psbilling.core.entities.features.GroupServiceFeature;
import ru.yandex.chemodan.app.psbilling.core.entities.groups.BalancePaymentInfo;
import ru.yandex.chemodan.app.psbilling.core.entities.groups.Group;
import ru.yandex.chemodan.app.psbilling.core.entities.groups.GroupProductEntity;
import ru.yandex.chemodan.app.psbilling.core.entities.groups.GroupService;
import ru.yandex.chemodan.app.psbilling.core.entities.groups.GroupServiceMember;
import ru.yandex.chemodan.app.psbilling.core.entities.groups.GroupType;
import ru.yandex.chemodan.app.psbilling.core.entities.products.FeatureScope;
import ru.yandex.chemodan.app.psbilling.core.entities.products.ProductFeatureEntity;
import ru.yandex.chemodan.app.psbilling.core.staff.YandexStaffService;
import ru.yandex.chemodan.app.psbilling.core.synchronization.engine.SynchronizerConfig;
import ru.yandex.chemodan.app.psbilling.core.synchronization.engine.Target;
import ru.yandex.chemodan.app.psbilling.core.tasks.execution.TaskScheduler;
import ru.yandex.chemodan.directory.client.DirectoryClient;
import ru.yandex.chemodan.directory.client.DirectoryOrganizationFeaturesResponse;
import ru.yandex.chemodan.directory.client.FetchLimitExceededException;
import ru.yandex.chemodan.directory.client.OrganizationNotFoundException;
import ru.yandex.commune.dynproperties.DynamicProperty;
import ru.yandex.inside.passport.PassportUid;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;

@RequiredArgsConstructor
public class GroupServicesActualizationServiceOld implements GroupServicesActualizationService {
    private static final Logger logger = LoggerFactory.getLogger(GroupServicesActualizationServiceOld.class);

    private final GroupProductDao groupProductDao;
    private final ProductFeatureDao productFeatureDao;
    private final ProductTemplateFeatureDao productTemplateFeatureDao;
    private final DirectoryClient directoryClient;
    private final GroupServiceTableSynchronizer tablesSynchronizer;
    private final TaskScheduler taskScheduler;
    private final GroupDao groupDao;
    private final GroupServiceDao groupServiceDao;
    private final YandexStaffService yandexStaffService;
    private final FeatureFlags featureFlags;

    private final DynamicProperty<Integer> directoryPageSize =
            new DynamicProperty<>("ps-billing.directory-client.page-size", 500);
    private final DynamicProperty<Integer> organizationSizeLimit =
            new DynamicProperty<>("ps-billing.directory-client.fetch-organization-size-limit", 100_000);
    private final DynamicProperty<Integer> maxGroupWaitForCommitSeconds =
            new DynamicProperty<>("ps-billing.max-wait-group-for-commit-seconds", 60);
    private final DynamicProperty<Integer> groupsSynchronizationIntervalMinutes =
            new DynamicProperty<>("ps-billing.group-sync-interval-minutes", 60);
    private final DynamicProperty<Integer> groupsSynchronizationThreadsCount =
            new DynamicProperty<>("ps-billing.group-sync-threads-count", 10);

    @Override
    public void scheduleGroupActualizationTasks() {
        ListF<UUID> groupIdsWithActiveServices = groupServiceDao.groupIdsToSync();
        logger.info("got {} tasks to schedule", groupIdsWithActiveServices.size());
        ExecutorService exec = Executors.newFixedThreadPool(groupsSynchronizationThreadsCount.get());
        try {
            for (final UUID groupId : groupIdsWithActiveServices) {
                exec.submit(() -> {
                    taskScheduler.schedule(new GroupActualizationTask(groupId));
                });
            }
        } finally {
            exec.shutdown();
        }
    }

    @Override
    public void scheduleForceGroupActualization(GroupType groupType, String externalOrganizationId) {
        Option<Group> group = groupDao.findGroup(groupType, externalOrganizationId);
        if (group.isPresent()) {
            scheduleForceGroupActualization(group.get().getId());
        } else {
            logger.info("group for type {} and externalId {} not found", groupType, externalOrganizationId);
        }
    }

    @Override
    public void scheduleForceGroupActualization(UUID internalGroupId) {
        groupDao.updateGroupNextSyncTime(internalGroupId, Instant.now());
        taskScheduler.schedule(new GroupActualizationTask(internalGroupId));
    }

    @Override
    public void actualize(UUID internalGroupId)  {
        try {
            actualizeWhileNecessary(internalGroupId);
        } catch (InterruptedException e) {
            throw new RuntimeException(e);
        }
    }

    public void actualizeWhileNecessary(UUID internalGroupId) throws InterruptedException {
        Group group;
        do {
            group = actualizeByGroupId(internalGroupId);
        } while (!groupDao.updateGroupNextSyncTimeIfNotChanged(group.getId(), group.getMembersNextSyncDt(),
                Instant.now().plus(Duration.standardMinutes(groupsSynchronizationIntervalMinutes.get()))));

        if (featureFlags.getAutoPayDisableOnUserIsNotAdmin().isEnabled()) {
            Option<BalancePaymentInfo> paymentInfo = group.getPaymentInfo();
            if (paymentInfo.isPresent() && paymentInfo.get().isB2bAutoBillingEnabled()) {
                PassportUid payer = paymentInfo.get().getPassportUid();
                if (!isPayerAdminInOrgs(payer)) {
                    logger.info("payer {} isn't admin in groups: " +
                                    "disable auto billing for clientId {}",
                            payer.toString(), paymentInfo.get().getClientId());
                    groupDao.setAutoBillingForClient(paymentInfo.get().getClientId(), false);
                }
            }
        }
    }

    private Group actualizeByGroupId(UUID internalGroupId) throws InterruptedException {
        Group group = findGroup(internalGroupId, Duration.standardSeconds(maxGroupWaitForCommitSeconds.get()));

        logger.info("Start synchronizing members of group {}", group);

        ListF<GroupService> servicesToSync = groupServiceDao.findGroupServicesToSync(group.getId());
        if (servicesToSync.isEmpty()) {
            return group;
        }

        SynchronizerConfig<GroupService, GroupServiceMember,
                GroupServiceMemberDao.InsertData> userMembersSyncConfig =
                new SynchronizerConfig<>(false, null, null);

        if (!group.isSynchronizationStopped()) {
            final Option<Boolean> organizationDisabledForBilling = isOrganizationDisabledForBilling(group);

            if (organizationDisabledForBilling.orElse(false)) {
                ListF<GroupService> paidGroupServices = groupServiceDao.find(group.getId(), Target.ENABLED)
                        .filter(gs -> !gs.isSkipTransactionsExport());
                paidGroupServices.forEach(gs -> groupServiceDao.setTargetToDisabled(gs.getId(), gs.getTarget()));
            }

            GroupMembersInfo groupMembersInfo = null;
            try {
                groupMembersInfo = getGroupMembersInfo(group);
            } catch (TooBigGroupException e) {
                logger.warn("synchronization skipped for group {} due to it's big size", group);
            }

            if (groupMembersInfo != null) {
                if (!groupMembersInfo.getGroupMembers().isPresent()) {
                    logger.warn("Group {} was deleted, disabling it's services", group);
                    servicesToSync.forEach(gs -> groupServiceDao.setTargetToDisabled(gs.getId(), gs.getTarget()));
                }

                CollectionF<String> users = groupMembersInfo.getGroupMembers().orElse(Cf.set());

                userMembersSyncConfig =
                        new SynchronizerConfig<>(
                                groupService ->
                                        users.toMap(uid -> uid, uid -> newGroupServiceMember(groupService, uid)),
                                GroupServiceMember::getUid);
            }
        }

        SynchronizerConfig<GroupService, GroupServiceFeature, GroupServiceFeatureDao.InsertData>
                groupServiceFeaturesConfig =
                new SynchronizerConfig<>(this::mapToGroupServiceFeatures,
                        groupServiceFeature -> groupServiceFeature.getProductFeatureId().toString());

        for (GroupService activeService : servicesToSync) {
            tablesSynchronizer.updateDataInChildTable(activeService.getId(),
                    userMembersSyncConfig,
                    groupServiceFeaturesConfig
            );
        }

        return group;
    }

    @NotNull
    private Group findGroup(UUID internalGroupId, Duration maxWaitTime) throws InterruptedException {
        Option<Group> groupO = Option.empty();
        Instant deadline = Instant.now().plus(maxWaitTime);
        while (!groupO.isPresent() && deadline.isAfterNow()) {
            groupO = groupDao.findByIdO(internalGroupId);
            if (groupO.isPresent()) {
                return groupO.get();
            } else {
                Thread.sleep(5000);
            }
        }

        throw new IllegalStateException("Group with id " + internalGroupId + " not found");
    }

    public GroupMembersInfo getGroupMembersInfo(Group group) throws TooBigGroupException {
        switch (group.getType()) {
            case ORGANIZATION:
                return getOrganizationMembers(group);
            case STAFF:
                return new GroupMembersInfo(getStaffMembers(group));
            case FAMILY:
                throw new NotImplementedException();
            default:
                throw new UnsupportedOperationException("Unexpected value: " + group.getType());
        }

    }

    private Option<Boolean> isOrganizationDisabledForBilling(Group group) {
        try {
            DirectoryOrganizationFeaturesResponse features =
                    directoryClient.getOrganizationFeatures(group.getExternalId());
            return Option.of(features.isDisablePsbillingProcessingActive());
        } catch (OrganizationNotFoundException e) {
            logger.warn("Organization {} was empty deleted", group.getExternalId());
            return Option.empty();
        }
    }

    private GroupMembersInfo getOrganizationMembers(Group group) throws TooBigGroupException {
        try {
            final Option<CollectionF<String>> groupMembers = Option.of(
                    directoryClient.usersInOrganization(
                                    group.getExternalId(),
                                    true,
                                    directoryPageSize.get(),
                                    Option.of(organizationSizeLimit.get())
                            )
                            .map(String::valueOf)
            );
            return new GroupMembersInfo(groupMembers);
        } catch (OrganizationNotFoundException e) {
            logger.warn("Organization {} was empty deleted", group.getExternalId());
            return new GroupMembersInfo(Option.empty());
        } catch (FetchLimitExceededException e) {
            throw new TooBigGroupException(group, e);
        }
    }

    private Option<CollectionF<String>> getStaffMembers(Group group) {
        return Option.of(yandexStaffService.getUidsForSyncing().map(PassportUid::toString).unique());
    }

    public void updateGroupServicesSyncState() {
        tablesSynchronizer.updateStatusInParentTable(false);
    }

    public void updateGroupServicesSyncState(int batchSize) {
        tablesSynchronizer.updateStatusInParentTable(batchSize, false);
    }

    private static GroupServiceMemberDao.InsertData newGroupServiceMember(GroupService activeService, String uid) {
        return GroupServiceMemberDao.InsertData.builder().groupServiceId(activeService.getId()).uid(uid).build();
    }

    @Data
    public static class GroupMembersInfo {
        private final Option<CollectionF<String>> groupMembers;
    }

    private MapF<String, GroupServiceFeatureDao.InsertData> mapToGroupServiceFeatures(GroupService groupService) {
        GroupProductEntity groupProduct = groupProductDao.findById(groupService.getGroupProductId());
        ListF<ProductFeatureEntity> productFeatures =
                productFeatureDao.findEnabledByUserProduct(groupProduct.getUserProductId());
        //create for only product features with feature_id
        return productFeatures.filter(pf -> pf.getFeatureId().isPresent() && pf.getScope().equals(FeatureScope.GROUP))
                .toMap(f -> f.getId().toString(),
                        f -> GroupServiceFeatureDao.InsertData.builder()
                                .productFeatureId(f.getId())
                                .productTemplateFeatureId(productTemplateFeatureDao.findByProductFeatureId(f.getId())
                                        .map(AbstractEntity::getId).orElse((UUID) null))
                                .groupServiceId(groupService.getId())
                                .groupId(groupService.getGroupId())
                                .build());
    }

    // Является ли плательщик админом хотя бы в одной из организаций, где он плательщик?
    public boolean isPayerAdminInOrgs(PassportUid payer) {
        SetF<String> groupsWherePayer = Cf.toSet(
                groupDao.findGroupsByPaymentInfoUid(payer).map(Group::getExternalId));
        SetF<String> groupsWhereBelongs = Cf.toSet(
                directoryClient.organizationsWhereUserIsAdmin(
                        payer.toString(), 10));

        return groupsWherePayer.intersects(groupsWhereBelongs);
    }
}
