package ru.yandex.chemodan.app.psbilling.core.synchronization.groupservice.actualizers;

import java.util.UUID;

import lombok.RequiredArgsConstructor;
import org.jetbrains.annotations.NotNull;
import org.joda.time.Duration;
import org.joda.time.Instant;

import ru.yandex.bolts.collection.CollectionF;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.psbilling.core.config.featureflags.FeatureFlags;
import ru.yandex.chemodan.app.psbilling.core.dao.features.GroupServiceFeatureDao;
import ru.yandex.chemodan.app.psbilling.core.dao.groups.GroupDao;
import ru.yandex.chemodan.app.psbilling.core.dao.groups.GroupProductDao;
import ru.yandex.chemodan.app.psbilling.core.dao.groups.GroupServiceDao;
import ru.yandex.chemodan.app.psbilling.core.dao.groups.GroupServiceMemberDao;
import ru.yandex.chemodan.app.psbilling.core.dao.products.ProductFeatureDao;
import ru.yandex.chemodan.app.psbilling.core.dao.products.ProductTemplateFeatureDao;
import ru.yandex.chemodan.app.psbilling.core.directory.DirectoryService;
import ru.yandex.chemodan.app.psbilling.core.directory.GroupMembersInfo;
import ru.yandex.chemodan.app.psbilling.core.entities.AbstractEntity;
import ru.yandex.chemodan.app.psbilling.core.entities.features.GroupServiceFeature;
import ru.yandex.chemodan.app.psbilling.core.entities.groups.BalancePaymentInfo;
import ru.yandex.chemodan.app.psbilling.core.entities.groups.Group;
import ru.yandex.chemodan.app.psbilling.core.entities.groups.GroupProductEntity;
import ru.yandex.chemodan.app.psbilling.core.entities.groups.GroupService;
import ru.yandex.chemodan.app.psbilling.core.entities.groups.GroupServiceMember;
import ru.yandex.chemodan.app.psbilling.core.entities.products.FeatureScope;
import ru.yandex.chemodan.app.psbilling.core.entities.products.ProductFeatureEntity;
import ru.yandex.chemodan.app.psbilling.core.synchronization.engine.SynchronizerConfig;
import ru.yandex.chemodan.app.psbilling.core.synchronization.engine.Target;
import ru.yandex.chemodan.app.psbilling.core.synchronization.groupservice.GroupActualizationTask;
import ru.yandex.chemodan.app.psbilling.core.synchronization.groupservice.GroupServiceTableSynchronizer;
import ru.yandex.chemodan.app.psbilling.core.synchronization.groupservice.TooBigGroupException;
import ru.yandex.chemodan.app.psbilling.core.tasks.execution.TaskScheduler;
import ru.yandex.commune.dynproperties.DynamicProperty;
import ru.yandex.inside.passport.PassportUid;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;

@RequiredArgsConstructor
public abstract class AbstractGroupServicesActualizer implements GroupServiceActualizer {
    private static final Logger logger = LoggerFactory.getLogger(AbstractGroupServicesActualizer.class);

    private final GroupProductDao groupProductDao;
    private final ProductFeatureDao productFeatureDao;
    private final ProductTemplateFeatureDao productTemplateFeatureDao;
    private final DirectoryService directoryService;
    private final GroupServiceTableSynchronizer tablesSynchronizer;
    private final TaskScheduler taskScheduler;
    private final GroupDao groupDao;
    private final GroupServiceDao groupServiceDao;
    private final FeatureFlags featureFlags;
    private final Group group;

    private final DynamicProperty<Integer> maxGroupWaitForCommitSeconds =
            new DynamicProperty<>("ps-billing.max-wait-group-for-commit-seconds", 60);
    private final DynamicProperty<Integer> groupsSynchronizationIntervalMinutes =
            new DynamicProperty<>("ps-billing.group-sync-interval-minutes", 60);

    @Override
    public void scheduleActualization() {
        groupDao.updateGroupNextSyncTime(group.getId(), Instant.now());
        taskScheduler.schedule(new GroupActualizationTask(group.getId()));
    }

    @Override
    public void actualize() {
        Group innerGroup;
        do {
            innerGroup = actualizeByGroupId(group.getId());
        } while (!groupDao.updateGroupNextSyncTimeIfNotChanged(innerGroup.getId(), innerGroup.getMembersNextSyncDt(),
                Instant.now().plus(Duration.standardMinutes(groupsSynchronizationIntervalMinutes.get()))));

        if (featureFlags.getAutoPayDisableOnUserIsNotAdmin().isEnabled()) {
            Option<BalancePaymentInfo> paymentInfo = innerGroup.getPaymentInfo();
            if (paymentInfo.isPresent() && paymentInfo.get().isB2bAutoBillingEnabled()) {
                PassportUid payer = paymentInfo.get().getPassportUid();
                if (!directoryService.isPayerAdminInOrgs(payer)) {
                    logger.info("payer {} isn't admin in groups: " +
                                    "disable auto billing for clientId {}",
                            payer.toString(), paymentInfo.get().getClientId());
                    groupDao.setAutoBillingForClient(paymentInfo.get().getClientId(), false);
                }
            }
        }
    }

    private Group actualizeByGroupId(UUID internalGroupId) {
        Group group = findGroup(internalGroupId, Duration.standardSeconds(maxGroupWaitForCommitSeconds.get()));

        logger.info("Start synchronizing members of group {}", group);

        ListF<GroupService> servicesToSync = groupServiceDao.findGroupServicesToSync(group.getId());
        if (servicesToSync.isEmpty()) {
            return group;
        }

        SynchronizerConfig<GroupService, GroupServiceMember,
                GroupServiceMemberDao.InsertData> userMembersSyncConfig =
                new SynchronizerConfig<>(false, null, null);

        if (!group.isSynchronizationStopped()) {
            boolean billingEnabled = isBillingEnabled();

            if (!billingEnabled) {
                logger.warn("Billing disabled for group {}. Disabling paid services", group);
                ListF<GroupService> paidGroupServices = groupServiceDao.find(group.getId(), Target.ENABLED)
                        .filter(gs -> !gs.isSkipTransactionsExport());
                paidGroupServices.forEach(gs -> groupServiceDao.setTargetToDisabled(gs.getId(), gs.getTarget()));
            }

            GroupMembersInfo groupMembersInfo;
            try {
                groupMembersInfo = getGroupMembersInfo();
                if (!groupMembersInfo.isGroupFound()) {
                    logger.warn("Group {} was deleted, disabling it's services", group);
                    servicesToSync.forEach(gs -> groupServiceDao.setTargetToDisabled(gs.getId(), gs.getTarget()));
                }

                CollectionF<String> users = groupMembersInfo.getGroupMembers();

                userMembersSyncConfig =
                        new SynchronizerConfig<>(
                                groupService ->
                                        users.toMap(uid -> uid, uid -> newGroupServiceMember(groupService, uid)),
                                GroupServiceMember::getUid);
            } catch (TooBigGroupException e) {
                logger.warn("synchronization skipped for group {} due to it's big size", group);
            }
        }

        SynchronizerConfig<GroupService, GroupServiceFeature, GroupServiceFeatureDao.InsertData>
                groupServiceFeaturesConfig =
                new SynchronizerConfig<>(this::mapToGroupServiceFeatures,
                        groupServiceFeature -> groupServiceFeature.getProductFeatureId().toString());

        for (GroupService activeService : servicesToSync) {
            tablesSynchronizer.updateDataInChildTable(activeService.getId(),
                    userMembersSyncConfig,
                    groupServiceFeaturesConfig
            );
        }

        return group;
    }

    @NotNull
    private Group findGroup(UUID internalGroupId, Duration maxWaitTime) {
        Option<Group> groupO = Option.empty();
        Instant deadline = Instant.now().plus(maxWaitTime);
        while (!groupO.isPresent() && deadline.isAfterNow()) {
            groupO = groupDao.findByIdO(internalGroupId);
            if (groupO.isPresent()) {
                return groupO.get();
            } else {
                try {
                    Thread.sleep(5000);
                } catch (InterruptedException e) {
                    throw new RuntimeException(e);
                }
            }
        }

        throw new IllegalStateException("Group with id " + internalGroupId + " not found");
    }

    @NotNull
    abstract GroupMembersInfo getGroupMembersInfo() throws TooBigGroupException;

    abstract boolean isBillingEnabled();

    private static GroupServiceMemberDao.InsertData newGroupServiceMember(GroupService activeService, String uid) {
        return GroupServiceMemberDao.InsertData.builder().groupServiceId(activeService.getId()).uid(uid).build();
    }

    private MapF<String, GroupServiceFeatureDao.InsertData> mapToGroupServiceFeatures(GroupService groupService) {
        GroupProductEntity groupProduct = groupProductDao.findById(groupService.getGroupProductId());
        ListF<ProductFeatureEntity> productFeatures =
                productFeatureDao.findEnabledByUserProduct(groupProduct.getUserProductId());
        //create for only product features with feature_id
        return productFeatures.filter(pf -> pf.getFeatureId().isPresent() && pf.getScope().equals(FeatureScope.GROUP))
                .toMap(f -> f.getId().toString(),
                        f -> GroupServiceFeatureDao.InsertData.builder()
                                .productFeatureId(f.getId())
                                .productTemplateFeatureId(productTemplateFeatureDao.findByProductFeatureId(f.getId())
                                        .map(AbstractEntity::getId).orElse((UUID) null))
                                .groupServiceId(groupService.getId())
                                .groupId(groupService.getGroupId())
                                .build());
    }
}
