package ru.yandex.chemodan.app.psbilling.core.synchronization.groupservice.actualizers;

import org.jetbrains.annotations.NotNull;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.chemodan.app.psbilling.core.config.featureflags.FeatureFlags;
import ru.yandex.chemodan.app.psbilling.core.dao.groups.GroupDao;
import ru.yandex.chemodan.app.psbilling.core.dao.groups.GroupProductDao;
import ru.yandex.chemodan.app.psbilling.core.dao.groups.GroupServiceDao;
import ru.yandex.chemodan.app.psbilling.core.dao.products.ProductFeatureDao;
import ru.yandex.chemodan.app.psbilling.core.dao.products.ProductTemplateFeatureDao;
import ru.yandex.chemodan.app.psbilling.core.directory.DirectoryService;
import ru.yandex.chemodan.app.psbilling.core.directory.GroupMembersInfo;
import ru.yandex.chemodan.app.psbilling.core.entities.groups.Group;
import ru.yandex.chemodan.app.psbilling.core.entities.groups.GroupType;
import ru.yandex.chemodan.app.psbilling.core.synchronization.groupservice.GroupServiceTableSynchronizer;
import ru.yandex.chemodan.app.psbilling.core.synchronization.groupservice.TooBigGroupException;
import ru.yandex.chemodan.app.psbilling.core.tasks.execution.TaskScheduler;
import ru.yandex.chemodan.directory.client.DirectoryOrganizationFeaturesResponse;
import ru.yandex.chemodan.directory.client.OrganizationNotFoundException;
import ru.yandex.commune.dynproperties.DynamicProperty;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;

public class OrganizationGroupServicesActualizer extends AbstractGroupServicesActualizer {
    private final DynamicProperty<Integer> directoryPageSize =
            new DynamicProperty<>("ps-billing.directory-client.page-size", 500);
    private final DynamicProperty<Integer> organizationSizeLimit =
            new DynamicProperty<>("ps-billing.directory-client.fetch-organization-size-limit", 100_000);

    private static final Logger logger = LoggerFactory.getLogger(OrganizationGroupServicesActualizer.class);
    private final GroupProductDao groupProductDao;
    private final ProductFeatureDao productFeatureDao;
    private final ProductTemplateFeatureDao productTemplateFeatureDao;
    private final DirectoryService directoryService;
    private final GroupServiceTableSynchronizer tablesSynchronizer;
    private final TaskScheduler taskScheduler;
    private final GroupDao groupDao;
    private final GroupServiceDao groupServiceDao;
    private final FeatureFlags featureFlags;
    private final Group group;
    private GroupMembersInfo groupMembersInfoCache;
    private Boolean isBillingEnabledCache;


    OrganizationGroupServicesActualizer(GroupProductDao groupProductDao, ProductFeatureDao productFeatureDao,
                                        ProductTemplateFeatureDao productTemplateFeatureDao, DirectoryService directoryService,
                                        GroupServiceTableSynchronizer tablesSynchronizer,
                                        TaskScheduler taskScheduler, GroupDao groupDao,
                                        GroupServiceDao groupServiceDao,
                                        FeatureFlags featureFlags, Group group) {
        super(groupProductDao, productFeatureDao, productTemplateFeatureDao, directoryService, tablesSynchronizer,
                taskScheduler, groupDao, groupServiceDao, featureFlags, group);
        this.directoryService = directoryService;
        this.group = group;
        this.groupProductDao = groupProductDao;
        this.productFeatureDao = productFeatureDao;
        this.productTemplateFeatureDao = productTemplateFeatureDao;
        this.tablesSynchronizer = tablesSynchronizer;
        this.taskScheduler = taskScheduler;
        this.groupDao = groupDao;
        this.groupServiceDao = groupServiceDao;
        this.featureFlags = featureFlags;
    }

    @Override
    public void actualize() {
        super.actualize();//actualize self
        //actualize children
        if (groupMembersInfoCache == null) {
            try {
                groupMembersInfoCache = getGroupMembersInfo();
            } catch (TooBigGroupException e) {
                logger.error(e);
            }
            if (groupMembersInfoCache == null)
                return; //no cache no sync;
        } if (isBillingEnabledCache == null)
            isBillingEnabledCache = isBillingEnabled();

        group.getChildGroups().stream()
                .filter(child -> child.getType().equals(GroupType.ORGANIZATION_USER))
                .map(child -> buildChildActualizer(child, groupMembersInfoCache, isBillingEnabledCache))
                .forEach(AbstractGroupServicesActualizer::actualize);

    }

    private @NotNull OrganizationGroupServicesActualizer.UserGroupServiceInnerActualizer buildChildActualizer(
            Group child, GroupMembersInfo groupMembersInfo, Boolean isBillingEnabledCache) {
        GroupMembersInfo childMembers;
        if (groupMembersInfo.getGroupMembers().containsTs(child.getExternalId())) {
            childMembers = new GroupMembersInfo(Cf.set(child.getExternalId()));
        } else {
            childMembers = new GroupMembersInfo(Cf.set());
        }
        return new UserGroupServiceInnerActualizer(groupProductDao, productFeatureDao,
        productTemplateFeatureDao, directoryService, tablesSynchronizer, taskScheduler, groupDao,
        groupServiceDao, featureFlags, child, childMembers, isBillingEnabledCache);
    }

    @Override
    @NotNull GroupMembersInfo getGroupMembersInfo() throws TooBigGroupException {
        groupMembersInfoCache = directoryService.getGroupMembersInfo(group);
        return groupMembersInfoCache;
    }

    @Override
    public boolean isBillingEnabled() {
        try {
            DirectoryOrganizationFeaturesResponse features = directoryService.getOrganizationFeatures(group);
            return !features.isDisablePsbillingProcessingActive();
        } catch (OrganizationNotFoundException e) {
            logger.warn("Organization {} was empty deleted", group.getExternalId());
            return true;
        }

    }

    private static class UserGroupServiceInnerActualizer extends AbstractGroupServicesActualizer {

        private final @NotNull GroupMembersInfo groupMembersInfo;
        private final boolean isBillingEnabled;

        public UserGroupServiceInnerActualizer(GroupProductDao groupProductDao, ProductFeatureDao productFeatureDao,
                                               ProductTemplateFeatureDao productTemplateFeatureDao,
                                               DirectoryService directoryService,
                                               GroupServiceTableSynchronizer tablesSynchronizer,
                                               TaskScheduler taskScheduler, GroupDao groupDao,
                                               GroupServiceDao groupServiceDao, FeatureFlags featureFlags, Group group,
                                               @NotNull GroupMembersInfo groupMembersInfo, boolean isBillingEnabled) {
            super(groupProductDao, productFeatureDao, productTemplateFeatureDao, directoryService, tablesSynchronizer,
                    taskScheduler, groupDao, groupServiceDao, featureFlags, group);
            this.groupMembersInfo = groupMembersInfo;
            this.isBillingEnabled = isBillingEnabled;
        }

        @Override
        @NotNull GroupMembersInfo getGroupMembersInfo() {
            return groupMembersInfo;
        }

        @Override
        boolean isBillingEnabled() {
            return isBillingEnabled;
        }
    }
}
