package ru.yandex.chemodan.app.psbilling.core.synchronization.userservice;

import java.util.UUID;

import lombok.AllArgsConstructor;
import org.joda.time.Duration;
import org.joda.time.Instant;

import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.chemodan.app.psbilling.core.dao.features.UserServiceFeatureDao;
import ru.yandex.chemodan.app.psbilling.core.dao.products.ProductFeatureDao;
import ru.yandex.chemodan.app.psbilling.core.dao.products.ProductTemplateFeatureDao;
import ru.yandex.chemodan.app.psbilling.core.dao.users.UserServiceDao;
import ru.yandex.chemodan.app.psbilling.core.entities.AbstractEntity;
import ru.yandex.chemodan.app.psbilling.core.entities.products.FeatureScope;
import ru.yandex.chemodan.app.psbilling.core.entities.products.ProductFeatureEntity;
import ru.yandex.chemodan.app.psbilling.core.entities.users.UserServiceEntity;
import ru.yandex.chemodan.app.psbilling.core.synchronization.engine.SynchronizerConfig;
import ru.yandex.commune.bazinga.BazingaTaskManager;

@AllArgsConstructor
public class UserServiceActualizationService {

    private final UserServiceTableSynchronizer tablesSynchronizer;
    private final BazingaTaskManager bazingaTaskManager;
    private final ProductFeatureDao productFeatureDao;
    private final UserServiceDao userServiceDao;
    private final ProductTemplateFeatureDao productTemplateFeatureDao;

    public void scheduleUserServiceActualization(UUID userServiceId) {
        bazingaTaskManager.schedule(new UserServiceActualizationTask(userServiceId));
    }

    public void scheduleUserServiceActualization(UUID userServiceId, Duration scheduleDelay) {
        bazingaTaskManager.schedule(new UserServiceActualizationTask(userServiceId), Instant.now().plus(scheduleDelay));
    }

    public void scheduleUserServicesActualization() {
        ListF<UUID> userServiceIds = userServiceDao.findRecordsInInitStatus();
        for (UUID userServiceId : userServiceIds) {
            bazingaTaskManager.schedule(new UserServiceActualizationTask(userServiceId));
        }
    }

    public void updateUserServicesSyncState() {
        tablesSynchronizer.updateStatusInParentTable(true);
    }

    public void actualizeUserService(UUID userServiceId) {
        tablesSynchronizer.updateDataInChildTable(userServiceId,
                new SynchronizerConfig<>(this::mapToUserServiceFeatures,
                        userServiceFeature -> userServiceFeature.getProductFeatureId().toString())
        );
    }

    private MapF<String, UserServiceFeatureDao.InsertData> mapToUserServiceFeatures(UserServiceEntity userService) {
        ListF<ProductFeatureEntity> productFeatures =
                productFeatureDao.findEnabledByUserProduct(userService.getUserProductId());
        //create for only product features with feature_id
        return productFeatures.filter(pf -> pf.getFeatureId().isPresent() && pf.getScope().equals(FeatureScope.USER))
                .toMap(f -> f.getId().toString(),
                        f -> UserServiceFeatureDao.InsertData.builder()
                                .productFeatureId(f.getId())
                                .productTemplateFeatureId(productTemplateFeatureDao.findByProductFeatureId(f.getId())
                                        .map(AbstractEntity::getId).orElse((UUID) null))
                                .userServiceId(userService.getId())
                                .uid(userService.getUid())
                                .build());
    }
}
