package ru.yandex.chemodan.app.psbilling.core.tasks.execution;

import java.util.UUID;

import org.joda.time.Duration;
import org.joda.time.Instant;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Lazy;
import org.springframework.transaction.annotation.Transactional;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.SetF;
import ru.yandex.chemodan.app.psbilling.core.config.featureflags.FeatureFlags;
import ru.yandex.chemodan.app.psbilling.core.entities.CustomPeriod;
import ru.yandex.chemodan.app.psbilling.core.entities.groups.Group;
import ru.yandex.chemodan.app.psbilling.core.entities.groups.GroupService;
import ru.yandex.chemodan.app.psbilling.core.entities.users.Order;
import ru.yandex.chemodan.app.psbilling.core.entities.users.OrderType;
import ru.yandex.chemodan.app.psbilling.core.mail.EventMailType;
import ru.yandex.chemodan.app.psbilling.core.mail.GroupServiceData;
import ru.yandex.chemodan.app.psbilling.core.mail.MailContext;
import ru.yandex.chemodan.app.psbilling.core.mail.Utils;
import ru.yandex.chemodan.app.psbilling.core.mail.tasks.B2BAdminAbandonedPaymentTask;
import ru.yandex.chemodan.app.psbilling.core.mail.tasks.B2BTariffAcquisitionEmailTask;
import ru.yandex.chemodan.app.psbilling.core.mail.tasks.BaseEmailTask;
import ru.yandex.chemodan.app.psbilling.core.mail.tasks.EmailPaymentsAwareMembersTask;
import ru.yandex.chemodan.app.psbilling.core.mail.tasks.SendLocalizedEmailTask;
import ru.yandex.chemodan.app.psbilling.core.mail.tasks.SendTransactionalEmailWithInvoiceCreationTask;
import ru.yandex.chemodan.app.psbilling.core.products.GroupProduct;
import ru.yandex.chemodan.app.psbilling.core.promos.tasks.ActivatePromoTask;
import ru.yandex.chemodan.app.psbilling.core.promos.tasks.PaymentErrorContext;
import ru.yandex.chemodan.app.psbilling.core.tasks.BaseTask;
import ru.yandex.chemodan.app.psbilling.core.tasks.BaseTaskParams;
import ru.yandex.chemodan.app.psbilling.core.tasks.execution.mail.LocalizedEmailTaskExecutor;
import ru.yandex.chemodan.app.psbilling.core.tasks.execution.mail.TransactionalEmailExecutor;
import ru.yandex.chemodan.app.psbilling.core.tasks.execution.promo.ActivatePromoTaskTaskExecutor;
import ru.yandex.chemodan.app.psbilling.core.tasks.policies.TaskPreExecutionPoliciesHolder;
import ru.yandex.chemodan.app.psbilling.core.tasks.policies.mail.EmailNeverSentBeforeTaskPreExecutionPolicy;
import ru.yandex.chemodan.app.psbilling.core.tasks.policies.mail.OnlyFreeUsersEmailTaskPreExecutionPolicy;
import ru.yandex.chemodan.app.psbilling.core.tasks.policies.mail.RegularEmailTaskPreExecutionPolicy;
import ru.yandex.chemodan.app.psbilling.core.tasks.policies.promo.PromoActivationPreExecutionPolicy;
import ru.yandex.chemodan.app.uaas.experiments.ExperimentsManager;
import ru.yandex.commune.bazinga.BazingaTaskManager;
import ru.yandex.commune.bazinga.impl.FullJobId;
import ru.yandex.commune.bazinga.scheduler.OnetimeTaskSupport;
import ru.yandex.commune.dynproperties.DynamicProperty;
import ru.yandex.commune.util.RetryUtils;
import ru.yandex.inside.passport.PassportUid;
import ru.yandex.inside.utils.Language;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;

import static ru.yandex.chemodan.app.psbilling.core.mail.Utils.toDuration;

public class TaskScheduler {
    private final static Logger logger = LoggerFactory.getLogger(TaskScheduler.class);

    private final ExperimentsManager experimentsManager;
    private final BazingaTaskManager bazingaTaskManager;
    private final FeatureFlags featureFlags;
    private static final int SCHEDULE_TASK_RETRY_COUNT = 3;
    private static final int SCHEDULE_TASK_INITIAL_SLEEP_MS = 500;
    private static final int SCHEDULE_TASK_COEFF = 2;

    private static final SetF<EventMailType> CREATE_INVOICE_EMAIL_TYPES =
            Cf.set(EventMailType.ISSUE_AN_INVOICE, EventMailType.BECAME_FREE);
    private static final Duration defaultEmailTaskDelay = Duration.standardMinutes(1); // to be sure that all data
    // for task is ready

    @Autowired
    @Lazy
    private TaskPreExecutionPoliciesHolder taskPreExecutionPoliciesHolder;


    private final DynamicProperty<String> abandonedCartCooldown = new DynamicProperty<>(
            "ps-billing-promo-emails-abandoned-cart-cooldown", CustomPeriod.fromHours(24).toString(),
            Utils::validateCustomPeriod);
    private final DynamicProperty<String> abandonedCartDelay = new DynamicProperty<>(
            "ps-billing-promo-emails-abandoned-cart-delay", CustomPeriod.fromHours(1).toString(),
            Utils::validateCustomPeriod);
    private final DynamicProperty<String> pleaseComeBackDelay = new DynamicProperty<>(
            "ps-billing-promo-emails-please-come-back-delay", CustomPeriod.fromHours(3 * 24).toString(),
            Utils::validateCustomPeriod);
    private final DynamicProperty<String> b2bTariffAcquisitionCooldown = new DynamicProperty<>(
            "ps-billing-b2b-tariff-acquisition-cooldown", CustomPeriod.fromHours(3 * 24).toString(),
            Utils::validateCustomPeriod);
    private final DynamicProperty<String> b2bTariffAcquisitionDelay = new DynamicProperty<>(
            "ps-billing-b2b-tariff-acquisition-delay", CustomPeriod.fromHours(3).toString(),
            Utils::validateCustomPeriod);
    private final DynamicProperty<String> b2bAbandonedAdminPaymentCooldown = new DynamicProperty<>(
            "ps-billing-b2b-abandoned-admin-payment-cooldown", CustomPeriod.fromHours(24).toString(),
            Utils::validateCustomPeriod);
    private final DynamicProperty<String> b2bAbandonedAdminPaymentDelay = new DynamicProperty<>(
            "ps-billing-b2b-abandoned-admin-payment-delay", CustomPeriod.fromHours(2).toString(),
            Utils::validateCustomPeriod);
    private final DynamicProperty<String> b2bExhaustedBalanceLatelyCooldown = new DynamicProperty<>(
            "ps-billing-b2b-exhausted-balance-lately-cooldown", CustomPeriod.fromHours(7 * 24).toString(),
            Utils::validateCustomPeriod);
    private final DynamicProperty<String> b2bUpsaleEmailCooldown = new DynamicProperty<>(
            "ps-billing-b2b-upsale-email-cooldown", CustomPeriod.fromHours(90 * 24).toString(),
            Utils::validateCustomPeriod);

    public TaskScheduler(ExperimentsManager experimentsManager, BazingaTaskManager bazingaTaskManager,
                         FeatureFlags featureFlags) {
        this.experimentsManager = experimentsManager;
        this.bazingaTaskManager = bazingaTaskManager;
        this.featureFlags = featureFlags;
    }

    @Transactional
    public <TParams> FullJobId schedule(OnetimeTaskSupport<TParams> task) {
        return schedule(task, Instant.now());
    }

    @Transactional
    public <TParams extends BaseTaskParams> Option<FullJobId> schedule(BaseTask<TParams> task) {
        return schedule(task, Instant.now());
    }

    @Transactional
    public <TParams> FullJobId schedule(OnetimeTaskSupport<TParams> task, Instant date) {
        return scheduleCore(task, date);
    }

    @Transactional
    public <TParams extends BaseTaskParams> Option<FullJobId> schedule(BaseTask<TParams> task, Instant date) {
        if (canSchedule(task.getParametersTyped())) {
            return Option.of(scheduleCore(task, date));
        }

        return Option.empty();
    }

    @Transactional
    public void schedulePleaseComeBackTask(PassportUid uid) {
        logger.info("schedulePleaseComeBackTask for uid={}", uid);
        ActivatePromoTask.Parameters parameters = new ActivatePromoTask.Parameters(
                "please_come_back_promo_product_set_request", uid, false, true,
                PromoActivationPreExecutionPolicy.KEY, ActivatePromoTaskTaskExecutor.KEY,
                Option.empty(), Option.empty());
        schedule(new ActivatePromoTask(parameters), Instant.now().plus(toDuration(pleaseComeBackDelay)));
    }

    @Transactional
    public void schedulePleaseComeBackTask(PassportUid uid, UUID notBoughtProductId) {
        logger.info("schedulePleaseComeBackTask for uid={}, notBoughtProductId=", uid, notBoughtProductId);
        ActivatePromoTask.Parameters parameters = new ActivatePromoTask.Parameters(
                "please_come_back_promo", uid, false, true,
                PromoActivationPreExecutionPolicy.KEY, ActivatePromoTaskTaskExecutor.KEY,
                Option.of(new PaymentErrorContext(Option.of(notBoughtProductId))), Option.empty());
        schedule(new ActivatePromoTask(parameters), Instant.now().plus(toDuration(pleaseComeBackDelay)));
    }

    @Transactional
    public void scheduleB2bWelcomeEmailTask(PassportUid uid, GroupService groupService) {
        logger.info("scheduleB2bWelcomeLetterTask for uid={}. GroupService={}", uid, groupService);
        MailContext mailContext = MailContext.builder()
                .to(uid)
                .language(Option.of(Language.RUSSIAN))
                .groupServices(Cf.list(GroupServiceData.fromGroupService(groupService)))
                .build();
        BaseEmailTask.Parameters parameters = new BaseEmailTask.Parameters(
                "b2b_welcome_email", mailContext, EmailNeverSentBeforeTaskPreExecutionPolicy.KEY,
                LocalizedEmailTaskExecutor.KEY, Option.empty());
        schedule(new SendLocalizedEmailTask(parameters));
    }

    @Transactional
    public void scheduleB2bTariffAcquisitionEmailTask(PassportUid uid) {
        logger.info("scheduleB2bTariffAcquisitionEmailTask for uid={}", uid);
        MailContext mailContext = MailContext.builder()
                .to(uid)
                .build();
        B2BTariffAcquisitionEmailTask.Parameters parameters = new B2BTariffAcquisitionEmailTask.Parameters(
                Instant.now(), "b2b_tariff_acquisition", mailContext,
                OnlyFreeUsersEmailTaskPreExecutionPolicy.KEY,
                LocalizedEmailTaskExecutor.KEY, Option.of(toDuration(b2bTariffAcquisitionCooldown)));
        schedule(new B2BTariffAcquisitionEmailTask(parameters), Instant.now().plus(toDuration(b2bTariffAcquisitionDelay)));
    }

    @Transactional
    public void scheduleB2bAdminAbandonedPaymentTask(PassportUid uid, long clientId, UUID userProductId, UUID groupId) {
        logger.info("scheduleB2bAdminAbandonedPaymentTask for uid={}, clientId={}, userProduct={}, groupId={}",
                uid, clientId, userProductId, groupId);
        MailContext mailContext = MailContext.builder()
                .to(uid)
                .language(Option.of(Language.RUSSIAN))
                .userProductId(Option.of(userProductId))
                .build();
        B2BAdminAbandonedPaymentTask.Parameters parameters = new B2BAdminAbandonedPaymentTask.Parameters(clientId,
                groupId, toDuration(b2bAbandonedAdminPaymentDelay), "b2b_admin_abandoned_payment",
                mailContext, RegularEmailTaskPreExecutionPolicy.KEY, LocalizedEmailTaskExecutor.KEY,
                Option.of(toDuration(b2bAbandonedAdminPaymentCooldown)));
        schedule(new B2BAdminAbandonedPaymentTask(parameters),
                Instant.now().plus(toDuration(b2bAbandonedAdminPaymentDelay)));
    }

    @Transactional
    public void scheduleAbandonedInappCartEmailTask(PassportUid uid) {
        MailContext mailContext = MailContext.builder()
                .to(uid)
                .build();
        scheduleAbandonedCartEmailTaskImpl("abandoned_cart_inapp", mailContext);
    }

    @Transactional
    public void scheduleAbandonedCartEmailTask(PassportUid uid, Option<UUID> productId) {
        MailContext mailContext = MailContext.builder()
                .userProductId(productId)
                .to(uid)
                .build();
        scheduleAbandonedCartEmailTaskImpl("abandoned_cart", mailContext);
    }

    private void scheduleAbandonedCartEmailTaskImpl(String emailKey, MailContext mailContext) {
        BaseEmailTask.Parameters parameters = new BaseEmailTask.Parameters(
                emailKey, mailContext, OnlyFreeUsersEmailTaskPreExecutionPolicy.KEY,
                LocalizedEmailTaskExecutor.KEY, Option.of(toDuration(abandonedCartCooldown)));
        schedule(new SendLocalizedEmailTask(parameters), Instant.now().plus(toDuration(abandonedCartDelay)));
    }

    @Transactional
    public void scheduleInGraceEmailTask(Order order) {
        if (order.getUserServiceId().isEmpty()) {
            logger.error("No userService for order, skip mail");
            return;
        }
        PassportUid uid = PassportUid.cons(Long.parseLong(order.getUid()));
        String emailKey;
        switch (order.getType()) {
            case INAPP_SUBSCRIPTION:
                if (!featureFlags.getInappStateChangesEmail().isEnabledForUid(uid))
                    return;
                emailKey = "inapp_in_grace";
                break;
            case SUBSCRIPTION:
                if (!featureFlags.getLongBillingPeriodForTrustEmailEnabled().isEnabledForUid(uid))
                    return;
                emailKey = "trust_in_grace";
                break;
            case ORDER:
            default:
                logger.error("Unexpected order type {}", order.getType());
                return;
        }
        UUID userServiceId = order.getUserServiceId().get();
        UUID userProductPriceId = order.getUserProductPriceId();
        scheduleOrderChangeStateEmailTask(emailKey, uid, userServiceId, userProductPriceId);
    }

    @Transactional
    public void scheduleOnHoldEmailTask(Order order) {
        PassportUid uid = PassportUid.cons(Long.parseLong(order.getUid()));
        if (order.getUserServiceId().isEmpty()) {
            logger.error("No userService for order, skip mail");
            return;
        }
        String emailKey;
        switch (order.getType()) {
            case INAPP_SUBSCRIPTION:
                if (!featureFlags.getInappStateChangesEmail().isEnabledForUid(uid))
                    return;
                emailKey = "inapp_on_hold";
                break;
            case SUBSCRIPTION:
                if (!featureFlags.getLongBillingPeriodForTrustEmailEnabled().isEnabledForUid(uid))
                    return;
                emailKey = "trust_on_hold";
                break;
            case ORDER:
            default:
                logger.error("Unexpected order type {}", order.getType());
                return;
        }
        UUID userServiceId = order.getUserServiceId().get();
        UUID userProductPriceId = order.getUserProductPriceId();
        scheduleOrderChangeStateEmailTask(emailKey, uid, userServiceId, userProductPriceId);
    }

    @Transactional
    public void scheduleSubscriptionFinishedEmailTask(PassportUid uid, UUID userServiceId, OrderType orderType,
                                                      UUID userProductPriceId) {
        String emailKey;
        switch (orderType) {
            case SUBSCRIPTION:
                emailKey = "pochta_subscription_has_been_ended";
                break;
            case INAPP_SUBSCRIPTION:
                if (!featureFlags.getInappStateChangesEmail().isEnabledForUid(uid))
                    return;
                emailKey= "inapp_finished";
                break;
            case ORDER:
            default:
                logger.error("Unexpected order type {}", orderType);
                return;
        }

        scheduleOrderChangeStateEmailTask(emailKey, uid, userServiceId, userProductPriceId);
    }

    @Transactional
    public void scheduleHoldIsOverEmailTask(PassportUid uid, UUID userServiceId, UUID userProductPriceId) {
        scheduleOrderChangeStateEmailTask("trust_finished", uid, userServiceId, userProductPriceId);
    }

    @Transactional
    public void scheduleTransactionalEmailTask(EventMailType eventMailType, MailContext mailContext) {
        scheduleTransactionalEmailTask(eventMailType, mailContext, Instant.now().plus(defaultEmailTaskDelay));
    }

    @Transactional
    public void scheduleTransactionalEmailTask(EventMailType eventMailType, MailContext mailContext, Instant date) {
        BaseEmailTask.Parameters parameters = new BaseEmailTask.Parameters(
                eventMailType.name(), mailContext, RegularEmailTaskPreExecutionPolicy.KEY,
                TransactionalEmailExecutor.KEY);
        BaseEmailTask task;
        if (CREATE_INVOICE_EMAIL_TYPES.containsTs(eventMailType)) {
            task = new SendTransactionalEmailWithInvoiceCreationTask(parameters);
        } else {
            task = new SendLocalizedEmailTask(parameters);
        }
        schedule(task, date);
    }

    @Transactional
    public void scheduleLocalizedEmailTask(String emailTemplateKey, MailContext mailContext) {
        scheduleLocalizedEmailTask(emailTemplateKey, mailContext, Option.empty());
    }

    @Transactional
    public void schedulePromoEmailTask(String emailTemplateKey, MailContext mailContext) {
        scheduleLocalizedEmailTask(emailTemplateKey, mailContext, Option.empty());
    }

    @Transactional
    public void scheduleB2bBalanceExhaustedLatelyEmail(PassportUid uid, String emailKey) {
        MailContext mailContext = MailContext.builder()
                .to(uid)
                .build();
        scheduleLocalizedEmailTask(emailKey, mailContext, Option.of(toDuration(b2bExhaustedBalanceLatelyCooldown)));
    }

    @Transactional
    public void scheduleLocalizedEmailTask(
            String emailTemplateKey, MailContext mailContext, Option<Duration> coolDown
    ) {
        BaseEmailTask.Parameters parameters = new BaseEmailTask.Parameters(
                emailTemplateKey, mailContext,
                RegularEmailTaskPreExecutionPolicy.KEY,
                LocalizedEmailTaskExecutor.KEY,
                coolDown
        );
        schedule(new SendLocalizedEmailTask(parameters), Instant.now().plus(defaultEmailTaskDelay));
    }

    @Transactional
    public void schedulePaymentsAwareMembersEmailTask(
            Group group, String emailTemplateKey, MailContext mailContext, Option<Duration> cooldown
    ) {
        EmailPaymentsAwareMembersTask.Parameters parameters = new EmailPaymentsAwareMembersTask.Parameters(
                group.getId(), emailTemplateKey, mailContext, RegularEmailTaskPreExecutionPolicy.KEY,
                LocalizedEmailTaskExecutor.KEY, cooldown);

        schedule(new EmailPaymentsAwareMembersTask(parameters), Instant.now());
    }

    @Transactional
    public void scheduleB2bUpsaleEmailTask(PassportUid uid, GroupProduct boughtProduct, GroupProduct nextProduct) {
        logger.info("scheduleB2bUpsaleEmailTask for uid={}", uid);

        MailContext mailContext = MailContext.builder()
                .to(uid)
                .language(Option.of(Language.RUSSIAN))
                .userProductId(Option.of(nextProduct.getUserProductId()))
                .oldUserProductId(Option.of(boughtProduct.getUserProductId()))
                .build();
        BaseEmailTask.Parameters parameters = new BaseEmailTask.Parameters(
                "b2b_upsale_email", mailContext, RegularEmailTaskPreExecutionPolicy.KEY,
                LocalizedEmailTaskExecutor.KEY, Option.of(toDuration(b2bUpsaleEmailCooldown)));
        schedule(new SendLocalizedEmailTask(parameters), Instant.now());
    }

    private boolean canSchedule(BaseTaskParams taskParams) {
        return taskPreExecutionPoliciesHolder.getEmailTaskConfiguration(taskParams).canSchedule(taskParams);
    }

    private void scheduleOrderChangeStateEmailTask(String emailKey, PassportUid uid, UUID userServiceId,
                                                   UUID userProductPriceId) {
        MailContext mailContext = MailContext.builder()
                .to(uid)
                .userServiceId(Option.of(userServiceId.toString()))
                .userProductPriceId(Option.of(userProductPriceId))
                .build();
        scheduleLocalizedEmailTask(emailKey, mailContext);
    }

    private ListF<String> getUserExperiments(PassportUid uid) {
        try {
            return experimentsManager.getFlags(uid.toUidOrZero().getUid());
        } catch (Exception e) {
            logger.info("Cannot fetch data from UaaS", e);
            return Cf.list();
        }
    }

    private <TParams> FullJobId scheduleCore(OnetimeTaskSupport<TParams> task, Instant date) {
        FullJobId fullJobId = RetryUtils.retry(SCHEDULE_TASK_RETRY_COUNT, SCHEDULE_TASK_INITIAL_SLEEP_MS,
                SCHEDULE_TASK_COEFF,
                () -> bazingaTaskManager.schedule(task, date));
        logger.info("scheduled task {} with id {}", task.getClass().getName(), fullJobId);
        return fullJobId;
    }
}
