package ru.yandex.chemodan.app.psbilling.core.tasks.execution.mail;

import lombok.AllArgsConstructor;
import org.jaudiotagger.tag.KeyNotFoundException;

import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.psbilling.core.dao.mail.EmailTemplateDao;
import ru.yandex.chemodan.app.psbilling.core.dao.mail.SentEmailInfoDao;
import ru.yandex.chemodan.app.psbilling.core.entities.mail.EmailTemplateEntity;
import ru.yandex.chemodan.app.psbilling.core.mail.SenderClient;
import ru.yandex.chemodan.app.psbilling.core.mail.dataproviders.LocalizedEmailSenderDataProvider;
import ru.yandex.chemodan.app.psbilling.core.mail.dataproviders.model.SenderContext;
import ru.yandex.chemodan.app.psbilling.core.mail.tasks.BaseEmailTask;
import ru.yandex.chemodan.app.psbilling.core.tasks.execution.TaskExecutor;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;

@AllArgsConstructor
public class LocalizedEmailTaskExecutor implements TaskExecutor<BaseEmailTask.Parameters> {
    public static final String KEY = LocalizedEmailTaskExecutor.class.getName();

    private static final Logger logger = LoggerFactory.getLogger(LocalizedEmailTaskExecutor.class);

    private final SenderClient senderClient;
    private final LocalizedEmailSenderDataProvider senderDataProvider;
    private final SentEmailInfoDao sentEmailInfoDao;
    private final EmailTemplateDao emailTemplateDao;

    @Override
    public String getKey() {
        return KEY;
    }

    @Override
    public void execute(BaseEmailTask.Parameters parameters) {
        logger.info("sending localized email with parameters {}", parameters);

        checkEmailTemplateExist(parameters);
        Option<SenderContext> senderContextO = senderDataProvider.buildSenderContext(
                parameters.getEmailKey(), parameters.getContext());
        if (!senderContextO.isPresent()) {
            logger.info("Will not send the email for context={}. No sender context has been generated",
                    parameters.getContext());
            return;
        }

        SenderContext senderContext = senderContextO.get();
        logger.info("built context: {}", senderContext);

        boolean wasSent = senderClient.sendEmail(senderContext);
        if (wasSent) {
            sentEmailInfoDao.createOrUpdate(SentEmailInfoDao.InsertData.builder()
                    .emailTemplateKey(parameters.getEmailKey())
                    .uid(parameters.getContext().getDistinctRecipient())
                    .build());
        }
    }

    private void checkEmailTemplateExist(BaseEmailTask.Parameters parameters) {
        Option<EmailTemplateEntity> emailTemplateO = emailTemplateDao.findByKeyO(parameters.getEmailKey());
        if (!emailTemplateO.isPresent()) {
            throw new KeyNotFoundException(
                    "Cannot process email template " + parameters.getEmailKey() + " (Unknown template)");
        }
    }
}
