package ru.yandex.chemodan.app.psbilling.core.tasks.execution.mail;

import lombok.AllArgsConstructor;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.SetF;
import ru.yandex.chemodan.app.psbilling.core.mail.EventMailType;
import ru.yandex.chemodan.app.psbilling.core.mail.MailContext;
import ru.yandex.chemodan.app.psbilling.core.mail.SenderClient;
import ru.yandex.chemodan.app.psbilling.core.mail.dataproviders.SenderDataProvider;
import ru.yandex.chemodan.app.psbilling.core.mail.dataproviders.model.SenderContext;
import ru.yandex.chemodan.app.psbilling.core.mail.dataproviders.model.SenderTemplateDefinition;
import ru.yandex.chemodan.app.psbilling.core.mail.tasks.BaseEmailTask;
import ru.yandex.chemodan.app.psbilling.core.tasks.execution.TaskExecutor;
import ru.yandex.commune.dynproperties.DynamicProperty;
import ru.yandex.inside.utils.Language;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;

@AllArgsConstructor
public class TransactionalEmailExecutor implements TaskExecutor<BaseEmailTask.Parameters> {
    public static final String KEY = TransactionalEmailExecutor.class.getName();

    private static final Logger logger = LoggerFactory.getLogger(TransactionalEmailExecutor.class);

    private final SenderClient senderClient;
    private final ListF<SenderDataProvider> senderDataProviders;

    private final DynamicProperty<ListF<String>> availableLanguages =
            new DynamicProperty<>("ps-billing-transactional-emails-valid-languages", Cf.list(Language.RUSSIAN.value()));

    @Override
    public String getKey() {
        return KEY;
    }

    @Override
    public void execute(BaseEmailTask.Parameters parameters) {
        Option<EventMailType> eventMailTypeO = Cf.x(EventMailType.values())
                .find(eventMailType -> eventMailType.name().equals(parameters.getEmailKey()));
        if (!eventMailTypeO.isPresent()) {
            throw new EnumConstantNotPresentException(EventMailType.class, parameters.getEmailKey());
        }

        EventMailType eventMailType = eventMailTypeO.get();
        MailContext mailContext = parameters.getContext();

        ListF<SenderDataProvider> senderDataProviders = this.senderDataProviders
                .filter(contextBuilder -> contextBuilder.isAcceptableFor(eventMailType, mailContext));
        if (senderDataProviders.isEmpty()) {
            logger.info("Sender data provider for {} and context={} has not been found", eventMailType, mailContext);
            return;
        }
        senderDataProviders.forEach(senderDataProvider -> sendEmail(mailContext, senderDataProvider));
    }

    private void sendEmail(MailContext mailContext, SenderDataProvider senderDataProvider) {
        Option<Language> language = senderDataProvider.getLanguage(mailContext.getTo());
        SetF<Language> validLanguages = availableLanguages.get().map(Language.R::fromValue).unique();
        if (!language.filter(validLanguages::containsTs).isPresent()) {
            logger.info("Incompatible user language {}", language);
            return;
        }
        mailContext.setLanguage(language);
        Option<SenderContext> senderContextO = senderDataProvider.buildSenderContext(mailContext);
        if (!senderContextO.isPresent()) {
            logger.info("Will not send the email for context={}. No sender context has been generated", mailContext);
            return;
        }
        SenderContext senderContext = senderContextO.get();
        Option<SenderTemplateDefinition> templateDefinitionO =
                senderDataProvider.selectMailTemplateDefinition(mailContext);
        if (!templateDefinitionO.isPresent()) {
            logger.info("Not template has been found for {}", mailContext);
            return;
        }

        senderContext.setSenderTemplateDefinition(templateDefinitionO.get());
        senderClient.sendEmail(senderContext);
    }
}
