package ru.yandex.chemodan.app.psbilling.core.texts;

import java.util.Objects;
import java.util.UUID;
import java.util.concurrent.CountDownLatch;

import org.joda.time.Duration;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.Tuple2;
import ru.yandex.chemodan.app.psbilling.core.dao.texts.TankerKeyDao;
import ru.yandex.chemodan.app.psbilling.core.entities.CustomPeriod;
import ru.yandex.chemodan.app.psbilling.core.entities.CustomPeriodUnit;
import ru.yandex.chemodan.app.psbilling.core.entities.texts.TankerKeyEntity;
import ru.yandex.chemodan.app.psbilling.core.entities.texts.TankerTranslationEntity;
import ru.yandex.commune.util.RetryUtils;
import ru.yandex.inside.tanker.TankerClient;
import ru.yandex.inside.tanker.model.TankerKeySet;
import ru.yandex.inside.tanker.model.TankerResponse;
import ru.yandex.misc.concurrent.CountDownLatches;
import ru.yandex.misc.lang.StringUtils;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;
import ru.yandex.misc.spring.Service;
import ru.yandex.misc.worker.AlarmThread;

import static ru.yandex.chemodan.app.psbilling.core.texts.TankerTranslation.DEFAULT_LANGUAGE;

public class TextsManager implements Service {
    private static final Logger logger = LoggerFactory.getLogger(TextsManager.class);

    private final TankerClient tankerClient;
    private final TankerKeyDao tankerKeyDao;

    private MapF<UUID, TankerTranslation> translationsCache = null;
    private ListF<TankerKeyEntity> translationsKeysCache = null;

    private final UpdateTranslationsCacheThread updateThread;
    private final CountDownLatch initializedLatch = new CountDownLatch(1);

    public TextsManager(TankerClient tankerClient, TankerKeyDao tankerKeyDao) {
        this(tankerClient, tankerKeyDao, Duration.standardMinutes(1));
    }

    public TextsManager(TankerClient tankerClient, TankerKeyDao tankerKeyDao, Duration cacheUpdatePeriod) {
        this.tankerClient = tankerClient;
        this.tankerKeyDao = tankerKeyDao;
        this.updateThread = new UpdateTranslationsCacheThread(cacheUpdatePeriod);
    }

    public MapF<UUID, TankerTranslation> findTranslations(ListF<UUID> tankerKeyIds) {
        awaitInitialization();
        return tankerKeyIds.toMapMappingToValue(this::findTranslation);
    }

    public TankerTranslation findPredefinedTankerTranslation(PredefinedTankerKey predefinedTankerKey) {
        return findTranslation(predefinedTankerKey.tankerProject, predefinedTankerKey.tankerKeySet,
                predefinedTankerKey.tankerKey);
    }

    public TankerTranslation findTranslation(UUID tankerKeyId) {
        awaitInitialization();
        return translationsCache.getOrElseApply(tankerKeyId, () -> new TankerTranslation(tankerKeyId, Cf.list()));
    }

    public TankerTranslation findTranslation(String tankerProject, String tankerKeySet, String tankerKey) {
        awaitInitialization();
        TankerKeyEntity key = translationsKeysCache.filter(e ->
                Objects.equals(tankerProject, e.getProjectName()) &&
                        Objects.equals(tankerKeySet, e.getKeySetName()) &&
                        Objects.equals(tankerKey, e.getKey())
        ).singleOrThrow("Searching for " + tankerProject + "/" + tankerKeySet + "/" + tankerKey);

        return findTranslation(key.getId());
    }

    public void addTankerKeyIfNotExist(String tankerProject, String tankerKeySet, String tankerKey) {
        tankerKeyDao.insertIfNotExist(tankerProject, tankerKeySet, tankerKey);
    }

    public void updateTranslations() {
        ListF<TankerKeyEntity> allKeys = tankerKeyDao.findAllKeys();
        MapF<Tuple2<String, String>, ListF<TankerKeyEntity>> keysInKeySets = allKeys
                .groupBy(k -> new Tuple2<>(k.getProjectName(), k.getKeySetName()));

        for (Tuple2<Tuple2<String, String>, ListF<TankerKeyEntity>> entry : keysInKeySets.entries()) {
            RetryUtils.retry(logger, 3, 1000, 2, () -> {
                logger.info("retrieving translations from tanker for project {} keyset {}",
                        entry.get1().get1(), entry.get1().get2());
                TankerResponse response =
                        tankerClient.retrieveDataByKeysetId(entry.get1().get1(), entry.get1().get2(), null);
                logger.info("response from tanker: {}", response);

                Option<TankerKeySet> keysetO = response.keysets.getO(entry.get1().get2());
                if (!keysetO.isPresent()) {
                    logger.warn(
                            "Unable to find keyset {} of project {} in tanker response. our keys {} wouldn't be " +
                                    "updated",
                            entry.get1().get2(), entry.get1().get1(), entry.get2());
                    return null;
                }
                TankerKeySet keySet = keysetO.get();

                for (TankerKeyEntity ourKey : entry.get2()) {
                    Option<ru.yandex.inside.tanker.model.TankerKey> tankerKeyO = keySet.keys.getO(ourKey.getKey());
                    if (!tankerKeyO.isPresent()) {
                        logger.warn("Unable to find key {} in tanker project {}, keyset {}",
                                ourKey.getKey(), entry.get1().get1(), entry.get1().get2());
                        continue;
                    }
                    ru.yandex.inside.tanker.model.TankerKey tankerKey = tankerKeyO.get();

                    ListF<TankerTranslationEntity> tankerTranslations = tankerKey.translations
                            .mapValues(t -> t.form)
                            .filterValues(Option::isPresent)
                            .mapValues(Option::get)
                            .mapValues(f -> f.text)
                            .filterValues(StringUtils::isNotBlank)
                            .mapEntries((locale, t) -> new TankerTranslationEntity(ourKey.getId(), locale, t));
                    logger.info("merging translations: {}", tankerTranslations);
                    tankerKeyDao.mergeTranslations(tankerTranslations);
                }
                return null;
            });
        }

        reloadTranslationsCache();
    }

    private void reloadTranslationsCache() {
        translationsKeysCache = tankerKeyDao.findAllKeys();
        MapF<UUID, ListF<TankerTranslationEntity>> allTranslations = tankerKeyDao.findAllTranslations();
        translationsCache = allTranslations.mapValuesWithKey(TankerTranslation::new);
        if (!isInitialized()) {
            initialized();
        }
    }

    public boolean isInitialized() {
        return initializedLatch.getCount() == 0;
    }

    protected void awaitInitialization() {
        if (!isInitialized()) {
            logger.info("Awaiting texts cache initialization. Make sure you've called start() method.");
        }
        CountDownLatches.await(initializedLatch);
    }

    private void initialized() {
        initializedLatch.countDown();
    }

    @Override
    public void start() {
        updateThread.startGracefully();
    }

    @Override
    public void stop() {
        updateThread.stopGracefully();
    }

    private class UpdateTranslationsCacheThread extends AlarmThread {

        public UpdateTranslationsCacheThread(Duration standardDelay) {
            super(UpdateTranslationsCacheThread.class.getName(), standardDelay.getMillis());
            setDaemon(true);
        }

        @Override
        protected void alarm() {
            reloadTranslationsCache();
        }
    }

    public String localizedCustomPeriodTitle(CustomPeriod cp, Option<String> language) {
        //маски вроде "{1} лет" похоже не поддержаны. пока что просто конкатенируем если период не равен единичке
        //1 месяц -> "месяц", 2 месяца -> "2 месяц". так хотя бы понятно и не вводит в заблуждение
        //+ если транслейшон не найден, взять первое английское user-friendly значение. а не пустую строчку
        CustomPeriodUnit cpu = cp.getUnit();
        String tankerTranslation = findPredefinedTankerTranslation(cpu.getTitle())
                .findLanguageTranslation(language.orElse(DEFAULT_LANGUAGE)).orElse(cpu.value());
        int periodValue = cp.getValue();
        String result = periodValue != 1 ? ("" + periodValue + " ") : "";
        result += tankerTranslation;
        return result;
    }
}
