package ru.yandex.chemodan.app.psbilling.core.users;

import javax.annotation.PostConstruct;

import com.google.common.annotations.VisibleForTesting;
import org.springframework.transaction.annotation.Transactional;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.psbilling.core.dao.users.UserInfoDao;
import ru.yandex.chemodan.app.psbilling.core.entities.users.UserInfo;
import ru.yandex.inside.geobase.Geobase;
import ru.yandex.inside.geobase.RegionNode;
import ru.yandex.inside.geobase.RegionType;
import ru.yandex.inside.passport.PassportUid;
import ru.yandex.inside.passport.blackbox2.Blackbox2;
import ru.yandex.inside.passport.blackbox2.protocol.response.BlackboxCorrectResponse;
import ru.yandex.inside.passport.blackbox2.protocol.response.BlackboxDbFields;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;
import ru.yandex.misc.net.LocalhostUtils;

public class UserInfoService {
    private static final Logger logger = LoggerFactory.getLogger(UserInfoService.class);

    private final UserInfoDao userInfoDao;
    private final Blackbox2 blackbox2;
    private final Geobase geobase;
    private MapF<String, String> regionIdsByCodes = Cf.concurrentHashMap();

    public UserInfoService(UserInfoDao userInfoDao, Blackbox2 blackbox2, Geobase geobase) {
        this.userInfoDao = userInfoDao;
        this.blackbox2 = blackbox2;
        this.geobase = geobase;
    }

    @Transactional
    public UserInfo findOrCreateUserInfo(PassportUid uid) {
        return userInfoDao.findByUid(uid).orElseGet(() -> {
            BlackboxCorrectResponse response = blackbox2.query().userInfo(LocalhostUtils.localAddress(), uid,
                    Cf.list(BlackboxDbFields.COUNTRY), Cf.list());
            return userInfoDao.createOrUpdate(UserInfoDao.InsertData.builder()
                    .uid(uid)
                    .regionId(response.getDbFields().getO(BlackboxDbFields.COUNTRY)
                            .map(this::mapCountryCodeToRegionId).filter(Option::isPresent).map(Option::get))
                    .build());
        });
    }

    private Option<String> mapCountryCodeToRegionId(String countryShortEnName) {
        return regionIdsByCodes.getO(countryShortEnName);
    }

    @PostConstruct
    public void updateCountriesCache() {
        if (geobase == null) {
            logger.warn("Geobase not initialized, so regionIdsByCodes map isn't updated");
            return;
        }
        ListF<Integer> regionIdsByType = geobase.getRegionIdsByType(RegionType.COUNTRY);
        if (regionIdsByType == null) {
            return;
        }
        ListF<RegionNode> regions = regionIdsByType.map(geobase::getRegionById)
                .filter(Option::isPresent)
                .map(Option::get);
        regions.forEach(
                region -> regionIdsByCodes.put(region.getShortEnName().toLowerCase(), String.valueOf(region.getId())));
    }

    @VisibleForTesting
    public void addRegionCode(String regionId, String code) {
        regionIdsByCodes.put(code, regionId);
    }
}
