package ru.yandex.chemodan.app.psbilling.core.users;

import java.util.UUID;

import org.joda.time.Instant;

import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.function.Function0;
import ru.yandex.chemodan.app.psbilling.core.entities.users.Order;
import ru.yandex.chemodan.app.psbilling.core.entities.users.OrderType;
import ru.yandex.chemodan.app.psbilling.core.entities.users.UserServiceBillingStatus;
import ru.yandex.chemodan.app.psbilling.core.entities.users.UserServiceEntity;
import ru.yandex.chemodan.app.psbilling.core.products.UserProduct;
import ru.yandex.chemodan.app.psbilling.core.products.UserProductManager;
import ru.yandex.chemodan.app.psbilling.core.products.UserProductPrice;
import ru.yandex.chemodan.app.psbilling.core.synchronization.engine.SynchronizationStatus;
import ru.yandex.chemodan.app.psbilling.core.synchronization.engine.Target;
import ru.yandex.misc.lang.DefaultObject;

public class UserService extends DefaultObject {
    private final UserServiceEntity entity;
    private final Function0<UserProduct> userProductProvider;
    private final Function0<Option<UserProductPrice>> userProductPriceProvider;
    private final UserServiceManager userServiceManager;

    public UserService(UserServiceEntity entity, UserProduct userProduct, Option<UserProductPrice> price,
                       UserServiceManager userServiceManager) {
        this(entity, () -> userProduct, () -> price, userServiceManager);
    }

    public UserService(UserServiceEntity entity, UserProductManager userProductManager,
                       UserServiceManager userServiceManager) {
        this(entity, () -> userProductManager.findById(entity.getUserProductId()),
                () -> entity.getProductPriceId().map(userProductManager::findPrice),
                userServiceManager);
    }

    private UserService(UserServiceEntity entity, Function0<UserProduct> userProductProvider,
                        Function0<Option<UserProductPrice>> userProductPriceProvider,
                        UserServiceManager userServiceManager) {
        this.entity = entity;
        this.userProductProvider = userProductProvider.memoize();
        this.userProductPriceProvider = userProductPriceProvider.memoize();
        this.userServiceManager = userServiceManager;
    }

    public Option<Instant> getNextCheckDate() {
        return entity.getNextCheckDate();
    }

    public Option<Instant> getDueDate() {
        return entity.getDueDate();
    }

    public Option<Instant> getSubscriptionUntil() {
        return entity.getDueDate().orElse(entity.getNextCheckDate());
    }

    public Option<Instant> getActualDisabledAt() {
        return entity.getActualDisabledAt();
    }

    public UUID getId() {
        return entity.getId();
    }

    public Option<UserServiceBillingStatus> getBillingStatus() {
        return entity.getBillingStatus();
    }

    public Target getTarget() {
        return entity.getTarget();
    }

    public boolean getAutoProlongEnabled() {
        return entity.getAutoProlongEnabled().isPresent() && entity.getAutoProlongEnabled().get();
    }

    public Option<Boolean> getAutoProlongEnabledO() {
        return entity.getAutoProlongEnabled();
    }

    public UUID getUserProductId() {
        return entity.getUserProductId();
    }

    public Option<UUID> getProductPriceId() {
        return entity.getProductPriceId();
    }

    public Instant getCreatedAt() {
        return entity.getCreatedAt();
    }

    public Option<UUID> getLastPaymentOrderId() {
        return entity.getLastPaymentOrderId();
    }

    public SynchronizationStatus getStatus() {
        return entity.getStatus();
    }

    public String getUid() {
        return entity.getUid();
    }

    public UserProduct getUserProduct() {
        return userProductProvider.apply();
    }

    public Option<UserProductPrice> getPrice() {
        return userProductPriceProvider.apply();
    }

    public Option<String> getPackageName() {
        return entity.getPackageName();
    }

    public Option<OrderType> getLastPaymentOrderType() {
        return userServiceManager.findLastPaymentOrder(this).map(Order::getType);
    }
}
