package ru.yandex.chemodan.app.psbilling.core.users;

import java.math.BigDecimal;
import java.math.RoundingMode;
import java.util.Objects;
import java.util.UUID;

import lombok.AllArgsConstructor;
import org.joda.time.Duration;
import org.joda.time.Instant;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.SetF;
import ru.yandex.chemodan.app.psbilling.core.billing.users.BillingActionsReportingService;
import ru.yandex.chemodan.app.psbilling.core.dao.users.OrderDao;
import ru.yandex.chemodan.app.psbilling.core.dao.users.UserServiceDao;
import ru.yandex.chemodan.app.psbilling.core.entities.products.BillingType;
import ru.yandex.chemodan.app.psbilling.core.entities.users.Order;
import ru.yandex.chemodan.app.psbilling.core.entities.users.OrderStatus;
import ru.yandex.chemodan.app.psbilling.core.entities.users.OrderType;
import ru.yandex.chemodan.app.psbilling.core.entities.users.UserServiceBillingStatus;
import ru.yandex.chemodan.app.psbilling.core.entities.users.UserServiceEntity;
import ru.yandex.chemodan.app.psbilling.core.groups.TrialService;
import ru.yandex.chemodan.app.psbilling.core.mail.EventMailType;
import ru.yandex.chemodan.app.psbilling.core.mail.MailContext;
import ru.yandex.chemodan.app.psbilling.core.products.UserProduct;
import ru.yandex.chemodan.app.psbilling.core.products.UserProductManager;
import ru.yandex.chemodan.app.psbilling.core.products.UserProductPrice;
import ru.yandex.chemodan.app.psbilling.core.synchronization.engine.Target;
import ru.yandex.chemodan.app.psbilling.core.synchronization.userservice.UserServiceActualizationService;
import ru.yandex.chemodan.app.psbilling.core.tasks.execution.TaskScheduler;
import ru.yandex.chemodan.app.psbilling.core.util.LockService;
import ru.yandex.chemodan.trust.client.TrustClient;
import ru.yandex.chemodan.trust.client.requests.OrderRequest;
import ru.yandex.chemodan.trust.client.requests.SupplementRequest;
import ru.yandex.chemodan.trust.client.requests.UpdateSubscriptionRequest;
import ru.yandex.chemodan.trust.client.responses.SubscriptionResponse;
import ru.yandex.chemodan.util.exception.AccessForbiddenException;
import ru.yandex.commune.dynproperties.DynamicProperty;
import ru.yandex.inside.passport.PassportUid;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;

@AllArgsConstructor
public class UserServiceManager {
    private static final Logger logger = LoggerFactory.getLogger(UserServiceManager.class);

    private final UserServiceDao userServiceDao;
    private final UserServiceActualizationService userServiceActualizationService;
    private final TrustClient trustClient;
    private final UserProductManager userProductManager;
    private final OrderDao orderDao;
    private final LockService lockService;
    private final TrialService trialService;
    private final TaskScheduler taskScheduler;
    private final DynamicProperty<Integer> delayOfServiceActivation =
            new DynamicProperty<>("ps-billing.service-activation-delay-seconds", 10);
    private final BillingActionsReportingService billingActionsReportingService;

    public UserService findById(UUID userServiceId) {
        return new UserService(userServiceDao.findById(userServiceId), userProductManager, this);
    }

    public ListF<UserService> findEnabledConflicting(String uid, UUID activeProductId) {
        return findAllEnabledConflicting(uid, Cf.list(activeProductId));
    }

    public ListF<UserService> findAllEnabledConflicting(String uid, ListF<UUID> activeProductIds) {
        ListF<UserService> activeServices = findEnabled(uid, Option.empty());
        return activeServices.filter(s -> s.getUserProduct().conflictsWithAny(activeProductIds));
    }

    public ListF<UserService> findEnabled(String uid, Option<UUID> productOwnerId) {
        return find(uid, productOwnerId, Option.of(Target.ENABLED));
    }

    public ListF<UserService> find(String uid, Option<UUID> productOwnerId) {
        return find(uid, productOwnerId, Option.empty());
    }

    public ListF<UserService> find(String uid, Option<UUID> productOwnerId, Option<Target> target) {
        return find(uid, productOwnerId, target, Option.empty());
    }

    public ListF<UserService> find(String uid, Option<UUID> productOwnerId, Option<Target> target, Option<OrderStatus> orderStatus) {
        ListF<UserServiceEntity> entities = userServiceDao.find(uid, productOwnerId, target, orderStatus);
        MapF<UUID, UserProduct> products = userProductManager
                .findByIds(entities.map(UserServiceEntity::getUserProductId).stableUnique())
                .toMapMappingToKey(UserProduct::getId);
        MapF<UUID, UserProductPrice> prices = userProductManager
                .findPrices(entities.filterMap(UserServiceEntity::getProductPriceId).stableUnique())
                .toMapMappingToKey(UserProductPrice::getId);
        return entities.map(e -> new UserService(e, products.getOrThrow(e.getUserProductId()),
                e.getProductPriceId().map(prices::getOrThrow), this));
    }


    public UserService createUserService(Order order, Instant nextCheckDate, UserServiceBillingStatus status) {
        UserProductPrice price = userProductManager.findPrice(order.getUserProductPriceId());
        UserProduct userProduct = price.getPeriod().getUserProduct();
        userProduct.getTrialDefinition().ifPresent(
                trialDefinition -> trialService.findOrCreateTrialUsage(trialDefinition,
                        PassportUid.cons(Long.parseLong(order.getUid())))
        );

        UserServiceEntity userService = userServiceDao.insert(UserServiceDao.InsertData.builder()
                .uid(order.getUid())
                .nextCheckDate(Option.of(nextCheckDate))
                .dueDate(Option.of(nextCheckDate))
                .userProductId(userProduct.getId())
                .userProductPriceId(Option.of(price.getId()))
                .autoProlongEnabled(Option.of(order.getType() != OrderType.ORDER))
                .billingStatus(Option.of(status))
                .paidByOrderId(Option.of(order.getId()))
                .packageName(order.getPackageName())
                .build(), Target.ENABLED);
        logger.info("created new user service id={}", userService.getId());

        userServiceActualizationService.scheduleUserServiceActualization(userService.getId(),
                Duration.standardSeconds(delayOfServiceActivation.get()));
        taskScheduler.scheduleTransactionalEmailTask(
                EventMailType.JUST_PAID,
                MailContext.builder()
                        .to(PassportUid.cons(Long.parseLong(order.getUid())))
                        .userServiceId(Option.of(userService.getId().toString()))
                        .groupIds(Cf.list())
                        .groupServices(Cf.list())
                        .language(Option.empty())
                        .build(),
                Instant.now().plus(Duration.standardSeconds(delayOfServiceActivation.get())));
        return new UserService(userService, userProductManager, this);
    }

    public void stopAutoProlong(PassportUid uid, UUID userServiceId, Option<String> localizedEmailKey) {
        stopAutoProlong(uid, findById(userServiceId), localizedEmailKey, false);
    }

    public void updateNextCheckDate(UUID serviceId, Instant subscriptionUntil, UserServiceBillingStatus billingStatus) {
        updateSubscriptionStatus(serviceId, billingStatus, subscriptionUntil, subscriptionUntil);
    }

    public void updateSubscriptionStatus(
            UUID serviceId, UserServiceBillingStatus billingStatus, Instant dueDate, Instant nextCheckDate
    ) {
        userServiceDao.updateSubscriptionStatus(serviceId, billingStatus, dueDate, nextCheckDate);
    }

    public void stopAutoProlong(UUID userServiceId, Instant subscriptionUntil) {
        userServiceDao.stopAutoProlong(userServiceId, subscriptionUntil);
    }

    public void stopAutoProlong(PassportUid uid, UserService service, Option<String> localizedEmailKeyToSchedule,
                                boolean force) {
        UserProduct product = service.getUserProduct();
        if (!service.getAutoProlongEnabled() && !force) {
            logger.info("autoprolong already stopped for service {}", service.getId());
            return;
        }

        boolean trustSubscriptionHasBeenStopped = false;

        if (product.getBillingType() == BillingType.TRUST) {
            stopTrustSubscription(uid, service, product);
            trustSubscriptionHasBeenStopped = true;
        } else if (product.getBillingType().isInappProduct() || product.getBillingType() == BillingType.GROUP) {
            throw new IllegalStateException("unable to stop prolong from ui");
        }

        logger.info("Autoprolong stopped for service: {}", service.getId());

        lockService.doWithUserLockedInTransaction(uid.toString(), () -> {
            UserService serviceReloaded = findById(service.getId());
            if (serviceReloaded.getBillingStatus().isPresent() && serviceReloaded.getTarget() != Target.DISABLED
                    && (serviceReloaded.getBillingStatus().get() == UserServiceBillingStatus.WAIT_AUTO_PROLONG ||
                    serviceReloaded.getBillingStatus().get() == UserServiceBillingStatus.FREE_PERIOD)) {
                disableService(service.getId());
            }
        });
        if (trustSubscriptionHasBeenStopped) {
            localizedEmailKeyToSchedule
                    .ifPresent(localizedEmailKey ->
                            taskScheduler.scheduleLocalizedEmailTask(
                                    localizedEmailKey,
                                    MailContext.builder()
                                            .to(uid)
                                            .userServiceId(Option.of(service.getId().toString()))
                                            .groupIds(Cf.list())
                                            .groupServices(Cf.list())
                                            .language(Option.empty())
                                            .build()
                            )
                    );
        }
    }

    private void stopTrustSubscription(PassportUid uid, UserService service, UserProduct product) {
        ListF<Order> orders = orderDao.findByServiceId(service.getId())
                .filter(o -> Cf.set(OrderStatus.PAID, OrderStatus.UPGRADED, OrderStatus.ON_HOLD).containsTs(o.getStatus()));

        // может быть много при ручном продлении, а в этом методе речь про остановку автоматического продления
        if (orders.size() > 1) {
            logger.error("unable to cancel autoprolong for non-subscription orders {}", orders);
            throw new IllegalStateException("unable to cancel autoprolong for non-subscription orders");
        } else if (orders.isEmpty()) {
            throw new IllegalStateException("order of the service not found");
        }
        Order order = orders.first();
        if (!Objects.equals(uid.toString(), order.getUid())) {
            throw new AccessForbiddenException("Not authorized");
        }

        stopTrustSubscription(order);
        userServiceDao.stopAutoProlong(service.getId());

        UserProductPrice price = userProductManager.findPrice(order.getUserProductPriceId());
        billingActionsReportingService.builder(BillingActionsReportingService.Action.UNSUBSCRIBE)
                .status("success")
                .order(order)
                .userProductPrice(price)
                .userProduct(product)
                .finish();
    }

    private static UpdateSubscriptionRequest buildStopProlongRequest(Order order) {
        UpdateSubscriptionRequest.UpdateSubscriptionRequestBuilder builder = UpdateSubscriptionRequest.builder()
                .uid(PassportUid.cons(Long.parseLong(order.getUid())))
                .orderId(order.getTrustOrderId())
                .finishTime(Instant.now())
                .trustServiceId(order.getTrustServiceId());

//        if (service.getBillingStatus().map(UserServiceBillingStatus.FREE_PERIOD::equals).getOrElse(false) &&
//                product.getTrialDefinitionId().isPresent())
//        {
//            builder.trialStopTime(Instant.now());
//        }
        //возникают ошибки error:invalid_trial_until_dt
        //  https://st.yandex-team.ru/TRUST-7484

        return builder.build();
    }

    public void stopTrustSubscription(Order order) {
        trustClient.updateSubscription(buildStopProlongRequest(order));
    }

    public void disableService(UUID userServiceId) {
        userServiceDao.setTargetState(Cf.list(userServiceId), Target.DISABLED);
        userServiceActualizationService.scheduleUserServiceActualization(userServiceId);
    }

    public Option<UserService> findServiceForUpgrade(
            String uid, BillingType typeOfProducts, Option<String> packageName, SetF<UUID> conflictingProducts) {
        ListF<UserService> boughtServices = findEnabled(uid, Option.empty())
                .filter(s -> s.getUserProduct().getBillingType() == typeOfProducts)
                .filter(s -> s.getUserProduct().conflictsWithAny(conflictingProducts))
                .filter(s -> s.getPackageName().equals(packageName));

        if (boughtServices.size() > 1) {
            logger.warn("Found more than 1 bought services, for upgrade will be used the first one");
        }
        //апгрейды только если есть цена, на которую можно ориентироваться
        return boughtServices.filter(s -> s.getPrice().isPresent())
                .sortedBy(s -> s.getPrice().get().getPrice())
                .firstO();
    }

    public void upgradeService(UserService userServiceUpgradeTo, Order orderUpgradeTo,
                               UserService userServiceUpgradeFrom, Order orderUpgradeFrom, boolean useTrust) {
        logger.info("start of service {} upgrade", userServiceUpgradeFrom.getId());
        disableService(userServiceUpgradeFrom.getId());
        orderDao.upgradeOrder(orderUpgradeFrom.getId(), orderUpgradeTo.getId());

        if (useTrust) {
            stopAutoProlong(PassportUid.cons(Long.parseLong(orderUpgradeFrom.getUid())),
                    userServiceUpgradeFrom, Option.empty(), true);
        } else {
            userServiceDao.stopAutoProlong(userServiceUpgradeFrom.getId());
        }

        addSupplementPeriod(userServiceUpgradeTo, orderUpgradeTo, userServiceUpgradeFrom, useTrust);
    }

    public Instant addSupplementPeriod(UserService userServiceUpgradeTo, Order orderUpgradeTo,
                                       UserService userServiceUpgradeFrom, boolean useTrust) {
        Duration supplementPeriod = calculateSupplementPeriod(userServiceUpgradeTo, userServiceUpgradeFrom);
        Instant newCheckDate = userServiceUpgradeTo.getNextCheckDate().get();

        if (!supplementPeriod.isLongerThan(Duration.ZERO)) {
            logger.info("duration to add 0 - skip adding time of usage");
            return newCheckDate;
        }
        if (useTrust) {
            SubscriptionResponse subscription = trustClient.getSubscription(OrderRequest.builder()
                    .uid(PassportUid.cons(Long.parseLong(orderUpgradeTo.getUid())))
                    .trustServiceId(orderUpgradeTo.getTrustServiceId())
                    .orderId(orderUpgradeTo.getTrustOrderId())
                    .build());

            newCheckDate = subscription.getSubscriptionUntil().plus(supplementPeriod);
            updateNextCheckDate(userServiceUpgradeTo.getId(), newCheckDate, UserServiceBillingStatus.PAID);

            trustClient.supplementSubscription(SupplementRequest.builder()
                    .trustServiceId(orderUpgradeTo.getTrustServiceId())
                    .uid(PassportUid.cons(Long.parseLong(orderUpgradeTo.getUid())))
                    .orderId(orderUpgradeTo.getTrustOrderId())
                    .expectedSubsUntilTs(subscription.getSubscriptionUntil())
                    .supplementUntilTs(newCheckDate)
                    .build());

        } else {
            newCheckDate = userServiceUpgradeTo.getNextCheckDate().get().plus(supplementPeriod);
            updateNextCheckDate(userServiceUpgradeTo.getId(), newCheckDate, UserServiceBillingStatus.PAID);
        }

        return newCheckDate;
    }

    //    oldPrice / oldDuration * unusedTime * newDuration / newPrice
    private Duration calculateSupplementPeriod(UserService userServiceUpgradeTo, UserService userServiceUpgradeFrom) {
        if (!userServiceUpgradeFrom.getNextCheckDate().isPresent()) {
            logger.warn("Service for upgrading hasn't nextCheckDate");
            return Duration.ZERO;
        }

        if (userServiceUpgradeFrom.getBillingStatus().isPresent()) {
            UserServiceBillingStatus billingStatus = userServiceUpgradeFrom.getBillingStatus().get();

            if (billingStatus == UserServiceBillingStatus.WAIT_AUTO_PROLONG ||
                    billingStatus == UserServiceBillingStatus.FREE_PERIOD) {
                return Duration.ZERO;
            }
        }

        long unusedPeriodMs = userServiceUpgradeFrom.getNextCheckDate().get().getMillis() -
                userServiceUpgradeTo.getCreatedAt().getMillis();

        if (unusedPeriodMs <= 0) {
            logger.warn("Service {} subscription has expired", userServiceUpgradeFrom.getId());
            return Duration.ZERO;
        }

        if (!userServiceUpgradeFrom.getPrice().isPresent() || !userServiceUpgradeTo.getPrice().isPresent()) {
            throw new IllegalArgumentException("One of services doesn't have price: userServiceUpgradeFromId=" +
                    userServiceUpgradeFrom.getId() + " userServiceUpgradeToId=" + userServiceUpgradeTo.getId());
        }
        if (!userServiceUpgradeFrom.getNextCheckDate().isPresent() || !userServiceUpgradeTo.getNextCheckDate().isPresent()) {
            throw new IllegalArgumentException("One of services doesn't have nextCheckDate: userServiceUpgradeFromId=" +
                    userServiceUpgradeFrom.getId() + " userServiceUpgradeToId=" + userServiceUpgradeTo.getId());
        }
        UserProductPrice priceUpgradeFrom = userServiceUpgradeFrom.getPrice().get();
        UserProductPrice priceUpgradeTo = userServiceUpgradeTo.getPrice().get();

        long supplementMillis = (long) priceUpgradeFrom.getPrice()
                .multiply(BigDecimal.valueOf(unusedPeriodMs)
                        .multiply(BigDecimal.valueOf(priceUpgradeTo.getPeriodMsTo(userServiceUpgradeTo.getNextCheckDate().get()))))
                .divide(priceUpgradeTo.getPrice()
                        .multiply(BigDecimal.valueOf(priceUpgradeFrom.getPeriodMsTo(userServiceUpgradeFrom.getNextCheckDate().get()))), 0, RoundingMode.HALF_DOWN)
                .doubleValue();
        return Duration.millis(supplementMillis);
    }

    public Option<Order> findLastPaymentOrder(UserService userService) {
        return userService.getLastPaymentOrderId()
                .map(orderDao::findById)
                .orElse(Option.empty());
    }

}
