package ru.yandex.chemodan.app.psbilling.core.util;

import java.util.function.BiFunction;
import java.util.function.Consumer;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.function.Function;
import ru.yandex.commune.util.RetryUtils;
import ru.yandex.misc.log.mlf.Logger;

public class BatchFetchingUtils {

    public static <T, U> ListF<T> collectBatchedEntities(
            BiFunction<Integer, Option<U>, ListF<T>> batchFetcher,
            Function<T, U> itemToKeyMapper,
            Integer batchSize,
            Logger logger)
    {
        ListF<T> accumulator = Cf.arrayList();
        fetchAndProcessBatchedEntities(batchFetcher, itemToKeyMapper, accumulator::add, batchSize, logger);
        return accumulator.unmodifiable();
    }

    public static <T, U> void fetchAndProcessBatchedEntities(
            BiFunction<Integer, Option<U>, ListF<T>> batchFetcher,
            Function<T, U> itemToKeyMapper,
            Consumer<T> process,
            Integer batchSize,
            Logger logger)
    {
        fetchAndProcessBatchedEntities(batchFetcher, itemToKeyMapper, process, batchSize, logger, true);
    }

    public static <T, U> void fetchAndProcessBatchedEntities(
            BiFunction<Integer, Option<U>, ListF<T>> batchFetcher,
            Function<T, U> itemToKeyMapper,
            Consumer<T> process,
            Integer batchSize,
            Logger logger,
            boolean isRetry)
    {
        Option<U> itemToStart = Option.empty();
        int selectedCnt = batchSize;
        while (selectedCnt == batchSize) {
            Option<U> itemToStartFinal = itemToStart;
            ListF<T> items;
            if (isRetry) {
                items = RetryUtils.retry(logger, 3, 500, 2,
                        () -> batchFetcher.apply(batchSize, itemToStartFinal)
                );
            } else {
                items = batchFetcher.apply(batchSize, itemToStartFinal);
            }
            items.forEach(process);
            logger.info("Another batch of {} rows fetched and processed", items.size());
            itemToStart = items.lastO().map(itemToKeyMapper);
            selectedCnt = items.size();
        }
    }

    public static <T, U> ListF<T> collectBatchedEntities(
            Function<ListF<U>, ListF<T>> batchFetcher,
            ListF<U> items,
            Integer batchSize
    ) {
        ListF<T> accumulator = Cf.arrayList();

        for (int i = 0; i < items.size(); i+=batchSize) {
            accumulator.addAll(
                    batchFetcher.apply(items.subList(i, Math.min(items.size(), i + batchSize) - 1))
            );
        }

        return accumulator;
    }
}
