package ru.yandex.chemodan.app.psbilling.core.util;

import java.util.UUID;
import java.util.function.Consumer;
import java.util.function.Function;

import lombok.AllArgsConstructor;
import org.springframework.transaction.support.TransactionTemplate;

import ru.yandex.bolts.function.Function0;
import ru.yandex.bolts.function.Function0V;
import ru.yandex.chemodan.app.psbilling.core.dao.groups.GroupDao;
import ru.yandex.chemodan.app.psbilling.core.dao.users.OrderDao;
import ru.yandex.chemodan.app.psbilling.core.dao.users.UserInfoDao;
import ru.yandex.chemodan.app.psbilling.core.entities.groups.Group;
import ru.yandex.chemodan.app.psbilling.core.entities.users.Order;
import ru.yandex.chemodan.app.psbilling.core.users.UserInfoService;
import ru.yandex.inside.passport.PassportUid;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;

@AllArgsConstructor
public class LockService {
    private static final Logger logger = LoggerFactory.getLogger(LockService.class);

    private final UserInfoDao userInfoDao;
    private final GroupDao groupDao;
    private final OrderDao orderDao;
    private final UserInfoService userInfoService;
    private final TransactionTemplate transactionTemplate;

    public <T> T doWithLockedInTransaction(UUID groupId, PassportUid uid, Function<Group, T> callback) {
        ensureUserExist(uid);
        return transactionTemplate.execute(status -> {
            Group group = lockGroup(groupId);
            lockUser(uid);
            return callback.apply(group);
        });
    }

    public void doWithUserLockedInTransaction(String uid, Function0V callback) {
        doWithUserLockedInTransaction(uid, () -> {
            callback.apply();
            return null;
        });
    }

    public <T> T doWithUserLockedInTransaction(String uid, Function0<T> callback) {
        PassportUid pUid = PassportUid.cons(Long.parseLong(uid));
        ensureUserExist(pUid);
        T result = transactionTemplate.execute(status -> {
            lockUser(pUid);
            logger.info("lock uid={} acquired", uid);
            return callback.apply();
        });
        logger.info("lock uid={} released", uid);
        return result;
    }

    public void doWithOrderLockedInTransaction(UUID orderId, Consumer<Order> callback) {
        doWithOrderLockedInTransactionT(orderId, order -> {
            callback.accept(order);
            return null;
        });
    }

    public <T> T doWithOrderLockedInTransactionT(UUID orderId, Function<Order, T> callback) {
        T result = transactionTemplate.execute(status -> {
            Order order = orderDao.lock(orderId);
            logger.info("lock order={} acquired", order);
            return callback.apply(order);
        });
        logger.info("lock order={} released", orderId);
        return result;
    }


    private void lockUser(PassportUid uid) {
        if (!userInfoDao.lockUser(uid).isPresent()) {
            throw new IllegalStateException("unable to lock user " + uid);
        }
    }

    private Group lockGroup(UUID groupId) {
        return groupDao.lockGroup(groupId);
    }

    private void ensureUserExist(PassportUid uid) {
        //убеждаемся что пользователь есть
        userInfoService.findOrCreateUserInfo(uid);
    }
}
