package ru.yandex.chemodan.directory.client;

import java.util.Optional;

import lombok.RequiredArgsConstructor;
import lombok.SneakyThrows;
import org.springframework.http.HttpEntity;
import org.springframework.http.HttpHeaders;
import org.springframework.http.HttpMethod;
import org.springframework.http.HttpStatus;
import org.springframework.http.ResponseEntity;
import org.springframework.web.client.HttpClientErrorException;
import org.springframework.web.client.RestTemplate;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.SetF;
import ru.yandex.commune.util.RetryUtils;
import ru.yandex.inside.passport.PassportUid;
import ru.yandex.misc.io.http.UriBuilder;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;

@RequiredArgsConstructor
public class DirectoryClient {
    private static final Logger logger = LoggerFactory.getLogger(DirectoryClient.class);

    public static final String UT_USER = "user";
    public static final String UT_ROBOT = "robot";

    private final String host;
    private final RestTemplate restTemplate;

    RestTemplate getRestTemplate() {
        return restTemplate;
    }

    /**
     * Возвращает уникальных пользователей из организации {@code organizationId}
     *
     * @param organizationId Идентификатор организации
     * @param excludeRobots  Признак исключения роботов (true  - всех НЕ роботов,
     *                       false - пользователи + роботы)
     * @param perPage        Количесво объектов выведено на странице
     * @param limit          Размер списка
     * @return Список уникальных пользователей
     * @throws FetchLimitExceededException если количество данных больше чем значение {@code limit}
     **/
    @SneakyThrows
    public SetF<Long> usersInOrganization(
            final String organizationId,
            final boolean excludeRobots,
            final int perPage,
            final Option<Integer> limit) {
        final SetF<Long> result = Cf.hashSet();

        final UriBuilder templateUri = uriBuilder()
                .appendPath("/v11/users/")
                .addParam("per_page", perPage)
                .addParam("page", "1");

        Optional<String> nextUri = Optional.of(
                UriBuilder.cons(templateUri.toUrl())
                        .addParam("user_type", UT_USER)
                        .toUrl()
        );

        boolean needFetchRobot = !excludeRobots;

        while (nextUri.isPresent()) {
            final HttpHeaders headers = organizationHttpHeaders(organizationId);
            final String request = nextUri.get();
            final DirectoryUsersResponse response = makeRequest(organizationId, headers, request);
            logger.info("Directory request: {} for orgId {}, response: {}", request, organizationId, response);

            response.getResult().stream()
                    .map(DirectoryUsersResponse.User::getId)
                    .forEach(result::add);
            nextUri = response.getLinks().map(DirectoryUsersResponse.Links::getNext);

            if (!nextUri.isPresent() && needFetchRobot) {
                nextUri = Optional.of(
                        UriBuilder.cons(templateUri.toUrl())
                                .addParam("user_type", UT_ROBOT)
                                .toUrl()
                );
                needFetchRobot = false;
            }

            if (limit.isPresent() && result.size() > limit.get()) {
                throw new FetchLimitExceededException(organizationId, limit.get());
            }
        }

        return result;
    }

    private DirectoryUsersResponse makeRequest(String organizationId, HttpHeaders headers, String request) {
        try {
            return RetryUtils.retryOrThrow(logger, 2, 1000, 2,
                    () -> restTemplate.exchange(
                                    request, HttpMethod.GET, new HttpEntity<Void>(headers),
                                    DirectoryUsersResponse.class)
                            .getBody(),
                    this::isOrganizationNotFoundException);
        } catch (Exception e) {
            if (hasOrganizationNotFoundInCause(e)) {
                throw new OrganizationNotFoundException(organizationId);
            }
            throw e;
        }
    }

    private boolean hasOrganizationNotFoundInCause(Exception e) {
        Throwable throwable = e;
        do {
            if (isOrganizationNotFoundException(throwable)) {
                return true;
            }
            throwable = throwable.getCause();

        } while (throwable != null);

        return false;
    }

    private boolean isOrganizationNotFoundException(Throwable e) {
        return (e instanceof HttpClientErrorException) &&
                ((HttpClientErrorException) e).getStatusCode() == HttpStatus.NOT_FOUND;
    }


    public ListF<String> organizationsWhereUserIsAdmin(String uid, int perPage) {
        ListF<String> result = Cf.arrayList();

        Optional<String> nextUri = Optional.of(
                uriBuilder()
                        .appendPath("/v11/organizations/")
                        .addParam("per_page", perPage)
                        .addParam("page", "1")
                        .addParam("admin_only", "true")
                        .addParam("show", "user")
                        .build()
                        .toString()
        );

        while (nextUri.isPresent()) {
            HttpHeaders headers = userHttpHeaders(uid);
            String request = nextUri.get();

            DirectoryOrganizationsResponse response = RetryUtils.retryOrThrow(logger, 3, 500, 2,
                    () -> restTemplate.exchange(request, HttpMethod.GET, new HttpEntity<Void>(headers),
                            DirectoryOrganizationsResponse.class).getBody());

            logger.info("Directory request: {} for uid {}, response: {}", request, uid, response);

            response.getResult().stream()
                    .map(DirectoryOrganizationsResponse.Organization::getId)
                    .forEach(result::add);

            nextUri = response.getNext();
        }

        return result;
    }

    private UriBuilder uriBuilder() {
        return UriBuilder.cons("https://" + host);
    }

    private HttpHeaders organizationHttpHeaders(String organizationId) {
        HttpHeaders headers = new HttpHeaders();
        headers.add("X-ORG-ID", organizationId);
        return headers;
    }

    private HttpHeaders userHttpHeaders(String uid) {
        HttpHeaders headers = new HttpHeaders();
        headers.add("X-UID", uid);
        headers.add("x-user-ip", "127.0.0.1");
        return headers;
    }

    public DirectoryOrganizationFeaturesResponse getOrganizationFeatures(final String organizationId) {
        final String uri = uriBuilder()
                .appendPath("/v11/organizations/")
                .appendPath(organizationId)
                .appendPath("features/")
                .build()
                .toString();

        try {
            return RetryUtils.retryOrThrow(
                    logger,
                    3,
                    500,
                    2,
                    () -> restTemplate.getForObject(uri, DirectoryOrganizationFeaturesResponse.class),
                    this::isOrganizationNotFoundException
            );
        } catch (final Exception e) {
            if (hasOrganizationNotFoundInCause(e)) {
                throw new OrganizationNotFoundException(organizationId);
            }
            throw e;
        }
    }

    public DirectoryOrganizationByIdResponse getOrganizationById(String organizationId) {
        String uri = uriBuilder()
                .appendPath("/v11/organizations/")
                .appendPath(organizationId)
                .appendPath("/")
                .addParam("fields", "organization_type")
                .build()
                .toString();
        return RetryUtils.retryOrThrow(logger, 3, 500, 2,
                () -> restTemplate.getForObject(uri, DirectoryOrganizationByIdResponse.class));
    }


    public Option<DirectoryUsersInfoResponse> getUserInfo(PassportUid uid, String organizationId) {
        final String request = uriBuilder()
                .appendPath("/v11/users/")
                .appendPath(uid.toString())
                .appendPath("/")
                .addParam("fields", "is_admin")
                .build()
                .toString();

        final HttpHeaders headers = organizationHttpHeaders(organizationId);

        try {
            final ResponseEntity<DirectoryUsersInfoResponse> response = RetryUtils.retryOrThrow(
                    logger,
                    3,
                    500,
                    2,
                    () -> restTemplate.exchange(
                            request,
                            HttpMethod.GET,
                            new HttpEntity<Void>(headers),
                            DirectoryUsersInfoResponse.class
                    ),
                    this::isOrganizationNotFoundException
            );
            logger.info("Directory request: {} for orgId {}, response: {}", request, organizationId, response);
            return Option.ofNullable(response.getBody());
        } catch (Exception e) {
            if (hasOrganizationNotFoundInCause(e)) {
                return Option.empty();
            }
            throw e;
        }
    }

    public Option<String> getOrganizationPartnerId(String organizationId) {
        String uri = uriBuilder()
                .appendPath("/v11/organizations/")
                .appendPath(organizationId)
                .appendPath("/")
                .addParam("fields", "partner_id")
                .build()
                .toString();
        DirectoryOrganizationPartnerIdResponse response = RetryUtils.retryOrThrow(
                logger, 3, 500, 2,
                () -> restTemplate.getForObject(uri, DirectoryOrganizationPartnerIdResponse.class),
                this::isOrganizationNotFoundException);
        return response.getPartnerId();
    }
}
