package ru.yandex.chemodan.trust.client;

import java.io.IOException;
import java.math.BigDecimal;
import java.net.URI;
import java.util.Map;

import com.fasterxml.jackson.annotation.JsonFormat;
import com.fasterxml.jackson.annotation.JsonInclude;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.SerializationFeature;
import com.google.common.annotations.VisibleForTesting;
import lombok.Setter;
import org.apache.http.client.HttpClient;
import org.apache.http.client.methods.HttpEntityEnclosingRequestBase;
import org.apache.http.client.methods.HttpUriRequest;
import org.springframework.http.HttpEntity;
import org.springframework.http.HttpHeaders;
import org.springframework.http.HttpMethod;
import org.springframework.http.MediaType;
import org.springframework.http.ResponseEntity;
import org.springframework.http.client.HttpComponentsClientHttpRequestFactory;
import org.springframework.http.converter.json.Jackson2ObjectMapperFactoryBean;
import org.springframework.http.converter.json.MappingJackson2HttpMessageConverter;
import org.springframework.web.client.HttpStatusCodeException;
import org.springframework.web.client.RestClientException;
import org.springframework.web.client.RestTemplate;

import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.function.Function0;
import ru.yandex.chemodan.trust.client.requests.AbstractUserRequest;
import ru.yandex.chemodan.trust.client.requests.CreateOrderRequest;
import ru.yandex.chemodan.trust.client.requests.CreatePaymentRequest;
import ru.yandex.chemodan.trust.client.requests.CreateProductRequest;
import ru.yandex.chemodan.trust.client.requests.CreateRefundRequest;
import ru.yandex.chemodan.trust.client.requests.GetPaymentMethodRequest;
import ru.yandex.chemodan.trust.client.requests.OrderRequest;
import ru.yandex.chemodan.trust.client.requests.PaymentRequest;
import ru.yandex.chemodan.trust.client.requests.ProductRequest;
import ru.yandex.chemodan.trust.client.requests.RefundRequest;
import ru.yandex.chemodan.trust.client.requests.SendInappReceiptRequest;
import ru.yandex.chemodan.trust.client.requests.SupplementRequest;
import ru.yandex.chemodan.trust.client.requests.TrustRequest;
import ru.yandex.chemodan.trust.client.requests.UpdateSubscriptionRequest;
import ru.yandex.chemodan.trust.client.responses.AppleReceiptResponse;
import ru.yandex.chemodan.trust.client.responses.BasicResponse;
import ru.yandex.chemodan.trust.client.responses.CreateOrderResponse;
import ru.yandex.chemodan.trust.client.responses.CreatePaymentResponse;
import ru.yandex.chemodan.trust.client.responses.GenericResponse;
import ru.yandex.chemodan.trust.client.responses.GetPaymentMethodsResponse;
import ru.yandex.chemodan.trust.client.responses.InappSubscriptionResponse;
import ru.yandex.chemodan.trust.client.responses.PaymentResponse;
import ru.yandex.chemodan.trust.client.responses.ProcessInappReceiptResponse;
import ru.yandex.chemodan.trust.client.responses.RefundResponse;
import ru.yandex.chemodan.trust.client.responses.SubscriptionResponse;
import ru.yandex.commune.json.jackson.JodaTimeModule;
import ru.yandex.commune.json.jackson.bolts.BoltsModule;
import ru.yandex.misc.io.http.UriBuilder;
import ru.yandex.misc.lang.StringUtils;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;

import static com.google.common.collect.Lists.newArrayList;

public class TrustClient {
    private static final Logger logger = LoggerFactory.getLogger(TrustClient.class);
    private static final String SERVICE_TOKEN_HEADER = "X-Service-Token";

    @Setter
    private String url;
    private RestTemplate restTemplate;
    private MapF<Integer, String> serviceTokens;
    private final ObjectMapper objectMapper;

    public TrustClient(HttpClient httpClient, String url, MapF<Integer, String> serviceTokens) {
        this.url = url;
        this.serviceTokens = serviceTokens;
        this.objectMapper = buildObjectMapper();
        restTemplate = new RestTemplate(new HttpComponentsClientHttpRequestWithBodyFactory(httpClient));
        restTemplate.setMessageConverters(newArrayList(new MappingJackson2HttpMessageConverter(objectMapper)));
    }

    private static final class HttpComponentsClientHttpRequestWithBodyFactory extends HttpComponentsClientHttpRequestFactory {
        public HttpComponentsClientHttpRequestWithBodyFactory(HttpClient httpClient) {
            super(httpClient);
        }

        @Override
        protected HttpUriRequest createHttpUriRequest(HttpMethod httpMethod, URI uri) {
            if (httpMethod == HttpMethod.GET) {
                return new HttpGetRequestWithEntity(uri);
            }
            return super.createHttpUriRequest(httpMethod, uri);
        }
    }

    private static final class HttpGetRequestWithEntity extends HttpEntityEnclosingRequestBase {
        public HttpGetRequestWithEntity(final URI uri) {
            super.setURI(uri);
        }

        @Override
        public String getMethod() {
            return HttpMethod.GET.name();
        }
    }

    static ObjectMapper buildObjectMapper() {
        TrustObjectMapperFactoryBean mapperFactory = new TrustObjectMapperFactoryBean();
        mapperFactory.setFeaturesToDisable(
                com.fasterxml.jackson.databind.DeserializationFeature.FAIL_ON_UNKNOWN_PROPERTIES,
                SerializationFeature.WRITE_DATES_AS_TIMESTAMPS);
        mapperFactory.setFeaturesToEnable(
                com.fasterxml.jackson.databind.DeserializationFeature.USE_BIG_DECIMAL_FOR_FLOATS,
                com.fasterxml.jackson.databind.DeserializationFeature.READ_UNKNOWN_ENUM_VALUES_USING_DEFAULT_VALUE
        );
        mapperFactory.setSerializationInclusion(JsonInclude.Include.NON_NULL);
        mapperFactory.afterPropertiesSet();

        ObjectMapper mapper = mapperFactory.getObject();
        mapper.registerModule(new BoltsModule());
        mapper.registerModule(new JodaTimeModule());
        return mapper;
    }

    public void createProduct(CreateProductRequest request) {
        ensureSuccess(exchange(uri("products"), HttpMethod.POST, request, BasicResponse.class));
    }

    public GenericResponse getProduct(ProductRequest request) {
        return ensureSuccess(exchange(uri("products/%s", request.getProductId()), HttpMethod.GET, request,
                GenericResponse.class));
    }

    public ProcessInappReceiptResponse processInappReceipt(SendInappReceiptRequest request) {
        return ensureSuccess(exchange(uri("inapp_receipt"), HttpMethod.PUT, request, ProcessInappReceiptResponse.class,
                () -> buildHeaders(request)));
    }

    public AppleReceiptResponse checkAppstoreReceipt(String receipt, Integer inappTrustServiceId) {
        SendInappReceiptRequest request = SendInappReceiptRequest.builder()
                .receipt(receipt)
                .storeId(InappStoreType.APPLE_APPSTORE)
                .trustServiceId(inappTrustServiceId)
                .build();
        return ensureSuccess(exchange(uri("inapp_check_receipt"), HttpMethod.GET, request, AppleReceiptResponse.class,
                () -> addJsonContentType(buildHeaders(request))));
    }

    public Map<?, ?> checkReceipt(InappStoreType storeType, String receipt, Integer inappTrustServiceId) {
        SendInappReceiptRequest request = SendInappReceiptRequest.builder()
                .receipt(receipt)
                .storeId(storeType)
                .trustServiceId(inappTrustServiceId)
                .build();
        return exchange(uri("inapp_check_receipt"), HttpMethod.GET, request, Map.class,
                () -> addJsonContentType(buildHeaders(request))).getBody();
    }

    public InappSubscriptionResponse getInappSubscription(OrderRequest request) {
        return ensureSuccess(exchange(uri("inapp_subscription/%s", request.getOrderId()), HttpMethod.GET, request,
                InappSubscriptionResponse.class, () -> buildHeaders(request)));
    }

    public void resyncInappSubscription(OrderRequest request) {
        ensureSuccess(exchange(uri("inapp_subscription/%s/resync", request.getOrderId()), HttpMethod.POST, request,
                BasicResponse.class, () -> buildHeaders(request)));
    }

    public String createOrder(CreateOrderRequest request) {
        return ensureSuccess(exchange(uri("orders"), HttpMethod.POST, request, CreateOrderResponse.class,
                () -> buildHeaders(request))).getOrderId();
    }

    public String createSubscription(CreateOrderRequest request) {
        return ensureSuccess(exchange(uri("subscriptions"), HttpMethod.POST, request, CreateOrderResponse.class,
                () -> buildHeaders(request))).getOrderId();
    }

    public void updateSubscription(UpdateSubscriptionRequest request) {
        ensureSuccess(exchange(uri("subscriptions/%s", request.getOrderId()), HttpMethod.PUT, request,
                BasicResponse.class, () -> buildHeaders(request)));
    }

    // https://wiki.yandex-team.ru/trust/payments/api/subscriptions/#post/subscriptions/id/resumption
    public SubscriptionResponse resumeSubscription(OrderRequest request) {
        return ensureSuccess(exchange(uri("subscriptions/%s/resumption", request.getOrderId()), HttpMethod.POST,
                request, SubscriptionResponse.class, () -> buildHeaders(request)));
    }

    public SubscriptionResponse getSubscription(OrderRequest request) {
        return ensureSuccess(exchange(uri("subscriptions/%s", request.getOrderId()), HttpMethod.GET, request,
                SubscriptionResponse.class, () -> buildHeaders(request)));
    }

    public GetPaymentMethodsResponse getPaymentMethods(GetPaymentMethodRequest request) {
        return ensureSuccess(exchange(uri("payment_methods"), HttpMethod.GET, request,
                GetPaymentMethodsResponse.class, () -> buildHeaders(request)));
    }

    public String createPayment(CreatePaymentRequest request) {
        return ensureSuccess(exchange(uri("payments"), HttpMethod.POST, request, CreatePaymentResponse.class,
                () -> buildHeaders(request))).getPurchaseToken();
    }

    public PaymentResponse startPayment(PaymentRequest request) {
        return ensureSuccess(exchange(uri("payments/%s/start", request.getPurchaseToken()), HttpMethod.POST, request,
                PaymentResponse.class, () -> buildHeaders(request)));
    }

    public PaymentResponse getPayment(PaymentRequest request) {
        return ensureSuccess(exchange(uri("payments/%s", request.getPurchaseToken()), HttpMethod.GET, request,
                PaymentResponse.class, () -> buildHeaders(request)));
    }

    public String createRefund(CreateRefundRequest request) {
        return ensureSuccess(exchange(uri("refunds"), HttpMethod.POST, request,
                RefundResponse.class, () -> buildHeaders(request))).getRefundId();
    }

    public void startRefund(RefundRequest request) {
        exchange(uri("refunds/%s/start", request.getRefundId()), HttpMethod.POST, request,
                RefundResponse.class, () -> buildHeaders(request));
    }

    public RefundResponse getRefund(RefundRequest request) {
        return exchange(uri("refunds/%s", request.getRefundId()), HttpMethod.GET, request,
                RefundResponse.class, () -> buildHeaders(request)).getBody();
    }

    public SubscriptionResponse supplementSubscription(SupplementRequest request) {
        return ensureSuccess(exchange(uri("subscriptions/%s/supplement", request.getOrderId()), HttpMethod.POST,
                request,
                SubscriptionResponse.class, () -> buildHeaders(request)));
    }


    private HttpHeaders addJsonContentType(HttpHeaders headers) {
        headers.add(HttpHeaders.CONTENT_TYPE, MediaType.APPLICATION_JSON_VALUE);
        return headers;
    }

    private HttpHeaders buildHeaders(AbstractUserRequest request) {
        HttpHeaders httpHeaders = new HttpHeaders();
        if (StringUtils.isNotEmpty(request.getRegionId())) {
            httpHeaders.add("X-Region-Id", request.getRegionId());
        }
        if (StringUtils.isNotEmpty(request.getUserIp())) {
            httpHeaders.add("X-User-Ip", request.getUserIp());
        }
        if (request.getUid() != null) {
            httpHeaders.add("X-Uid", request.getUid().toString());
        }
        return httpHeaders;
    }

    private void addServiceToken(HttpHeaders headers, TrustRequest request) {
        String token = serviceTokens.getOrThrow(request.getTrustServiceId(),
                "Unable to find serviceToken for trustService " + request.getTrustServiceId());
        headers.add(SERVICE_TOKEN_HEADER, token);
    }


    private static <R extends BasicResponse> R ensureSuccess(ResponseEntity<R> response) {
        R body = response.getBody();
        if (body == null || !body.isSuccess()) {
            throw new TrustException(Option.of(response.getStatusCode()), body);
        }
        return body;
    }

    private <R> ResponseEntity<R> get(UriBuilder urlBuilder, Class<R> responseType) {
        return exchange(urlBuilder, HttpMethod.GET, null, responseType, null);
    }

    private UriBuilder uri(String path, Object... variables) {
        return UriBuilder.cons(url).appendPath(String.format(path, variables));
    }

    private <R> ResponseEntity<R> exchange(UriBuilder urlBuilder, HttpMethod method, TrustRequest request,
                                           Class<R> responseType) {
        return exchange(urlBuilder, method, request, responseType, null);
    }

    private <R> ResponseEntity<R> exchange(UriBuilder urlBuilder, HttpMethod method, TrustRequest request,
                                           Class<R> responseType,
                                           Function0<HttpHeaders> headersProducer) {
        HttpHeaders headers = headersProducer != null ? headersProducer.apply() : new HttpHeaders();
        addServiceToken(headers, request);
        HttpEntity<Object> entity = new HttpEntity<>(request.toRequestBody(), headers);

        logger.info("Calling TRUST {} '{}' with body entity: {}", method, urlBuilder, request.toRequestBody());
        try {
            ResponseEntity<R> responseEntity = restTemplate.exchange(urlBuilder.build(), method, entity, responseType);

            logger.info("TRUST parsed response http code: {}, body entity: {}",
                    responseEntity.getStatusCode(), responseEntity.getBody());

            return responseEntity;
        } catch (HttpStatusCodeException e) {
            ResponseEntity<R> responseEntity = ResponseEntity.status(e.getStatusCode())
                    .headers(e.getResponseHeaders())
                    .body(parseErrorBody(responseType, e));

            logger.error("TRUST parsed response http code: {}, body entity: {}",
                    responseEntity.getStatusCode(), responseEntity.getBody(), e);

            return responseEntity;
        } catch (RestClientException e) {
            throw new TrustException(e.getMessage(), e);
        }
    }

    private <R> R parseErrorBody(Class<R> responseType, HttpStatusCodeException httpException) {
        MediaType contentType = httpException.getResponseHeaders().getContentType();
        if (!MediaType.APPLICATION_JSON.isCompatibleWith(contentType)) {
            return null;
        }

        byte[] body = httpException.getResponseBodyAsByteArray();

        try {
            return objectMapper.readValue(body, responseType);
        } catch (IOException e) {
            logger.warn("TRUST Error parse body: {}", new String(body), e);
            return null;
        }
    }

    private static class TrustObjectMapperFactoryBean extends Jackson2ObjectMapperFactoryBean {
        @Override
        public void afterPropertiesSet() {
            super.afterPropertiesSet();
            getObject().configOverride(BigDecimal.class).setFormat(JsonFormat.Value.forShape(JsonFormat.Shape.STRING));
        }
    }

    @VisibleForTesting
    RestTemplate getRestTemplate() {
        return restTemplate;
    }
}
