package ru.yandex.chemodan.trust.client.responses;

import java.math.BigDecimal;
import java.text.DecimalFormat;
import java.text.ParseException;

import com.fasterxml.jackson.annotation.JsonAnyGetter;
import com.fasterxml.jackson.annotation.JsonAnySetter;
import com.fasterxml.jackson.annotation.JsonProperty;
import com.fasterxml.jackson.databind.annotation.JsonDeserialize;
import com.fasterxml.jackson.databind.annotation.JsonSerialize;
import lombok.AllArgsConstructor;
import lombok.Builder;
import lombok.Getter;
import lombok.NoArgsConstructor;
import org.joda.time.Instant;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.trust.client.DateAdapter;
import ru.yandex.misc.lang.StringUtils;

@Getter
@AllArgsConstructor
@NoArgsConstructor
@Builder
public class SubscriptionResponse extends BasicResponse {
    private static DecimalFormat format = new DecimalFormat("#.##");

    static {
        format.setParseBigDecimal(true);
    }

    @JsonProperty("order_id")
    private String orderId;
    @JsonProperty("product_id")
    private String productId;
    @JsonProperty("subs_period")
    private String subscriptionPeriod;
    @JsonProperty("subs_period_count")
    private int subscriptionPeriodCount;
    @JsonProperty("subs_until_ts")
    @JsonSerialize(using = DateAdapter.Serializer.class)
    @JsonDeserialize(using = DateAdapter.Deserializer.class)
    private Instant subscriptionUntil;
    @JsonProperty("finish_ts")
    @JsonSerialize(using = DateAdapter.Serializer.class)
    @JsonDeserialize(using = DateAdapter.Deserializer.class)
    private Instant finishTime;
    @JsonProperty("payments")
    private ListF<String> paymentIds;
    @JsonProperty("region_id")
    private String regionId;
    @JsonProperty("current_amount")
    private String[][] currentAmount;
    @JsonProperty("subs_state")
    private int subscriptionState;
    @JsonProperty("begin_ts")
    @JsonSerialize(using = DateAdapter.Serializer.class)
    @JsonDeserialize(using = DateAdapter.Deserializer.class)
    private Instant createdAt;
    @JsonProperty("grace_intervals")
    private ListF<InstantInterval> graceIntervals;
    @JsonProperty("hold_intervals")
    private ListF<InstantInterval> holdIntervals;

    private MapF<String, Object> otherAttributes = Cf.hashMap();

    public SubscriptionState getSubscriptionState() {
        return SubscriptionState.fromInt(subscriptionState);
    }

    @JsonAnySetter
    public void addAttribute(String name, Object value) {
        otherAttributes.put(name, value);
    }

    @JsonAnyGetter
    public MapF<String, Object> getOtherAttributes() {
        return otherAttributes;
    }

    public Option<BigDecimal> getCurrentAmount() {
        if (currentAmount == null || currentAmount.length != 1) {
            return Option.empty();
        }

        ListF<String> amountAndCurrency = Cf.x(currentAmount[0]);
        return amountAndCurrency.filterMap(c -> {
            Option<Number> num = Option.empty();
            try {
                num = Option.of(format.parse(c));
            } catch (ParseException ignored) {
            }
            return num;
        }).cast(BigDecimal.class).firstO();
    }

    public Option<String> getCurrency() {
        if (currentAmount == null || currentAmount.length != 1) {
            return Option.empty();
        }
        ListF<String> amountAndCurrency = Cf.x(currentAmount[0]);
        return amountAndCurrency.filter(StringUtils::isAlpha).firstO();
    }

    // https://wiki.yandex-team.ru/trust/payments/api/subscriptions/
    // Возможные значения для subs_state :
    //0 - идет триальный период, способ оплаты не указан;
    //1 - идет триальный период, способ оплаты указан;
    //2 - триальный период закончился, неоплачена;
    //3 - оплачена;
    //4 - остановлена
    //5 - остановлена, и у продукта есть subs_introductory_period
    //6 - идет introductory_period, оплачена
    //7 - introductory_period закончился, неоплачена
    public enum SubscriptionState {
        Unknown(-1),
        TrialNoPaymentMethod(0),
        TrialWithPaymentMethod(1),
        TrialFinishedAndNotPaid(2),
        Paid(3),
        Stopped(4),
        StoppedWithIntroductory(5),
        PaidWithIntroductory(6),
        NotPaidAndIntroductoryFinished(7);

        @Getter
        private final int value;

        SubscriptionState(int value) {
            this.value = value;
        }

        private static final MapF<Integer, SubscriptionState> values = Cf.toMap(Cf.Tuple2List.fromPairs(
                TrialNoPaymentMethod.value, TrialNoPaymentMethod,
                TrialWithPaymentMethod.value, TrialWithPaymentMethod,
                TrialFinishedAndNotPaid.value, TrialFinishedAndNotPaid,
                Paid.value, Paid,
                Stopped.value, Stopped,
                StoppedWithIntroductory.value, StoppedWithIntroductory,
                PaidWithIntroductory.value, PaidWithIntroductory,
                NotPaidAndIntroductoryFinished.value, NotPaidAndIntroductoryFinished
        ));

        public static SubscriptionState fromInt(int value) {
            if (values.containsKeyTs(value)) {
                return values.getTs(value);
            }
            return Unknown;
        }

        public boolean isInTrial() {
            return this.equals(TrialNoPaymentMethod) || this.equals(TrialWithPaymentMethod);
        }
    }
}
