package ru.yandex.chemodan.app.psbilling.web.actions;

import java.util.UUID;

import lombok.AllArgsConstructor;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.psbilling.core.entities.groups.Group;
import ru.yandex.chemodan.app.psbilling.core.entities.groups.GroupService;
import ru.yandex.chemodan.app.psbilling.core.groups.GroupsManager;
import ru.yandex.chemodan.app.psbilling.web.actions.groups.GroupActionsHelper;
import ru.yandex.chemodan.app.psbilling.web.actions.users.UserBillingActionsHelper;
import ru.yandex.chemodan.app.psbilling.web.converter.ConverterToUUID;
import ru.yandex.chemodan.app.psbilling.web.model.GroupServicesPojo;
import ru.yandex.chemodan.app.psbilling.web.model.GroupTypeApi;
import ru.yandex.chemodan.app.psbilling.web.model.ServiceStatus;
import ru.yandex.chemodan.app.psbilling.web.model.ServiceStatusFilter;
import ru.yandex.chemodan.app.psbilling.web.model.UserServicesPojo;
import ru.yandex.chemodan.app.psbilling.web.services.SubscriptionService;
import ru.yandex.chemodan.directory.client.DirectoryClient;
import ru.yandex.chemodan.util.exception.A3ExceptionWithStatus;
import ru.yandex.commune.a3.action.ActionContainer;
import ru.yandex.commune.a3.action.HttpMethod;
import ru.yandex.commune.a3.action.Path;
import ru.yandex.commune.a3.action.parameter.bind.annotation.PathParam;
import ru.yandex.commune.a3.action.parameter.bind.annotation.RequestParam;
import ru.yandex.commune.a3.action.parameter.bind.annotation.SpecialParam;
import ru.yandex.inside.passport.PassportUidOrZero;
import ru.yandex.inside.passport.tvm2.web.Tvm2Filter;
import ru.yandex.misc.db.masterSlave.MasterSlavePolicy;
import ru.yandex.misc.db.masterSlave.WithMasterSlavePolicy;
import ru.yandex.misc.io.http.HttpStatus;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;
import ru.yandex.misc.web.servlet.HttpServletRequestX;

/**
 * @author yashunsky
 */
@ActionContainer
@AllArgsConstructor
public class MailSupportActions {
    private static final Logger logger = LoggerFactory.getLogger(MailSupportActions.class);

    private final int mailSupportClientId;

    private final GroupsManager groupsManager;
    private final GroupActionsHelper groupActionsHelper;
    private final UserBillingActionsHelper userBillingActionsHelper;
    private final SubscriptionService subscriptionService;
    private final DirectoryClient directoryClient;

    @Path(value = "/v1/mailsupport/groups/{groupType}/{groupExternalId}/services", methods = HttpMethod.GET)
    @WithMasterSlavePolicy(MasterSlavePolicy.R_SYNC_SM)
    public GroupServicesPojo getGroupServices(
            @PathParam("groupType") GroupTypeApi groupType,
            @PathParam("groupExternalId") String groupExternalId,
            @RequestParam(value = "lang", required = false) Option<String> languageO,
            @RequestParam(value = "status", required = false) Option<String> statusO,
            @SpecialParam HttpServletRequestX req)
    {
        checkAccess(req);

        Group group = groupsManager.findGroupTrustedOrThrow(groupType.toCoreEnum(), groupExternalId);
        return groupActionsHelper.getGroupServices(group, languageO, group.getOwnerUid(), Cf.list(),
                ServiceStatus.statusesToTarget(statusO));
    }

    @Path(value = "/v1/mailsupport/users/services", methods = HttpMethod.GET)
    @WithMasterSlavePolicy(MasterSlavePolicy.R_SYNC_SM)
    public UserServicesPojo getUserServices(
            @RequestParam("uid") PassportUidOrZero uid,
            @RequestParam(value = "lang", required = false) String language,
            @RequestParam(value = "status", required = false) Option<String> statusO,
            @SpecialParam HttpServletRequestX req)
    {
        checkAccess(req);
        return userBillingActionsHelper.getUserServices(
                uid, Option.empty(), language, Cf.list(), ServiceStatusFilter.parseFilters(statusO,
                        ServiceStatusFilter.ENABLED));
    }

    @Path(value = "/v1/mailsupport/group/make_free_subscription", methods = HttpMethod.POST)
    @WithMasterSlavePolicy(MasterSlavePolicy.RW_M)
    public void makeFreeSubscription(
            @RequestParam("groupExternalId") String groupExternalId,
            @RequestParam("code") String code,
            @SpecialParam HttpServletRequestX req) {
        boolean isPartnerClient = directoryClient.getOrganizationPartnerId(groupExternalId).isPresent();
        logger.info("Received request to make free subscription for groupExternalId={}, code={}, " +
                        "isPartnerClient={}", groupExternalId, code, isPartnerClient);
        checkAccess(req);

        if (!groupsManager.hasPaymentInfoForOrganization(groupExternalId)) {
            throw new A3ExceptionWithStatus("There is no payment info for organization " + groupExternalId,
                    HttpStatus.SC_400_BAD_REQUEST);
        }

        GroupService groupService = subscriptionService.makeFreeSubscription(groupExternalId, code);
        logger.info("Subscription successful: {}", groupService.toString());
    }

    @Path(value = "/v1/mailsupport/group/revoke_subscription", methods = HttpMethod.POST)
    @WithMasterSlavePolicy(MasterSlavePolicy.RW_M)
    public void revokeSubscription(
            @RequestParam("groupExternalId") String groupExternalId,
            @RequestParam(value = "serviceId", required = false, customConverter = ConverterToUUID.class)
                    Option<UUID> serviceId,
            @SpecialParam HttpServletRequestX req) {
        boolean isPartnerClient = directoryClient.getOrganizationPartnerId(groupExternalId).isPresent();
        logger.info("Received request to revoke subscription " +
                "for groupExternalId={}, serviceId={}, isPartnerClient={}",
                groupExternalId, serviceId, isPartnerClient);
        checkAccess(req);

        subscriptionService.revokeGroupSubscription(groupExternalId, serviceId);
    }

    private void checkAccess(HttpServletRequestX req) {
        Option<Integer> clientId = Option.ofNullable(req.getAttribute(Tvm2Filter.TVM_CLIENT_ID_ATTR))
                .map(tvmAttribute -> (Integer) tvmAttribute);

        if (!clientId.isPresent()) {
            throw new A3ExceptionWithStatus("No tvm client id provided", HttpStatus.SC_401_UNAUTHORIZED);
        } else if (!clientId.isSome(mailSupportClientId)) {
            throw new A3ExceptionWithStatus("Access not allowed for " + clientId.get(), HttpStatus.SC_403_FORBIDDEN);
        }
    }

}
