package ru.yandex.chemodan.app.psbilling.web.actions.groups;

import java.util.Currency;

import lombok.AllArgsConstructor;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.psbilling.core.entities.cards.CardPurpose;
import ru.yandex.chemodan.app.psbilling.core.entities.groups.BalancePaymentInfo;
import ru.yandex.chemodan.app.psbilling.core.entities.groups.Group;
import ru.yandex.chemodan.app.psbilling.core.groups.GroupsManager;
import ru.yandex.chemodan.app.psbilling.core.groups.OptionalFeaturesManager;
import ru.yandex.chemodan.app.psbilling.core.model.RequestInfo;
import ru.yandex.chemodan.app.psbilling.core.promocodes.model.SafePromoCode;
import ru.yandex.chemodan.app.psbilling.web.converter.ConverterToSafePromoCode;
import ru.yandex.chemodan.app.psbilling.web.model.GroupAddonsPojo;
import ru.yandex.chemodan.app.psbilling.web.model.GroupPojo;
import ru.yandex.chemodan.app.psbilling.web.model.GroupProductPojo;
import ru.yandex.chemodan.app.psbilling.web.model.GroupProductSetPojo;
import ru.yandex.chemodan.app.psbilling.web.model.GroupServicesPojo;
import ru.yandex.chemodan.app.psbilling.web.model.GroupTypeApi;
import ru.yandex.chemodan.app.psbilling.web.model.GroupsByPayerResponsePojo;
import ru.yandex.chemodan.app.psbilling.web.model.PaymentCardsPojo;
import ru.yandex.chemodan.app.psbilling.web.model.ServiceStatus;
import ru.yandex.chemodan.app.psbilling.web.model.TrustCardBindingFormPojo;
import ru.yandex.chemodan.app.psbilling.web.validation.GroupAdminPermissionValidation;
import ru.yandex.chemodan.app.psbilling.web.validation.GroupValidator;
import ru.yandex.chemodan.util.exception.A3ExceptionWithStatus;
import ru.yandex.chemodan.util.exception.AccessForbiddenException;
import ru.yandex.chemodan.util.web.Tvm2UidParameterBinder;
import ru.yandex.chemodan.web.EmptyPojo;
import ru.yandex.commune.a3.action.ActionContainer;
import ru.yandex.commune.a3.action.HttpMethod;
import ru.yandex.commune.a3.action.Path;
import ru.yandex.commune.a3.action.parameter.bind.annotation.BindWith;
import ru.yandex.commune.a3.action.parameter.bind.annotation.PathParam;
import ru.yandex.commune.a3.action.parameter.bind.annotation.RequestListParam;
import ru.yandex.commune.a3.action.parameter.bind.annotation.RequestParam;
import ru.yandex.commune.a3.security.UnauthorizedException;
import ru.yandex.inside.passport.PassportUid;
import ru.yandex.inside.passport.PassportUidOrZero;
import ru.yandex.misc.db.masterSlave.MasterSlavePolicy;
import ru.yandex.misc.db.masterSlave.WithMasterSlavePolicy;
import ru.yandex.misc.env.EnvironmentType;
import ru.yandex.misc.env.EnvironmentTypeHolder;
import ru.yandex.misc.io.http.HttpStatus;
import ru.yandex.misc.lang.StringUtils;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;

/**
 * common methods for all group types
 */
@ActionContainer
@AllArgsConstructor
public class GroupActions {
    private static final Logger logger = LoggerFactory.getLogger(GroupActions.class);

    private final GroupsManager groupsManager;
    private final GroupActionsHelper groupActionsHelper;
    private final OptionalFeaturesManager optionalFeaturesManager;
    private final GroupAdminPermissionValidation groupAdminPermissionValidation;
    private final GroupValidator groupValidator;

    @Path(value = "/v1/groups/{groupType}/{groupExternalId}/services", methods = HttpMethod.GET)
    @WithMasterSlavePolicy(MasterSlavePolicy.R_SYNC_SM)
    public GroupServicesPojo getGroupServices(
            @BindWith(Tvm2UidParameterBinder.class) PassportUidOrZero uid,
            @PathParam("groupType") GroupTypeApi groupType,
            @PathParam("groupExternalId") String groupId,
            @RequestParam(value = "lang", required = false) Option<String> language,
            @RequestParam(value = "status", required = false) Option<String> statusO) {
        logger.info("getGroupServices: uid: {}, group_type: {}, group_external_id: {}, status = {}, lang = {}",
                uid, groupType, groupId, statusO, language);

        if (!uid.isAuthenticated()) {
            throw new UnauthorizedException("uid isn't specified");
        }

        groupAdminPermissionValidation.check(uid.toUid(), groupId);

        Group group = groupsManager.findGroupOrThrow(uid.toPassportUid(), groupType.toCoreEnum(), groupId);

        return groupActionsHelper
                .getGroupServices(group, language, uid.toPassportUid(), Cf.list(),
                        ServiceStatus.statusesToTarget(statusO));
    }

    @Path(value = "/v1/groups/{groupType}/{groupExternalId}/addons", methods = HttpMethod.GET)
    @WithMasterSlavePolicy(MasterSlavePolicy.R_SYNC_SM)
    public GroupAddonsPojo getGroupAddons(
            @BindWith(Tvm2UidParameterBinder.class) PassportUidOrZero uid,
            @PathParam("groupType") GroupTypeApi groupType,
            @PathParam("groupExternalId") String groupId,
            @RequestParam(value = "lang", required = false) Option<String> language,
            @RequestParam(value = "status", required = false) Option<String> statusO) {
        logger.info("getGroupAddons: uid: {}, group_type: {}, group_external_id: {}, status = {}, lang = {}",
                uid, groupType, groupId, statusO, language);

        if (!uid.isAuthenticated()) {
            throw new UnauthorizedException("uid isn't specified");
        }

        groupAdminPermissionValidation.check(uid.toUid(), groupId);

        Group group = groupsManager.findGroupOrThrow(uid.toPassportUid(), groupType.toCoreEnum(), groupId);

        return groupActionsHelper.getGroupAddons(group, language, uid.toPassportUid(), Cf.list(),true,
                ServiceStatus.statusesToTarget(statusO));
    }

    @Path(value = "/v1/groups/{groupType}/{groupExternalId}/feature/{featureCode}", methods = HttpMethod.PUT)
    @WithMasterSlavePolicy(MasterSlavePolicy.R_M)
    public EmptyPojo enableOptionalFeature(
            @PathParam("groupType") GroupTypeApi groupType,
            @PathParam("groupExternalId") String groupId,
            @PathParam("featureCode") String featureCode) {
        logger.info("enableOptionalFeature: group_type: {}, group_external_id: {}, " +
                "featureCode = {}", groupType, groupId, featureCode);

        Group group = groupsManager.findGroupTrustedOrThrow(groupType.toCoreEnum(), groupId);
        optionalFeaturesManager.enable(group, group.getOwnerUid(), featureCode);

        return EmptyPojo.INSTANCE;
    }

    @Path(value = "/v1/groups/{groupType}/{groupExternalId}/feature/{featureCode}", methods = HttpMethod.DELETE)
    @WithMasterSlavePolicy(MasterSlavePolicy.R_M)
    public EmptyPojo disableOptionalFeature(
            @PathParam("groupType") GroupTypeApi groupType,
            @PathParam("groupExternalId") String groupId,
            @PathParam("featureCode") String featureCode) {
        logger.info("disableOptionalFeature: group_type: {}, group_external_id: {}, " +
                "featureCode = {}", groupType, groupId, featureCode);

        Group group = groupsManager.findGroupTrustedOrThrow(groupType.toCoreEnum(), groupId);
        optionalFeaturesManager.disable(group, group.getOwnerUid(), featureCode);

        return EmptyPojo.INSTANCE;
    }

    @Path(value = "/v1/groups/{groupType}/{groupExternalId}", methods = HttpMethod.GET)
    @WithMasterSlavePolicy(MasterSlavePolicy.R_SM)
    public GroupPojo getGroup(
            @BindWith(Tvm2UidParameterBinder.class) PassportUidOrZero uid,
            @PathParam("groupType") GroupTypeApi groupType,
            @PathParam("groupExternalId") String groupId) {
        logger.info("getGroup: uid: {} group_type: {}, group_external_id: {}",
                uid, groupType, groupId);
        if (!uid.isAuthenticated()) {
            throw new UnauthorizedException("uid isn't specified");
        }

        groupAdminPermissionValidation.check(uid.toUid(), groupId);

        return groupActionsHelper.getGroupPojo(uid.toPassportUid(), groupType.toCoreEnum(), groupId);
    }

    @Path(value = "/v1/groups/products/{productId}", methods = HttpMethod.GET)
    @WithMasterSlavePolicy(MasterSlavePolicy.R_SM)
    public GroupProductPojo getProductInfo(
            @BindWith(Tvm2UidParameterBinder.class) PassportUidOrZero uid,
            @RequestParam(value = "groupType", ignoreEmpty = true, required = false) Option<GroupTypeApi> groupType,
            @RequestParam(value = "groupId", ignoreEmpty = true, required = false) Option<String> groupId,
            @RequestParam(value = "lang", required = false) Option<String> language,
            @PathParam("productId") String productId) {
        logger.info("getProductInfo: uid: {}, group_type: {}, group_external_id: {}, productId = {}",
                uid, groupType, groupId, productId);
        if (!uid.isAuthenticated()) {
            throw new UnauthorizedException("uid isn't specified");
        }

        if (groupId.isPresent()) {
            groupAdminPermissionValidation.check(uid.toUid(), groupId.get());
        }

        Option<Group> group = groupActionsHelper.getGroup(uid.toPassportUid(), groupType, groupId);
        return groupActionsHelper.getProductInfo(group, language, uid.toPassportUid(), productId, Cf.list());
    }

    @Path(value = "/v1/groups/{groupType}/{groupExternalId}/reset_payment_data", methods = HttpMethod.PUT)
    @WithMasterSlavePolicy(MasterSlavePolicy.R_M)
    public EmptyPojo resetGroupPaymentData(
            @RequestParam(value = "groupType") GroupTypeApi groupType,
            @RequestParam(value = "groupId") String groupId) {
        if (EnvironmentTypeHolder.environmentType == EnvironmentType.PRODUCTION) {
            throw new IllegalStateException("unable to reset payment data in production environment");
        }

        Group group = groupsManager.findGroupTrustedOrThrow(groupType.toCoreEnum(), groupId);
        groupsManager.removePaymentData(group);

        return EmptyPojo.INSTANCE;
    }

    @Path(value = "/v1/groups/{groupType}/{groupExternalId}/payment/settings/autobilling", methods = HttpMethod.PUT)
    @WithMasterSlavePolicy(MasterSlavePolicy.R_M)
    public EmptyPojo enableAutobilling(
            @BindWith(Tvm2UidParameterBinder.class) PassportUidOrZero uid,
            @PathParam("groupType") GroupTypeApi groupType,
            @PathParam("groupExternalId") String groupId) {
        logger.info("enableAutobilling: group_type: {}, group_external_id: {}",
                groupType, groupId);

        setAutobilling(uid, groupType, groupId, true);

        return EmptyPojo.INSTANCE;
    }

    @Path(value = "/v1/groups/{groupType}/{groupExternalId}/payment/settings/autobilling", methods = HttpMethod.DELETE)
    @WithMasterSlavePolicy(MasterSlavePolicy.R_M)
    public EmptyPojo disableAutobilling(
            @BindWith(Tvm2UidParameterBinder.class) PassportUidOrZero uid,
            @PathParam("groupType") GroupTypeApi groupType,
            @PathParam("groupExternalId") String groupId) {
        logger.info("disableAutobilling: group_type: {}, group_external_id: {}",
                groupType, groupId);

        setAutobilling(uid, groupType, groupId, false);

        return EmptyPojo.INSTANCE;
    }

    @Path(value = "/v1/groups/productsets/{key}/products", methods = HttpMethod.GET)
    @WithMasterSlavePolicy(MasterSlavePolicy.R_SM)
    public GroupProductSetPojo getGroupProductSet(
            @BindWith(Tvm2UidParameterBinder.class) PassportUidOrZero uid,
            @PathParam(value = "key") String productSetKey,
            @RequestParam(value = "groupType", ignoreEmpty = true, required = false) Option<GroupTypeApi> groupType,
            @RequestParam(value = "groupId", ignoreEmpty = true, required = false) Option<String> groupId,
            @RequestParam(value = "lang", required = false) String language,
            @RequestParam(value = "fromLanding", ignoreEmpty = true, required = false) boolean fromLanding,
            @RequestListParam(value = "productExperiments", required = false) ListF<String> productExperiments,
            @RequestParam(value = "promoCode", required = false, customConverter = ConverterToSafePromoCode.class) Option<SafePromoCode> promoCode,
            @RequestParam(value = "userAgent", required = false) Option<String> userAgent,
            @RequestParam(value = "userIp", required = false) Option<String> userIp,
            @RequestParam(value = "userDeviceId", required = false) Option<String> userDeviceId,
            @RequestParam(value = "userLoginId", required = false) Option<String> userLoginId
    ) {
        logger.info("getProductSet -> uid: {}, key: {}, group: {}, lang: {}, promoCode: {}, fromLanding: {}, " +
                        "userAgent: {}, userIp: {}, userDeviceId: {}, userLoginId: {}",
                uid,
                productSetKey,
                groupId,
                language,
                promoCode,
                fromLanding,
                userAgent,
                userIp,
                userDeviceId,
                userLoginId
        );

        Option<PassportUid> uidO = uid.toPassportUidO();
        Option<Group> groupO = Option.empty();

        if (groupId.isPresent()) {
            if (!uid.isAuthenticated()) {
                throw new UnauthorizedException("uid isn't specified");
            }
            groupO = groupActionsHelper.getGroup(uid.toPassportUid(), groupType, groupId);
        }

        RequestInfo requestInfo = RequestInfo.cons(userAgent, userIp, userDeviceId, userLoginId);

        return groupActionsHelper.getGroupProductSet(
                uidO,
                productSetKey,
                groupO,
                StringUtils.isBlank(language) ? Option.empty() : Option.of(language),
                Cf.list(),
                fromLanding,
                productExperiments,
                promoCode,
                requestInfo
        );
    }

    @Path(value = "/v1/groups/organization/by_payer", methods = HttpMethod.GET)
    @WithMasterSlavePolicy(MasterSlavePolicy.R_SM)
    public GroupsByPayerResponsePojo getGroupsByPayer(
            @BindWith(Tvm2UidParameterBinder.class) PassportUidOrZero uid,
            @RequestParam(value = "lang", required = false) String language
    ) {
        logger.info("getGroupsByPayer -> uid: {}, lang: {}", uid, language);
        if (!uid.isAuthenticated()) {
            throw new UnauthorizedException("uid isn't specified");
        }

        return groupActionsHelper.getGroupsByPayer(uid.toUid(), language);
    }

    @Path(value = "/v1/groups/{groupType}/{groupExternalId}/payment/cards", methods = HttpMethod.GET)
    @WithMasterSlavePolicy(MasterSlavePolicy.R_SM)
    public PaymentCardsPojo getGroupPaymentCards(
            @BindWith(Tvm2UidParameterBinder.class) PassportUidOrZero uid,
            @PathParam("groupType") GroupTypeApi groupType,
            @PathParam("groupExternalId") String groupId) {
        logger.info("getGroupPaymentCards: uid: {}, group_type: {}, group_external_id: {}",
                uid, groupType, groupId);

        if (!uid.isAuthenticated()) {
            throw new UnauthorizedException("uid isn't specified");
        }

        groupAdminPermissionValidation.check(uid.toUid(), groupId);

        Group group = groupsManager.findGroupOrThrow(uid.toPassportUid(), groupType.toCoreEnum(), groupId);

        groupValidator.checkIsNotPartnerClient(group);

        Option<BalancePaymentInfo> paymentInfo = checkAndGetPaymentInfo(group, uid.toUid(), true);

        if (paymentInfo.isPresent()) {
            return groupActionsHelper.getPaymentCards(paymentInfo.get());
        }

        return new PaymentCardsPojo(Cf.list());
    }

    @Path(value = "/v1/groups/{groupType}/{groupExternalId}/payment/cards/{cardExternalId}", methods = HttpMethod.PUT)
    @WithMasterSlavePolicy(MasterSlavePolicy.R_M)
    public EmptyPojo setCardPurpose(
            @BindWith(Tvm2UidParameterBinder.class) PassportUidOrZero uid,
            @PathParam("groupType") GroupTypeApi groupType,
            @PathParam("groupExternalId") String groupId,
            @PathParam("cardExternalId") String cardExternalId,
            @RequestParam("purpose") String purpose) {
        logger.info("setCardPurpose: group_type: {}, group_external_id: {}, purpose: {}",
                groupType, groupId, purpose);

        if (!uid.isAuthenticated()) {
            throw new UnauthorizedException("uid isn't specified");
        }

        groupAdminPermissionValidation.check(uid.toUid(), groupId);

        CardPurpose cardPurpose;
        try {
            cardPurpose = CardPurpose.R.valueOf(purpose);
        } catch (IllegalArgumentException ex) {
            throw new A3ExceptionWithStatus("unsupported_card_purpose",
                    "Unsupported card purpose", HttpStatus.SC_400_BAD_REQUEST);
        }

        PassportUid callerUid = uid.toUid();
        Group group = groupsManager.findGroupOrThrow(callerUid, groupType.toCoreEnum(), groupId);

        groupValidator.checkIsNotPartnerClient(group);

        BalancePaymentInfo paymentInfo = checkAndGetPaymentInfo(group, callerUid, false).get();

        groupActionsHelper.setCardPurpose(callerUid, cardExternalId, cardPurpose, paymentInfo);

        return EmptyPojo.INSTANCE;
    }

    @Path(value = "/v1/groups/{groupType}/{groupExternalId}/payment/cards/binding_form", methods = HttpMethod.GET)
    @WithMasterSlavePolicy(MasterSlavePolicy.R_M)
    public TrustCardBindingFormPojo getCardBindingForm(
            @BindWith(Tvm2UidParameterBinder.class) PassportUidOrZero uid,
            @PathParam("groupType") GroupTypeApi groupType,
            @PathParam("groupExternalId") String groupId,
            @RequestParam("currency") String currency,
            @RequestParam(value = "theme", required = false) Option<String> theme,
            @RequestParam(value = "title", required = false) Option<String> title) {

        logger.info("getCardBindingForm: uid: {}, group_type: {}, " +
                        "group_external_id: {}, currency: {}",
                uid, groupType, groupId, currency);

        if (!uid.isAuthenticated()) {
            throw new UnauthorizedException("uid isn't specified");
        }

        groupAdminPermissionValidation.check(uid.toUid(), groupId);

        Group group = groupsManager.findGroupOrThrow(uid.toPassportUid(),
                groupType.toCoreEnum(), groupId);

        groupValidator.checkIsNotPartnerClient(group);

        checkAndGetPaymentInfo(group, uid.toUid(), false);

        return groupActionsHelper.getCardBindingForm(uid.toUid(),
                Currency.getInstance(currency), theme, title);
    }

    @Path(value = "/v1/payments/binding/{bindingId}/notify", methods = HttpMethod.GET)
    @WithMasterSlavePolicy(MasterSlavePolicy.R_M)
    public EmptyPojo checkTrustCardBindingNotification(@PathParam("bindingId") String bindingId) {
        logger.info("checkTrustCardBindingNotification: bindingId: {}", bindingId);

        groupActionsHelper.checkCardBindingStatus(bindingId);

        logger.info("checkTrustCardBindingNotification completed-> bindingId: {}", bindingId);
        return EmptyPojo.INSTANCE;
    }

    private void setAutobilling(PassportUidOrZero uid, GroupTypeApi groupType,
                                String groupId, boolean enableAutobilling) {
        if (!uid.isAuthenticated()) {
            throw new UnauthorizedException("uid isn't specified");
        }

        groupAdminPermissionValidation.check(uid.toUid(), groupId);

        Group group = groupsManager.findGroupOrThrow(uid.toPassportUid(), groupType.toCoreEnum(), groupId);

        groupValidator.checkIsNotPartnerClient(group);

        BalancePaymentInfo paymentInfo = checkAndGetPaymentInfo(group, uid.toUid(), false).get();

        if (paymentInfo.isB2bAutoBillingEnabled() == enableAutobilling) {
            return; // настройка уже в нужном состоянии
        }

        if (enableAutobilling) {
            groupActionsHelper.enableAutoBilling(paymentInfo);
        } else {
            groupActionsHelper.disableAutoBilling(paymentInfo);
        }
    }

    private Option<BalancePaymentInfo> checkAndGetPaymentInfo(Group group, PassportUid callerUid,
                                                              boolean allowEmptyPaymentInfo) {
        Option<BalancePaymentInfo> paymentInfo = group.getPaymentInfo();
        if (paymentInfo.isEmpty()) {
            logger.warn("group {} has no payment info", group);
            if (allowEmptyPaymentInfo) {
                return paymentInfo;
            }
            throw new A3ExceptionWithStatus("no_payment_info",
                    "No payment info for this organization", HttpStatus.SC_400_BAD_REQUEST);
        }

        if (!paymentInfo.get().getPassportUid().equalsTs(callerUid)) {
            logger.warn("the caller uid {} is different from the payer uid {}",
                    callerUid.toString(), paymentInfo.get().getPassportUid().toString());
            throw new AccessForbiddenException("the caller uid is different from the payer uid");
        }

        return paymentInfo;
    }
}
