package ru.yandex.chemodan.app.psbilling.web.actions.groups;

import java.util.Currency;
import java.util.UUID;

import lombok.AllArgsConstructor;
import org.jetbrains.annotations.NotNull;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.SetF;
import ru.yandex.bolts.collection.Tuple2;
import ru.yandex.bolts.collection.impl.ArrayListF;
import ru.yandex.chemodan.app.psbilling.core.balance.BalanceService;
import ru.yandex.chemodan.app.psbilling.core.billing.groups.CheckGroupBillingStatusTask;
import ru.yandex.chemodan.app.psbilling.core.billing.groups.GroupBillingService;
import ru.yandex.chemodan.app.psbilling.core.billing.groups.GroupBillingStatus;
import ru.yandex.chemodan.app.psbilling.core.billing.groups.payment.CardBindingChecker;
import ru.yandex.chemodan.app.psbilling.core.config.featureflags.FeatureFlags;
import ru.yandex.chemodan.app.psbilling.core.dao.cards.CardDao;
import ru.yandex.chemodan.app.psbilling.core.dao.cards.TrustCardBindingDao;
import ru.yandex.chemodan.app.psbilling.core.dao.groups.GroupServicePriceOverrideDao;
import ru.yandex.chemodan.app.psbilling.core.dao.groups.billing.ClientBalanceDao;
import ru.yandex.chemodan.app.psbilling.core.entities.cards.CardBindingStatus;
import ru.yandex.chemodan.app.psbilling.core.entities.cards.CardEntity;
import ru.yandex.chemodan.app.psbilling.core.entities.cards.CardPurpose;
import ru.yandex.chemodan.app.psbilling.core.entities.cards.CardStatus;
import ru.yandex.chemodan.app.psbilling.core.entities.cards.TrustCardBinding;
import ru.yandex.chemodan.app.psbilling.core.entities.groups.BalancePaymentInfo;
import ru.yandex.chemodan.app.psbilling.core.entities.groups.Group;
import ru.yandex.chemodan.app.psbilling.core.entities.groups.GroupProductType;
import ru.yandex.chemodan.app.psbilling.core.entities.groups.GroupService;
import ru.yandex.chemodan.app.psbilling.core.entities.groups.GroupServicePriceOverride;
import ru.yandex.chemodan.app.psbilling.core.entities.groups.GroupType;
import ru.yandex.chemodan.app.psbilling.core.entities.groups.billing.ClientBalanceEntity;
import ru.yandex.chemodan.app.psbilling.core.groups.GroupServicesManager;
import ru.yandex.chemodan.app.psbilling.core.groups.GroupsManager;
import ru.yandex.chemodan.app.psbilling.core.model.RequestInfo;
import ru.yandex.chemodan.app.psbilling.core.products.ExperimentalProductFeature;
import ru.yandex.chemodan.app.psbilling.core.products.GroupProduct;
import ru.yandex.chemodan.app.psbilling.core.products.GroupProductManager;
import ru.yandex.chemodan.app.psbilling.core.products.GroupProductQuery;
import ru.yandex.chemodan.app.psbilling.core.promocodes.PromoCodeService;
import ru.yandex.chemodan.app.psbilling.core.promocodes.model.PromoCodeData;
import ru.yandex.chemodan.app.psbilling.core.promocodes.model.SafePromoCode;
import ru.yandex.chemodan.app.psbilling.core.synchronization.engine.Target;
import ru.yandex.chemodan.app.psbilling.core.tasks.execution.TaskScheduler;
import ru.yandex.chemodan.app.psbilling.core.util.ActionResult;
import ru.yandex.chemodan.app.psbilling.core.util.LanguageService;
import ru.yandex.chemodan.app.psbilling.web.exceptions.WebActionException;
import ru.yandex.chemodan.app.psbilling.web.model.GroupAddonPojo;
import ru.yandex.chemodan.app.psbilling.web.model.GroupAddonsPojo;
import ru.yandex.chemodan.app.psbilling.web.model.GroupPojo;
import ru.yandex.chemodan.app.psbilling.web.model.GroupProductPojo;
import ru.yandex.chemodan.app.psbilling.web.model.GroupProductSetPojo;
import ru.yandex.chemodan.app.psbilling.web.model.GroupServicePojo;
import ru.yandex.chemodan.app.psbilling.web.model.GroupServicesPojo;
import ru.yandex.chemodan.app.psbilling.web.model.GroupTypeApi;
import ru.yandex.chemodan.app.psbilling.web.model.GroupsByPayerResponsePojo;
import ru.yandex.chemodan.app.psbilling.web.model.PaymentCardPojo;
import ru.yandex.chemodan.app.psbilling.web.model.PaymentCardsPojo;
import ru.yandex.chemodan.app.psbilling.web.model.PriceOverridesPojo;
import ru.yandex.chemodan.app.psbilling.web.model.ServiceStatus;
import ru.yandex.chemodan.app.psbilling.web.model.TrustCardBindingFormPojo;
import ru.yandex.chemodan.app.psbilling.web.services.ProductsService;
import ru.yandex.chemodan.balanceclient.model.response.GetBoundPaymentMethodsResponse;
import ru.yandex.chemodan.balanceclient.model.response.GetCardBindingURLResponse;
import ru.yandex.chemodan.util.exception.A3ExceptionWithStatus;
import ru.yandex.chemodan.util.exception.BadRequestException;
import ru.yandex.inside.passport.PassportUid;
import ru.yandex.misc.io.http.HttpStatus;
import ru.yandex.misc.lang.StringUtils;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;

/**
 * @author yashunsky
 */
@AllArgsConstructor
public class GroupActionsHelper {
    private static final Logger logger = LoggerFactory.getLogger(GroupActionsHelper.class);

    private final GroupServicesManager groupServicesManager;
    private final GroupsManager groupsManager;
    private final GroupProductManager groupProductManager;
    private final GroupServicePriceOverrideDao groupServicePriceOverrideDao;
    private final ProductsService productsService;
    private final BalanceService balanceService;
    private final GroupBillingService groupBillingService;
    private final FeatureFlags featureFlags;
    private final ClientBalanceDao clientBalanceDao;
    private final TaskScheduler taskScheduler;
    private final CardDao cardDao;
    private final TrustCardBindingDao trustCardBindingDao;
    private final CardBindingChecker cardBindingChecker;
    private final LanguageService languageService;
    private final PromoCodeService promoCodeService;

    public GroupServicesPojo getGroupServices(Group group, Option<String> languageO, PassportUid uid,
                                              ListF<ExperimentalProductFeature> expFeatures,
                                              Target... targets) {
        return getGroupServices(group, languageO, uid, expFeatures, false, targets);
    }

    public GroupServicesPojo getGroupServices(Group group, Option<String> languageO, PassportUid uid,
                                              ListF<ExperimentalProductFeature> expFeatures, boolean withSubgroups,
                                              Target... targets) {
        String language = languageO.orElseGet(() -> languageService.findUserLanguage(uid));

        MapF<GroupService, GroupProduct> serviceWithProducts = getServiceWithProducts(group, withSubgroups,
                GroupProductType.MAIN, targets);

        MapF<UUID, GroupProductPojo> productPojosById = productsService.mapGroupProductsPojo(
                serviceWithProducts.values(), Option.of(uid), Option.of(group), language,
                expFeatures, false);

        MapF<UUID, ListF<GroupServicePriceOverride>> priceOverrides =
                groupServicePriceOverrideDao.findByGroupServices(serviceWithProducts.keys().map(GroupService::getId));

        ListF<GroupServicePojo> groupServicePojos = new ArrayListF<>();
        for (GroupService groupService : serviceWithProducts.keys()) {
            productPojosById.getO(groupService.getGroupProductId())
                    .map(product -> buildGroupServicePojo(group, product, priceOverrides, groupService,
                            serviceWithProducts.getTs(groupService)))
                    .ifPresent(groupServicePojos::add);
        }

        return new GroupServicesPojo(groupServicePojos);
    }

    public GroupAddonsPojo getGroupAddons(Group group, Option<String> languageO, PassportUid uid,
                                          ListF<ExperimentalProductFeature> expFeatures, Target... targets) {
        return getGroupAddons(group, languageO, uid, expFeatures, true, targets);
    }

    public GroupAddonsPojo getGroupAddons(Group group, Option<String> languageO, PassportUid uid,
                                          ListF<ExperimentalProductFeature> expFeatures, boolean withSubgroups,
                                          Target... targets) {
        String language = languageO.orElseGet(() -> languageService.findUserLanguage(uid));

        MapF<GroupService, GroupProduct> serviceWithProducts = getServiceWithProducts(group, withSubgroups,
                GroupProductType.ADDON, targets);

        MapF<UUID, GroupProductPojo> productPojosById = productsService.mapGroupProductsPojo(
                serviceWithProducts.values(), Option.of(uid), Option.of(group), language,
                expFeatures, false);

        return buildGroupAddonsPojo(serviceWithProducts, productPojosById);
    }

    private GroupAddonsPojo buildGroupAddonsPojo(MapF<GroupService, GroupProduct> serviceWithProducts,
                                                 MapF<UUID, GroupProductPojo> productPojosById) {
        MapF<GroupService, Group> serviceWithGroups = Cf.hashMap();
        ListF<GroupService> groupServices = serviceWithProducts.keys();
        MapF<UUID, ListF<GroupServicePriceOverride>> priceOverrides =
                groupServicePriceOverrideDao.findByGroupServices(groupServices.map(GroupService::getId));

        ListF<GroupAddonPojo> addonPojos = Cf.arrayList();
        serviceWithProducts.forEach((service, product) -> {
            Group group = serviceWithGroups.computeIfAbsent(service, key -> groupsManager.findById(key.getGroupId()));
            productPojosById.getO(product.getId())
                    .map(productPojo -> buildGroupServicePojo(
                            group, productPojo, priceOverrides, service, product))
                    .map(groupServicePojo -> new GroupAddonPojo(
                            groupServicePojo, GroupTypeApi.fromCoreEnum(group.getType()), group.getExternalId()))
                    .ifPresent(addonPojos::add);
        });
        return new GroupAddonsPojo(addonPojos);
    }


    public GroupProductPojo getProductInfo(Option<Group> group, Option<String> languageO, PassportUid uid,
                                           String productId, ListF<ExperimentalProductFeature> expFeatures) {
        GroupProduct groupProduct = groupProductManager.findProduct(productId);

        Option<PassportUid> uidO = Option.of(uid);
        String language = languageO.orElseGet(() -> languageService.findUserLanguage(uidO));

        return productsService.mapGroupProductsPojo(Cf.list(groupProduct), uidO, group, language, expFeatures, false)
                .getOrThrow(groupProduct.getId());
    }

    public Option<Group> getGroup(PassportUid uid, Option<GroupTypeApi> groupTypeO, Option<String> groupIdO) {
        if ((groupTypeO.isEmpty() && groupIdO.isPresent()) || groupTypeO.isPresent() && groupIdO.isEmpty()) {
            throw WebActionException.cons400("parameters group_id and group_type should be filled simultaneously");
        }

        if (groupIdO.isEmpty() || StringUtils.isBlank(groupIdO.get())) {
            return Option.empty();
        }

        return groupsManager.findGroup(uid, groupTypeO.get().toCoreEnum(), groupIdO.get());
    }

    public GroupProductSetPojo getGroupProductSet(
            Option<PassportUid> uid,
            String productSetKey,
            Option<Group> group,
            Option<String> languageO,
            ListF<ExperimentalProductFeature> expFeatures,
            boolean fromLanding,
            ListF<String> productExperiments,
            Option<SafePromoCode> promoCode,
            RequestInfo requestInfo
    ) {
        if (group.isPresent()) {
            if (groupsManager.isPartnerClient(group.get().getId())) {
                return new GroupProductSetPojo(Cf.list(), Option.empty());
            }

            if (fromLanding && featureFlags.getB2bPleaseComeBackEmailEnabled().isEnabled() && uid.isPresent()) {
                taskScheduler.scheduleB2bTariffAcquisitionEmailTask(uid.get());
            }
        }

        Option<PromoCodeData> promoCodeData = promoCode
                .map(p -> promoCodeService.canActivateGroupPromoCode(p, group, uid, requestInfo));

        GroupProductQuery query = new GroupProductQuery(
                productSetKey,
                uid,
                group,
                productExperiments,
                promoCodeData
                        .map(e -> e.getPromoTemplateId().orElseThrow(() -> new BadRequestException("PromoTemplate not" +
                                " found")))
        );

        return productsService.buildGroupProductSetPojo(
                groupProductManager.findGroupProductsWithPromo(query),
                uid,
                group,
                languageO,
                expFeatures,
                promoCodeData
        );
    }

    public GroupPojo getGroupPojo(PassportUid uid, GroupType groupType, String groupId) {
        Group group = groupsManager.findGroupOrThrow(uid, groupType, groupId);
        ListF<GroupService> activeGroupServices =
                groupServicesManager.find(group.getId(), ServiceStatus.statusesToTarget(Option.empty()))
                        .filterNot(GroupService::isHidden);
        ListF<PassportUid> groupPayers = balanceService.findGroupPayers(group);

        GroupBillingStatus actualGroupStatus = groupBillingService.actualGroupBillingStatus(group);
        if (!group.getStatus().equals(actualGroupStatus.getStatus())) {
            logger.info("group status in DB {} does not equal to actual status {}. will update", group.getStatus(),
                    actualGroupStatus.getStatus());
            taskScheduler.schedule(
                    new CheckGroupBillingStatusTask(group.getPaymentInfo().get().getPassportUid()));
        }

        ListF<ClientBalanceEntity> clientBalanceEntities = clientBalanceDao.findByGroup(group);

        return new GroupPojo(group, actualGroupStatus.getStatus(), groupPayers, activeGroupServices,
                groupsManager.calculateGroupNextBillingDate(group),
                actualGroupStatus.getFirstDebtPaymentDeadline(),
                clientBalanceEntities.map(e -> new GroupPojo.BalanceAmount(
                        e.getBalanceAmount(),
                        e.getBalanceCurrency().toString(),
                        e.getBalanceVoidAt()
                )),
                group.getPaymentInfo().map(BalancePaymentInfo::isB2bAutoBillingEnabled).orElse(false)
        );
    }

    public void enableAutoBilling(BalancePaymentInfo paymentInfo) {
        ActionResult result = groupBillingService.setAutoBillingOn(paymentInfo);
        if (!result.isSuccess()) {
            logger.warn(result.getMessage());
            throw new A3ExceptionWithStatus(result.getErrorCode(), result.getError(),
                    HttpStatus.SC_400_BAD_REQUEST);
        }
    }

    public void disableAutoBilling(BalancePaymentInfo paymentInfo) {
        groupBillingService.setAutoBillingOff(paymentInfo);
    }

    public GroupsByPayerResponsePojo getGroupsByPayer(PassportUid uid, String language) {
        ListF<Group> groups = groupsManager.findGroupsByPayer(uid);
        ListF<GroupService> groupServices =
                groupServicesManager.find(groups.map(Group::getId), Target.ENABLED).filter(x -> !x.isHidden());

        MapF<UUID, ListF<GroupService>> groupServicesByGroup = Cf.hashMap();
        for (GroupService groupService : groupServices) {
            UUID groupId = groupService.getGroupId();
            groupServicesByGroup.put(
                    groupId,
                    groupServicesByGroup.getOrElse(groupId, Cf.list()).plus(groupService)
            );
        }

        MapF<UUID, GroupProduct> groupProducts = Cf.toMap(
                groupProductManager.findByIds(
                        groupServices.map(GroupService::getGroupProductId).unique()
                ).map(x -> new Tuple2<>(x.getId(), x))
        );

        ListF<GroupsByPayerResponsePojo.GroupsByPayerPojo> groupPojos = groups.map(
                group -> new GroupsByPayerResponsePojo.GroupsByPayerPojo(
                        Integer.valueOf(group.getExternalId()),
                        groupServicesByGroup.getOrElse(group.getId(), Cf.list()).map(
                                gs -> {
                                    GroupProduct groupProduct = groupProducts.getOrThrow(gs.getGroupProductId());
                                    return new GroupsByPayerResponsePojo.GroupByPayerServicePojo(
                                            gs.getId(),
                                            groupProduct.getTitle().flatMapO(x -> x.findByLangOrDefault(language)),
                                            groupProduct.getPricePerUserInMonth(),
                                            groupProduct.getPriceCurrency().toString(),
                                            groupProduct.getPaymentType().value()
                                    );
                                }
                        )
                )
        );

        return new GroupsByPayerResponsePojo(groupPojos);
    }

    public PaymentCardsPojo getPaymentCards(BalancePaymentInfo paymentInfo) {
        PassportUid uid = paymentInfo.getPassportUid();
        ListF<GetBoundPaymentMethodsResponse> cardsFromBalance =
                Cf.arrayList(balanceService.getBoundPaymentMethods(uid.getUid()));
        MapF<String, CardEntity> savedCards =
                cardDao.findCardsByUid(uid).toMapMappingToKey(CardEntity::getExternalId);
        SetF<String> expiredCards = Cf.hashSet();
        SetF<String> cardsBecameActive = Cf.hashSet();

        // возвращаем карты из списка баланса, кроме тех, что expired в балансе
        ListF<PaymentCardPojo> cards = new ArrayListF<>();
        for (GetBoundPaymentMethodsResponse balanceCard : cardsFromBalance) {
            final String cardExternalId = balanceCard.getPaymentMethodId();
            Option<CardEntity> savedCard = savedCards.getO(cardExternalId);
            Option<CardStatus> status = savedCard.map(CardEntity::getStatus);

            if (balanceCard.isExpired()) {
                if (status.isPresent() && status.get() != CardStatus.EXPIRED) {
                    expiredCards.add(cardExternalId);
                }
                continue;
            }

            if (status.isPresent() && status.get() != CardStatus.ACTIVE) {
                cardsBecameActive.add(cardExternalId);
            }

            boolean isPrimary = (savedCard.isPresent() &&
                    (savedCard.get().getPurpose() == CardPurpose.B2B_PRIMARY));
            cards.add(new PaymentCardPojo(
                    cardExternalId,
                    isPrimary,
                    balanceCard.getAccount(),
                    balanceCard.getPaymentSystem(),
                    balanceCard.getHolder(),
                    balanceCard.getExpirationYear(),
                    balanceCard.getExpirationMonth()
            ));
        }

        // обновляем статусы карт:
        // - которые в балансе expired, а у нас не expired -- выставляем expired
        // - которые в балансе не expired, а у нас disabled -- активируем
        // - которых нет в балансе, а у нас не disabled -- дизейблим

        if (expiredCards.isNotEmpty()) {
            cardDao.updateStatus(expiredCards.toList(), CardStatus.EXPIRED);
        }

        if (cardsBecameActive.isNotEmpty()) {
            cardDao.updateStatus(cardsBecameActive.toList(), CardStatus.ACTIVE);
        }

        SetF<String> cardIdsFromBalance = cardsFromBalance.toMapMappingToKey(
                GetBoundPaymentMethodsResponse::getPaymentMethodId).keySet();
        // чтобы не дизейблить то, что уже задизейблено
        savedCards = savedCards.filter(c -> c._2.getStatus() != CardStatus.DISABLED);
        SetF<String> disabledCards = savedCards.keySet().minus(cardIdsFromBalance);
        if (disabledCards.isNotEmpty()) {
            cardDao.updateStatus(disabledCards.toList(), CardStatus.DISABLED);
        }

        if (paymentInfo.isB2bAutoBillingEnabled()) {
            SetF<String> activePrimaryCards = savedCards
                    .filter(c -> c._2.getStatus() == CardStatus.ACTIVE &&
                            c._2.getPurpose() == CardPurpose.B2B_PRIMARY)
                    .keySet();
            if (activePrimaryCards.size() > 1) {
                logger.warn("{} has several active b2b primary cards", uid.toString());
            }
            for (String activePrimaryCard : activePrimaryCards) {
                if (disabledCards.containsTs(activePrimaryCard) ||
                        expiredCards.containsTs(activePrimaryCard)) {
                    // только что задизейблили основную карту, которая была активной
                    logger.warn("disable autopay for {}: card {} is no longer available or expired",
                            uid.toString(), activePrimaryCard);
                    groupBillingService.setAutoBillingOff(paymentInfo);
                    break;
                }
            }
        }

        return new PaymentCardsPojo(cards);
    }

    public void setCardPurpose(PassportUid uid, String cardExternalId,
                               CardPurpose cardPurpose, BalancePaymentInfo paymentInfo) {
        Option<CardEntity> card = cardDao.updatePurpose(uid, cardExternalId, cardPurpose);
        if (card.isEmpty()) {
            throw new A3ExceptionWithStatus("unknown card id",
                    "Unknown card id", HttpStatus.SC_400_BAD_REQUEST);
        }

        if (cardPurpose.equals(CardPurpose.B2B_PRIMARY) &&
                !paymentInfo.isB2bAutoBillingEnabled()) {
            groupBillingService.setAutoBillingOn(paymentInfo, true);
        }
    }

    public TrustCardBindingFormPojo getCardBindingForm(PassportUid uid, Currency currency, Option<String> theme,
                                                       Option<String> title) {
        TrustCardBinding binding = trustCardBindingDao.insert(TrustCardBindingDao.InsertData.builder()
                .status(CardBindingStatus.INIT)
                .operatorUid(uid)
                .build());

        GetCardBindingURLResponse bindingUrlResponse;
        try {
            bindingUrlResponse = balanceService.getCardBindingURL(uid.getUid(),
                    currency.getCurrencyCode(), binding.getId().toString(), theme, title);
        } catch (Exception e) {
            logger.error("Error getting card binding url from Balance: {}", e);
            trustCardBindingDao.updateStatusIfInNotSuccess(binding.withStatus(CardBindingStatus.ERROR)
                    .withError(Option.of("get binding url error")));
            throw e;
        }

        trustCardBindingDao.setTransactionId(binding.getId(), bindingUrlResponse.getPurchaseToken());

        return new TrustCardBindingFormPojo(bindingUrlResponse.getBindingUrl(), binding.getId().toString());
    }

    public void checkCardBindingStatus(String bindingId) {
        cardBindingChecker.checkCardBindingStatus(UUID.fromString(bindingId));
    }

    private MapF<GroupService, GroupProduct> getServiceWithProducts(Group group, boolean withSubgroups,
                                                                    GroupProductType groupProductType,
                                                                    Target[] targets) {
        return groupServicesManager
                .findWithProducts(group.getId(), withSubgroups, targets)
                .filter((service, product) -> product.getProductType() == groupProductType);
    }

    @NotNull
    private GroupServicePojo buildGroupServicePojo(Group group, GroupProductPojo productPojo,
                                                   MapF<UUID,
                                                           ListF<GroupServicePriceOverride>> priceOverrides,
                                                   GroupService groupService, GroupProduct product) {
        return new GroupServicePojo(
                groupService.getId().toString(),
                ServiceStatus.fromCoreEnum(groupService.getTarget()),
                groupService.getCreatedAt(),
                groupService.getTarget() == Target.ENABLED ?
                        Option.of(groupsManager.calculateGroupNextBillingDate(group)) : Option.empty(),
                productPojo,
                groupService.isSkipTransactionsExport() || product.isFree(),
                priceOverrides.getO(groupService.getId())
                        .map(p -> p.filterNot(GroupServicePriceOverride::isHidden).map(PriceOverridesPojo::new))
                        .getOrNull(),
                groupService.getActualDisabledAt());
    }
}
