package ru.yandex.chemodan.app.psbilling.web.actions.groups;

import lombok.AllArgsConstructor;

import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.psbilling.core.entities.groups.Group;
import ru.yandex.chemodan.app.psbilling.core.groups.GroupsManager;
import ru.yandex.chemodan.app.psbilling.web.model.GroupTypeApi;
import ru.yandex.chemodan.app.psbilling.web.model.GroupsExternalIdsPojo;
import ru.yandex.chemodan.app.psbilling.web.validation.GroupAdminPermissionValidation;
import ru.yandex.chemodan.util.web.Tvm2UidParameterBinder;
import ru.yandex.commune.a3.action.ActionContainer;
import ru.yandex.commune.a3.action.HttpMethod;
import ru.yandex.commune.a3.action.Path;
import ru.yandex.commune.a3.action.parameter.bind.annotation.BindWith;
import ru.yandex.commune.a3.action.parameter.bind.annotation.PathParam;
import ru.yandex.commune.a3.action.parameter.bind.annotation.RequestListParam;
import ru.yandex.commune.a3.action.parameter.bind.annotation.RequestParam;
import ru.yandex.commune.a3.security.UnauthorizedException;
import ru.yandex.inside.passport.PassportUidOrZero;
import ru.yandex.misc.db.masterSlave.MasterSlavePolicy;
import ru.yandex.misc.db.masterSlave.WithMasterSlavePolicy;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;

/**
 * @author yashunsky
 */
@ActionContainer
@AllArgsConstructor
public class GroupAgreementsActions {
    private static final Logger logger = LoggerFactory.getLogger(GroupAgreementsActions.class);
    private final GroupsManager groupsManager;
    private final GroupAdminPermissionValidation groupAdminPermissionValidation;

    @Path(value = "/v1/groups/{groupType}/{groupExternalId}/accept_agreement", methods = HttpMethod.POST)
    @WithMasterSlavePolicy(MasterSlavePolicy.R_M)
    public void acceptAgreement(
            @BindWith(Tvm2UidParameterBinder.class) PassportUidOrZero uid,
            @PathParam("groupType") GroupTypeApi groupType,
            @PathParam("groupExternalId") String groupExternalId,
            @RequestParam("productOwnerCode") String productOwnerCode,
            @RequestParam(value = "clid", required = false) Option<String> clid) {
        logger.info("accept_agreement -> " +
                        "uid: {}, groupType : {}, groupExternalId: {}, productOwnerCode: {}, clid: {}",
                uid, groupType, groupExternalId, productOwnerCode, clid);

        if (!uid.isAuthenticated()) {
            throw new UnauthorizedException("uid isn't specified");
        }

        groupAdminPermissionValidation.check(uid.toUid(), groupExternalId);

        boolean isEdu = groupType.equals(GroupTypeApi.ORGANIZATION)
                && groupsManager.checkOrgFeatures(groupExternalId).isEdu();

        groupsManager.acceptAgreementForGroup(
                uid.toPassportUid(), groupType.toCoreEnum(), groupExternalId, isEdu, productOwnerCode, clid);

        logger.info("accept_agreement -> " +
                "uid: {}, groupType : {}, groupExternalId: {}, productOwnerCode: {}, clid: {}" +
                "completed with result {}", uid, groupType, groupExternalId, productOwnerCode, clid);
    }

    @Path(value = "/v1/groups/{groupType}/filter_by_accepted_agreement", methods = HttpMethod.GET)
    @WithMasterSlavePolicy(MasterSlavePolicy.R_SM)
    public GroupsExternalIdsPojo filterByAcceptedAgreement(
            @PathParam("groupType") GroupTypeApi groupType,
            @RequestListParam("groupIds") ListF<String> groupIds,
            @RequestParam("productOwnerCode") String productOwnerCode) {
        logger.info("filter_by_accepted_agreement -> " +
                        "groupType : {}, groupIds: {}, productOwnerCode: {}",
                groupType, groupIds, productOwnerCode);
        GroupsExternalIdsPojo result =
                new GroupsExternalIdsPojo(groupsManager.filterGroupsByAcceptedAgreement(
                        groupType.toCoreEnum(), groupIds, productOwnerCode).map(Group::getExternalId));
        logger.info("filter_by_accepted_agreement -> " +
                "groupType : {}, groupIds: {}, productOwnerCode: {}" +
                "completed with result {}", groupType, groupIds, productOwnerCode, result);
        return result;
    }

}
