package ru.yandex.chemodan.app.psbilling.web.actions.groups;

import java.math.BigDecimal;
import java.util.UUID;

import lombok.AllArgsConstructor;
import org.apache.commons.codec.digest.DigestUtils;
import org.joda.time.LocalDate;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.psbilling.core.billing.groups.GroupBillingService;
import ru.yandex.chemodan.app.psbilling.core.billing.groups.GroupServiceTransactionsCalculationService;
import ru.yandex.chemodan.app.psbilling.core.billing.groups.Money;
import ru.yandex.chemodan.app.psbilling.core.billing.groups.exceptions.FewMoneyForPaymentException;
import ru.yandex.chemodan.app.psbilling.core.billing.groups.payment.PaymentChecker;
import ru.yandex.chemodan.app.psbilling.core.dao.products.ProductOwnerDao;
import ru.yandex.chemodan.app.psbilling.core.entities.groups.BalancePaymentInfo;
import ru.yandex.chemodan.app.psbilling.core.entities.groups.Group;
import ru.yandex.chemodan.app.psbilling.core.entities.groups.GroupService;
import ru.yandex.chemodan.app.psbilling.core.entities.groups.GroupType;
import ru.yandex.chemodan.app.psbilling.core.entities.products.ProductOwner;
import ru.yandex.chemodan.app.psbilling.core.groups.GroupsManager;
import ru.yandex.chemodan.app.psbilling.web.converter.ConverterToUUID;
import ru.yandex.chemodan.app.psbilling.web.exceptions.WebActionException;
import ru.yandex.chemodan.app.psbilling.web.model.ActivateAddonRequest;
import ru.yandex.chemodan.app.psbilling.web.model.GroupInvoicePojo;
import ru.yandex.chemodan.app.psbilling.web.model.GroupInvoicesPojo;
import ru.yandex.chemodan.app.psbilling.web.model.GroupTypeApi;
import ru.yandex.chemodan.app.psbilling.web.model.PayloadPojo;
import ru.yandex.chemodan.app.psbilling.web.model.PaymentDataListPojo;
import ru.yandex.chemodan.app.psbilling.web.model.PaymentDataPojo;
import ru.yandex.chemodan.app.psbilling.web.model.PhPaymentDataPojo;
import ru.yandex.chemodan.app.psbilling.web.model.TrustPaymentFormPojo;
import ru.yandex.chemodan.app.psbilling.web.model.UrPaymentDataPojo;
import ru.yandex.chemodan.app.psbilling.web.services.PaymentDataService;
import ru.yandex.chemodan.app.psbilling.web.services.SubscriptionService;
import ru.yandex.chemodan.app.psbilling.web.validation.GroupAdminPermissionValidation;
import ru.yandex.chemodan.app.psbilling.web.validation.GroupValidator;
import ru.yandex.chemodan.util.exception.BadRequestException;
import ru.yandex.chemodan.util.web.Tvm2UidParameterBinder;
import ru.yandex.chemodan.web.EmptyPojo;
import ru.yandex.commune.a3.action.ActionContainer;
import ru.yandex.commune.a3.action.HttpMethod;
import ru.yandex.commune.a3.action.Path;
import ru.yandex.commune.a3.action.parameter.bind.BoundByJackson;
import ru.yandex.commune.a3.action.parameter.bind.annotation.BindWith;
import ru.yandex.commune.a3.action.parameter.bind.annotation.PathParam;
import ru.yandex.commune.a3.action.parameter.bind.annotation.RequestListParam;
import ru.yandex.commune.a3.action.parameter.bind.annotation.RequestParam;
import ru.yandex.commune.a3.security.UnauthorizedException;
import ru.yandex.commune.bazinga.BazingaBender;
import ru.yandex.inside.passport.PassportUid;
import ru.yandex.inside.passport.PassportUidOrZero;
import ru.yandex.misc.db.masterSlave.MasterSlavePolicy;
import ru.yandex.misc.db.masterSlave.WithMasterSlavePolicy;
import ru.yandex.misc.lang.StringUtils;
import ru.yandex.misc.lang.Validate;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;

@ActionContainer
@AllArgsConstructor
public class GroupBillingActions {
    private static final Logger logger = LoggerFactory.getLogger(GroupBillingActions.class);
    private final GroupsManager groupsManager;
    private final GroupBillingService groupBillingService;
    private final GroupServiceTransactionsCalculationService groupServiceTransactionsCalculationService;
    private final SubscriptionService subscriptionService;
    private final PaymentDataService paymentDataService;
    private final ProductOwnerDao productOwnerDao;
    private final PaymentChecker paymentChecker;

    private final GroupAdminPermissionValidation groupAdminPermissionValidation;
    private final GroupValidator groupValidator;

    @Path(value = "/v1/groups/organization/{organizationId}/payment_data", methods = HttpMethod.PUT)
    @WithMasterSlavePolicy(MasterSlavePolicy.R_M)
    public void setUrPaymentData(
            @BindWith(Tvm2UidParameterBinder.class) PassportUidOrZero uid,
            @PathParam("organizationId") String organizationId,
            @RequestParam(value = "paymentDataId", required = false) Option<String> paymentDataId,
            @RequestParam("productOwnerCode") String productOwnerCode,
            @RequestParam(value = "clid", required = false) Option<String> clid,
            @BoundByJackson Option<UrPaymentDataPojo> paymentData
    ) {
        setPaymentData(uid, organizationId, paymentDataId, productOwnerCode, clid, paymentData);
    }

    @Path(value = "/v1/groups/organization/{organizationId}/ph/payment_data", methods = HttpMethod.PUT)
    @WithMasterSlavePolicy(MasterSlavePolicy.R_M)
    public void setPhPaymentData(
            @BindWith(Tvm2UidParameterBinder.class) PassportUidOrZero uid,
            @PathParam("organizationId") String organizationId,
            @RequestParam(value = "paymentDataId", required = false) Option<String> paymentDataId,
            @RequestParam("productOwnerCode") String productOwnerCode,
            @RequestParam(value = "clid", required = false) Option<String> clid,
            @BoundByJackson Option<PhPaymentDataPojo> paymentData
    ) {
        setPaymentData(uid, organizationId, paymentDataId, productOwnerCode, clid, paymentData);
    }

    public <T extends PaymentDataPojo> void setPaymentData(PassportUidOrZero uid, String organizationId,
                                                           Option<String> paymentDataId, String productOwnerCode,
                                                           Option<String> clid, Option<T> paymentData) {
        logger.info("set_payment_data -> " +
                        "uid: {}, organizationId : {}, paymentDataId: {}, paymentData: {}, productOwnerCode: {}, " +
                        "clid: {}",
                uid, organizationId, paymentDataId, paymentData, productOwnerCode, clid);
        if (!uid.isAuthenticated()) {
            throw new UnauthorizedException("uid isn't specified");
        }

        groupAdminPermissionValidation.check(uid.toUid(), organizationId);

        Option<ProductOwner> productOwner = productOwnerDao.findByCode(productOwnerCode);
        if (!productOwner.isPresent()) {
            throw new BadRequestException("unknown product owner " + productOwnerCode);
        }

        BalancePaymentInfo balancePaymentInfo = paymentDataService.buildBalancePaymentData(uid.toPassportUid(),
                paymentDataId, paymentData);

        groupsManager.createOrUpdateOrganization(uid.toPassportUid(), organizationId, false, productOwner.get(),
                balancePaymentInfo, clid);
    }

    @Path(value = "/v1/groups/organization/{organizationId}/subscribe_with_payment_data", methods = HttpMethod.POST)
    @WithMasterSlavePolicy(MasterSlavePolicy.R_M)
    public void buyUr(
            @BindWith(Tvm2UidParameterBinder.class) PassportUidOrZero uid,
            @PathParam("organizationId") String organizationId,
            @RequestParam("productId") String productId,
            @RequestParam(value = "paymentDataId", required = false) Option<String> paymentDataId,
            @BoundByJackson Option<UrPaymentDataPojo> paymentData,
            @RequestParam(value = "deduplicationKey", required = false) Option<String> deduplicationKey,
            @RequestParam(value = "clid", required = false) Option<String> clid,
            @RequestListParam(value = "productExperiments", required = false) ListF<String> productExperiments) {
        buy(uid, organizationId, productId, paymentDataId, paymentData, deduplicationKey, clid, productExperiments);
    }

    @Path(value = "/v1/groups/organization/{organizationId}/ph/subscribe_with_payment_data", methods = HttpMethod.POST)
    @WithMasterSlavePolicy(MasterSlavePolicy.R_M)
    public void buyPh(
            @BindWith(Tvm2UidParameterBinder.class) PassportUidOrZero uid,
            @PathParam("organizationId") String organizationId,
            @RequestParam("productId") String productId,
            @RequestParam(value = "paymentDataId", required = false) Option<String> paymentDataId,
            @BoundByJackson Option<PhPaymentDataPojo> paymentData,
            @RequestParam(value = "deduplicationKey", required = false) Option<String> deduplicationKey,
            @RequestParam(value = "clid", required = false) Option<String> clid,
            @RequestListParam(value = "productExperiments", required = false) ListF<String> productExperiments) {
        buy(uid, organizationId, productId, paymentDataId, paymentData, deduplicationKey, clid, productExperiments);
    }

    public <T extends PaymentDataPojo> void buy(PassportUidOrZero uid, String organizationId, String productId,
                                                Option<String> paymentDataId, Option<T> paymentData,
                                                Option<String> deduplicationKey, Option<String> clid,
                                                ListF<String> productExperiments) {
        logger.info("subscribe_with_payment_data -> " +
                        "uid: {}, organizationId: {}, productId: {}, paymentDataId: {}," +
                        "paymentData: {}, clid: {}, productExperiments: {}",
                uid, organizationId, productId, paymentDataId, paymentData, clid, productExperiments);
        if (!uid.isAuthenticated()) {
            throw new UnauthorizedException("uid isn't specified");
        }

        groupAdminPermissionValidation.check(uid.toUid(), organizationId);

        groupValidator.checkOrganizationIsNotPartnerClient(uid.toUid(), organizationId);

        GroupService groupService = subscriptionService.subscribeOrganization(uid.toPassportUid(), organizationId,
                productId, paymentDataId, paymentData, deduplicationKey, clid, productExperiments);
        logger.info("subscribe_with_payment_data -> " +
                        "uid: {}, organizationId : {}, productId: {}, paymentDataId: {}, paymentData: {}, clid: {} " +
                        "completed with result {}",
                uid, organizationId, productId, paymentDataId, paymentData, groupService, clid);
    }

    @Path(value = "/v1/groups/organization/{organizationId}/subscribe", methods = HttpMethod.POST)
    @WithMasterSlavePolicy(MasterSlavePolicy.R_M)
    public void buy(
            @BindWith(Tvm2UidParameterBinder.class) PassportUidOrZero uid,
            @PathParam("organizationId") String organizationId, //connect
            @RequestParam("productId") String productId,
            @RequestParam(value = "deduplicationKey", required = false) Option<String> deduplicationKey) {
        logger.info("buy -> uid: {}, organizationId : {}, productId: {}, deduplicationKey: {}",
                uid, organizationId, productId, deduplicationKey);
        if (!uid.isAuthenticated()) {
            throw new UnauthorizedException("uid isn't specified");
        }

        groupAdminPermissionValidation.check(uid.toUid(), organizationId);

        groupValidator.checkOrganizationIsNotPartnerClient(uid.toUid(), organizationId);

        GroupService groupService = subscriptionService
                .subscribeOrganization(uid.toPassportUid(), organizationId, productId, deduplicationKey, Cf.list());

        logger.info("subscribe -> uid: {}, organizationId : {}, productId: {}, deduplicationKey: {} " +
                "completed with result {}", uid, organizationId, productId, deduplicationKey, groupService);
    }

    @Path(value = "/v1/groups/organization/{organizationId}/subscribe_for_addon", methods = HttpMethod.POST)
    @WithMasterSlavePolicy(MasterSlavePolicy.R_M)
    public void subscribeForAddon(
            @BindWith(Tvm2UidParameterBinder.class) PassportUidOrZero uid,
            @PathParam("organizationId") String organizationId, //connect
            @BoundByJackson ActivateAddonRequest activateAddonRequest,
            @RequestListParam(value = "productExperiments", required = false) ListF<String> productExperiments) {
        logger.info("subscribeForAddon -> uid: {}, organizationId : {}, productExperiments: {}, body: {}, ",
                uid, organizationId, productExperiments, activateAddonRequest);
        if (!uid.isAuthenticated()) {
            throw new UnauthorizedException("uid isn't specified");
        }

        groupAdminPermissionValidation.check(uid.toUid(), organizationId);

        groupValidator.checkOrganizationIsNotPartnerClient(uid.toUid(), organizationId);

        subscriptionService.subscribeOrganizationSubgroup(uid.toPassportUid(),
                organizationId, activateAddonRequest.getSubgroupExternalIds(),
                activateAddonRequest.getSubgroupType().toCoreEnum(), activateAddonRequest.getAddonProductCode(),
                productExperiments);
    }

    @Path(value = "/v1/groups/organization/{organizationId}/unsubscribe", methods = HttpMethod.POST)
    @WithMasterSlavePolicy(MasterSlavePolicy.R_M)
    public void unsubscribe(
            @BindWith(Tvm2UidParameterBinder.class) PassportUidOrZero uid,
            @PathParam("organizationId") String organizationId, //connect
            @RequestParam(value = "serviceId", customConverter = ConverterToUUID.class) UUID serviceId) {
        logger.info("unsubscribe -> uid: {}, organizationId : {}, serviceId: {}",
                uid, organizationId, serviceId);
        if (!uid.isAuthenticated()) {
            throw new UnauthorizedException("uid isn't specified");
        }

        groupAdminPermissionValidation.check(uid.toUid(), organizationId);

        groupValidator.checkOrganizationIsNotPartnerClient(uid.toUid(), organizationId);

        subscriptionService.unsubscribeOrganization(uid.toPassportUid(), organizationId, serviceId);
        logger.info("unsubscribe -> uid: {}, organizationId : {}, serviceId: {} completed",
                uid, organizationId, serviceId);
    }

    @Path(value = "/v1/groups/organization/available_payment_data", methods = HttpMethod.GET)
    @WithMasterSlavePolicy(MasterSlavePolicy.R_SYNC_SM)
    public PaymentDataListPojo getOrganizationPaymentDataV1(
            @BindWith(Tvm2UidParameterBinder.class) PassportUidOrZero uid,
            @RequestParam(value = "groupExternalId", required = false) Option<String> groupExternalIdO) {
        logger.info("getOrganizationPaymentData -> uid: {}, group_external_id: {} ", uid, groupExternalIdO);
        if (!uid.isAuthenticated()) {
            throw new UnauthorizedException("uid isn't specified");
        }

        groupExternalIdO.ifPresent(id ->  groupAdminPermissionValidation.check(uid.toUid(), id));

        MapF<String, PaymentDataPojo> paymentInfoPojos = groupExternalIdO
                .map(groupExternalId -> paymentDataService.getByGroupExternalId(GroupType.ORGANIZATION, groupExternalId))
                .orElseGet(Cf::map);

        if (paymentInfoPojos.isEmpty()) {
            paymentInfoPojos = paymentDataService.getByUid(uid.toUid());
        }
        return new PaymentDataListPojo(paymentInfoPojos);
    }

    @Path(value = "/v2/groups/organization/available_payment_data", methods = HttpMethod.GET)
    @WithMasterSlavePolicy(MasterSlavePolicy.R_SYNC_SM)
    public PaymentDataListPojo getOrganizationPaymentDataV2(
            @BindWith(Tvm2UidParameterBinder.class) PassportUidOrZero uid,
            @RequestParam(value = "groupExternalId", required = false) Option<String> groupExternalIdO) {
        logger.info("getOrganizationPaymentDataV2 -> uid: {}, group_external_id: {} ", uid, groupExternalIdO);

        if (!uid.isAuthenticated()) {
            throw new UnauthorizedException("uid isn't specified");
        }

        groupExternalIdO.ifPresent(id -> groupAdminPermissionValidation.check(uid.toUid(), id));

        return new PaymentDataListPojo(
                groupExternalIdO
                        .map(groupExternalId -> paymentDataService.getByGroupExternalId(GroupType.ORGANIZATION, groupExternalId))
                        .orElseGet(() -> paymentDataService.getByUid(uid.toUid()))
        );
    }

    @Path(value = "/v1/groups/{groupType}/{groupExternalId}/invoices", methods = HttpMethod.GET)
    @WithMasterSlavePolicy(MasterSlavePolicy.R_SM)
    public GroupInvoicesPojo getGroupInvoices(@BindWith(Tvm2UidParameterBinder.class) PassportUidOrZero uid,
                                              @PathParam("groupType") GroupTypeApi groupType,
                                              @PathParam("groupExternalId") String groupId,
                                              @RequestParam("amount") Option<Double> amount,
                                              @RequestParam("currency") Option<String> currency) {
        logger.info("getGroupInvoices: uid: {} group_type: {}, group_external_id: {}, amount: {}",
                uid, groupType, groupId, amount);
        if (!uid.isAuthenticated()) {
            throw new UnauthorizedException("uid isn't specified");
        }

        groupAdminPermissionValidation.check(uid.toUid(), groupId);

        if (amount.isPresent() != currency.isPresent()) {
            throw WebActionException.cons400("The amount and currency are required at the same time");
        }
        Option<Money> money = amount.map(x -> new Money(BigDecimal.valueOf(x), currency.get()));

        Group group = groupsManager.findGroupOrThrow(uid.toPassportUid(), groupType.toCoreEnum(), groupId);

        //make invoice from owner, not uid from request
        try {
            return new GroupInvoicesPojo(
                    groupBillingService.createInvoices(group, money).map(GroupInvoicePojo::new));
        } catch (FewMoneyForPaymentException e) {
            logger.error(e);
            throw WebActionException.cons400("Few money for payment");
        }
    }

    @Path(value = "/v1/groups/transactions/recalculation", methods = HttpMethod.POST)
    @WithMasterSlavePolicy(MasterSlavePolicy.R_M)
    public String restartTransactionsCalculation(
            @RequestParam("startDate") LocalDate startDate,
            @RequestParam("endDate") LocalDate endDate,
            @RequestParam(value = "export", required = false) boolean export
    ) {
        Validate.le(startDate, endDate);
        return groupServiceTransactionsCalculationService.scheduleRecalculation(startDate, endDate, export);
    }

    @Path(value = "/v1/groups/transactions/recalculation", methods = HttpMethod.GET)
    @WithMasterSlavePolicy(MasterSlavePolicy.R_SYNC_SM)
    public String getRecalculationStatus(@RequestParam("jobId") String jobId) {
        return groupServiceTransactionsCalculationService.getRecalculationJob(jobId).
                map(BazingaBender.mapper::serializeJson).map(String::new)
                .orElseThrow(() -> WebActionException.cons404("not_found", "Job not found"));
    }

    @Path(value = "/v1/groups/{groupType}/{groupExternalId}/payment/trust-form", methods = HttpMethod.GET)
    @WithMasterSlavePolicy(MasterSlavePolicy.R_M)
    public TrustPaymentFormPojo getTrustPaymentForm(@PathParam("groupType") GroupTypeApi groupType,
                                                    @PathParam("groupExternalId") String externalGroupId,
                                                    @BindWith(Tvm2UidParameterBinder.class) PassportUidOrZero uid,
                                                    @RequestParam("redirectUrl") String redirectUrl,
                                                    @RequestParam(value = "theme", required = false) Option<String> theme,
                                                    @RequestParam(value = "title", required = false) Option<String> title) {
        logger.info("getTrustPaymentForm: uid: {} group_type: {}, group_external_id: {}",
                uid, groupType, externalGroupId);
        if (!uid.isAuthenticated()) {
            throw new UnauthorizedException("uid isn't specified");
        }

        groupAdminPermissionValidation.check(uid.toUid(), externalGroupId);

        PassportUid passportUid = uid.toPassportUid();
        Group group = groupsManager.findGroupOrThrow(passportUid, groupType.toCoreEnum(), externalGroupId);

        groupValidator.checkIsNotPartnerClient(group);

        return new TrustPaymentFormPojo(groupBillingService.getTrustPaymentFormUrl(passportUid, group, redirectUrl,
                theme, title, Option.empty(), Option.empty(), Option.empty()).getUrl());
    }

    @Path(value = "/v2/groups/{groupType}/{groupExternalId}/payment/trust-form", methods = HttpMethod.POST)
    @WithMasterSlavePolicy(MasterSlavePolicy.R_M)
    public TrustPaymentFormPojo getTrustPaymentFormV2(@PathParam("groupType") GroupTypeApi groupType,
                                                      @PathParam("groupExternalId") String externalGroupId,
                                                      @BindWith(Tvm2UidParameterBinder.class) PassportUidOrZero uid,
                                                      @RequestParam("redirectUrl") String redirectUrl,
                                                      @RequestParam(value = "theme", required = false) Option<String> theme,
                                                      @RequestParam(value = "title", required = false) Option<String> title,
                                                      @RequestParam(value = "amount", required = false) Option<Double> amount,
                                                      @RequestParam(value = "currency", required = false) Option<String> currency,
                                                      @RequestParam(value = "groupProductCode", required = false) Option<String> groupProductCode,
                                                      @BoundByJackson PayloadPojo payloadPojo) {
        logger.info("getTrustPaymentForm: uid: {}, groupType: {}, groupExternalId: {}, redirectUrl: {}, " +
                "theme: {}, title: {}, amount: {}, currency: {}, groupProductCode: {}, payload: {}", uid, groupType,
                externalGroupId, redirectUrl, theme, title, amount, currency, groupProductCode, payloadPojo);
        if (!uid.isAuthenticated()) {
            throw new UnauthorizedException("uid isn't specified");

        }

        groupAdminPermissionValidation.check(uid.toUid(), externalGroupId);

        if (amount.isPresent() != currency.isPresent()) {
            throw WebActionException.cons400("The amount and currency are required at the same time");
        }
        Option<Money> money = amount.map(x -> new Money(BigDecimal.valueOf(x), currency.get()));

        PassportUid passportUid = uid.toPassportUid();
        Group group = groupsManager.findGroupOrThrow(passportUid, groupType.toCoreEnum(), externalGroupId);

        groupValidator.checkIsNotPartnerClient(group);

        try {
            return new TrustPaymentFormPojo(groupBillingService.getTrustPaymentFormUrl(passportUid, group, redirectUrl,
                    theme, title, groupProductCode, money, payloadPojo.getPayload()).getUrl());
        } catch (FewMoneyForPaymentException e) {
            logger.error(e);
            throw WebActionException.cons400("Few money for payment");
        }
    }

    @Path(value = "/v1/payments/request/{requestId}/notify", methods = HttpMethod.POST)
    @WithMasterSlavePolicy(MasterSlavePolicy.R_M)
    public EmptyPojo checkTrustPaymentNotification(@PathParam("requestId") String requestId,
                                                   @RequestParam("transactionId") String transactionId) {
        logger.info("checkTrustPaymentNotification: requestId: {} transactionId: {}", requestId,
                StringUtils.isNotBlank(transactionId) ? DigestUtils.md5Hex(transactionId) : "<empty>");
        paymentChecker.checkPaymentRequestStatus(requestId, transactionId);
        logger.info("checkTrustPaymentNotification completed-> requestId: {}, transactionId: {}", requestId,
                StringUtils.isNotBlank(transactionId) ? DigestUtils.md5Hex(transactionId) : "<empty>");
        return EmptyPojo.INSTANCE;
    }
}
