package ru.yandex.chemodan.app.psbilling.web.actions.groups;

import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;

import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.psbilling.core.entities.groups.Group;
import ru.yandex.chemodan.app.psbilling.core.groups.GroupsManager;
import ru.yandex.chemodan.app.psbilling.core.model.RequestInfo;
import ru.yandex.chemodan.app.psbilling.core.promocodes.PromoCodeService;
import ru.yandex.chemodan.app.psbilling.core.promocodes.model.PromoCodeActivationResult;
import ru.yandex.chemodan.app.psbilling.core.promocodes.model.SafePromoCode;
import ru.yandex.chemodan.app.psbilling.web.converter.ConverterToSafePromoCode;
import ru.yandex.chemodan.app.psbilling.web.converter.ToPromoCodeActivatedRsPojo;
import ru.yandex.chemodan.app.psbilling.web.model.ActivatePromoCodePayload;
import ru.yandex.chemodan.app.psbilling.web.model.GroupTypeApi;
import ru.yandex.chemodan.app.psbilling.web.model.PromoCodeActivatedResponsePojo;
import ru.yandex.chemodan.app.psbilling.web.validation.GroupAdminPermissionValidation;
import ru.yandex.chemodan.util.exception.BadRequestException;
import ru.yandex.chemodan.util.web.Tvm2UidParameterBinder;
import ru.yandex.commune.a3.action.ActionContainer;
import ru.yandex.commune.a3.action.HttpMethod;
import ru.yandex.commune.a3.action.Path;
import ru.yandex.commune.a3.action.parameter.bind.BoundByJackson;
import ru.yandex.commune.a3.action.parameter.bind.annotation.BindWith;
import ru.yandex.commune.a3.action.parameter.bind.annotation.PathParam;
import ru.yandex.commune.a3.action.parameter.bind.annotation.RequestParam;
import ru.yandex.commune.a3.security.UnauthorizedException;
import ru.yandex.inside.passport.PassportUidOrZero;
import ru.yandex.misc.db.masterSlave.MasterSlavePolicy;
import ru.yandex.misc.db.masterSlave.WithMasterSlavePolicy;
import ru.yandex.misc.lang.StringUtils;

@Slf4j
@ActionContainer
@RequiredArgsConstructor
public class GroupPromoCodeActions {

    private final GroupAdminPermissionValidation groupAdminPermissionValidation;
    private final PromoCodeService promoCodeService;
    private final GroupsManager groupsManager;
    private final ToPromoCodeActivatedRsPojo toPromoCodeActivatedRsPojo;

    @Path(value = "/v1/groups/promo_code/{code}/activate", methods = HttpMethod.POST)
    @WithMasterSlavePolicy(MasterSlavePolicy.R_M)
    public PromoCodeActivatedResponsePojo activateGroupPromoCode(
            @BindWith(Tvm2UidParameterBinder.class) PassportUidOrZero uid,
            @PathParam(value = "code", customConverter = ConverterToSafePromoCode.class) SafePromoCode promoCode,
            @RequestParam(value = "groupType", ignoreEmpty = true) GroupTypeApi groupType,
            @RequestParam(value = "groupExternalId", ignoreEmpty = true) String groupExternalId,
            @RequestParam(value = "userAgent", required = false) Option<String> userAgent,
            @RequestParam(value = "userIp", required = false) Option<String> userIp,
            @RequestParam(value = "userDeviceId", required = false) Option<String> userDeviceId,
            @RequestParam(value = "userLoginId", required = false) Option<String> userLoginId
    ) {
        log.info("activateGroupPromoCode: uid {}, promoCode {}, groupType: {}, groupExternalId {}, userAgent {}, " +
                        "userIp {}, userDeviceId {}, userLoginId {}",
                uid,
                promoCode,
                groupType,
                groupExternalId,
                userAgent,
                userIp,
                userDeviceId,
                userLoginId
        );

        if (!uid.isAuthenticated()) {
            throw new UnauthorizedException("uid isn't specified");
        }

        groupAdminPermissionValidation.check(uid.toUid(), groupExternalId);

        Group group = groupsManager.findGroupOrThrow(uid.toUid(), groupType.toCoreEnum(), groupExternalId);
        RequestInfo requestInfo = RequestInfo.cons(userAgent, userIp, userDeviceId, userLoginId);

        PromoCodeActivationResult result = promoCodeService.activateGroupPromoCode(
                promoCode,
                group,
                uid.toUid(), requestInfo
        );

        return toPromoCodeActivatedRsPojo.apply(result);
    }

    @Path(value = "/v1/groups/promo_code/activate", methods = HttpMethod.POST)
    @WithMasterSlavePolicy(MasterSlavePolicy.RW_M)
    public PromoCodeActivatedResponsePojo activateFromBody(
            @BindWith(Tvm2UidParameterBinder.class) PassportUidOrZero uid,
            @BoundByJackson ActivatePromoCodePayload rq,
            @RequestParam(value = "groupType", ignoreEmpty = true) GroupTypeApi groupType,
            @RequestParam(value = "groupExternalId", ignoreEmpty = true) String groupExternalId,
            @RequestParam(value = "userAgent", required = false) Option<String> userAgent,
            @RequestParam(value = "userIp", required = false) Option<String> userIp,
            @RequestParam(value = "userDeviceId", required = false) Option<String> userDeviceId,
            @RequestParam(value = "userLoginId", required = false) Option<String> userLoginId
    ) {
        log.info("activateGroupPromoCode: uid {}, rq: {}, groupType: {}, groupExternalId {}, userAgent {}, " +
                        "userIp {}, userDeviceId {}, userLoginId {}",
                uid,
                rq,
                groupType,
                groupExternalId,
                userAgent,
                userIp,
                userDeviceId,
                userLoginId
        );

        if (!uid.isAuthenticated()) {
            throw new UnauthorizedException("uid isn't specified");
        }

        boolean isBlankPromoCode = Option.ofNullable(rq)
                .map(ActivatePromoCodePayload::getPromoCode)
                .map(SafePromoCode::getOriginalPromoCode)
                .map(StringUtils::isBlank)
                .orElse(true);

        if (isBlankPromoCode) {
            throw new BadRequestException("promo code is required");
        }

        groupAdminPermissionValidation.check(uid.toUid(), groupExternalId);

        Group group = groupsManager.findGroupOrThrow(uid.toUid(), groupType.toCoreEnum(), groupExternalId);
        RequestInfo requestInfo = RequestInfo.cons(userAgent, userIp, userDeviceId, userLoginId);

        PromoCodeActivationResult result = promoCodeService.activateGroupPromoCode(
                rq.getPromoCode(),
                group,
                uid.toUid(), requestInfo
        );

        return toPromoCodeActivatedRsPojo.apply(result);
    }

    @Path(value = "/v1/groups/promo_code/{code}/check", methods = HttpMethod.POST)
    @WithMasterSlavePolicy(MasterSlavePolicy.R_M)
    public void checkPromoCode(
            @BindWith(Tvm2UidParameterBinder.class) PassportUidOrZero uid,
            @PathParam(value = "code", customConverter = ConverterToSafePromoCode.class) SafePromoCode promoCode,
            @RequestParam(value = "userAgent", required = false) Option<String> userAgent,
            @RequestParam(value = "userIp", required = false) Option<String> userIp,
            @RequestParam(value = "userDeviceId", required = false) Option<String> userDeviceId,
            @RequestParam(value = "userLoginId", required = false) Option<String> userLoginId
    ) {
        log.info("checkGroupPromoCode: uid {}, promoCode {}, userAgent {}, userIp {}, userDeviceId {}, userLoginId {}",
                uid, promoCode, userAgent, userIp, userDeviceId, userLoginId);

        if (!uid.isAuthenticated()) {
            throw new UnauthorizedException("uid isn't specified");
        }

        RequestInfo requestInfo = RequestInfo.cons(userAgent, userIp, userDeviceId, userLoginId);

        promoCodeService.canActivateGroupPromoCode(promoCode, Option.empty(), Option.of(uid.toUid()), requestInfo);
    }

    @Path(value = "/v1/groups/promo_code/check", methods = HttpMethod.POST)
    @WithMasterSlavePolicy(MasterSlavePolicy.R_M)
    public void checkPromoCode(
            @BindWith(Tvm2UidParameterBinder.class) PassportUidOrZero uid,
            @BoundByJackson ActivatePromoCodePayload rq,
            @RequestParam(value = "userAgent", required = false) Option<String> userAgent,
            @RequestParam(value = "userIp", required = false) Option<String> userIp,
            @RequestParam(value = "userDeviceId", required = false) Option<String> userDeviceId,
            @RequestParam(value = "userLoginId", required = false) Option<String> userLoginId
    ) {
        log.info("checkGroupPromoCode: uid {}, rq {}, userAgent {}, userIp {}, userDeviceId {}, userLoginId {}",
                uid, rq, userAgent, userIp, userDeviceId, userLoginId);

        if (!uid.isAuthenticated()) {
            throw new UnauthorizedException("uid isn't specified");
        }

        boolean isBlankPromoCode = Option.ofNullable(rq)
                .map(ActivatePromoCodePayload::getPromoCode)
                .map(SafePromoCode::getOriginalPromoCode)
                .map(StringUtils::isBlank)
                .orElse(true);

        if (isBlankPromoCode) {
            throw new BadRequestException("promo code is required");
        }

        RequestInfo requestInfo = RequestInfo.cons(userAgent, userIp, userDeviceId, userLoginId);

        promoCodeService.canActivateGroupPromoCode(rq.getPromoCode(), Option.empty(), Option.of(uid.toUid()),
                requestInfo);
    }

}
