package ru.yandex.chemodan.app.psbilling.web.actions.promos;

import java.io.IOException;
import java.net.InetAddress;
import java.util.Arrays;

import lombok.extern.slf4j.Slf4j;
import org.joda.time.Instant;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.SetF;
import ru.yandex.chemodan.app.psbilling.core.model.RequestInfo;
import ru.yandex.chemodan.app.psbilling.core.promocodes.PromoCodeService;
import ru.yandex.chemodan.app.psbilling.core.promocodes.exception.PromoCodeCantBeActivatedException;
import ru.yandex.chemodan.app.psbilling.core.promocodes.model.PromoCodeActivationFail;
import ru.yandex.chemodan.app.psbilling.core.promocodes.model.PromoCodeActivationResult;
import ru.yandex.chemodan.app.psbilling.core.promocodes.model.PromoCodeActivationType;
import ru.yandex.chemodan.app.psbilling.core.promocodes.model.PromoCodeFailCodeEnum;
import ru.yandex.chemodan.app.psbilling.core.promocodes.model.SafePromoCode;
import ru.yandex.chemodan.app.psbilling.core.promocodes.tasks.GeneratePromoCodeTask;
import ru.yandex.chemodan.app.psbilling.web.converter.ConverterToSafePromoCode;
import ru.yandex.chemodan.app.psbilling.web.model.ActivatePromoCodePayload;
import ru.yandex.chemodan.app.psbilling.web.model.PromoCodeActivatedResponsePojo;
import ru.yandex.chemodan.app.psbilling.web.model.PromoCodeTypeApi;
import ru.yandex.chemodan.boot.admin.idm.IdmAdminRolesService;
import ru.yandex.chemodan.util.auth.YateamAuthUtils;
import ru.yandex.chemodan.util.exception.A3ExceptionWithStatus;
import ru.yandex.chemodan.util.exception.BadRequestException;
import ru.yandex.chemodan.util.web.Tvm2UidParameterBinder;
import ru.yandex.commune.a3.action.ActionContainer;
import ru.yandex.commune.a3.action.HttpMethod;
import ru.yandex.commune.a3.action.Path;
import ru.yandex.commune.a3.action.parameter.bind.BoundByJackson;
import ru.yandex.commune.a3.action.parameter.bind.annotation.BindWith;
import ru.yandex.commune.a3.action.parameter.bind.annotation.PathParam;
import ru.yandex.commune.a3.action.parameter.bind.annotation.RequestParam;
import ru.yandex.commune.a3.action.parameter.bind.annotation.SpecialParam;
import ru.yandex.commune.a3.security.UnauthorizedException;
import ru.yandex.inside.passport.PassportUidOrZero;
import ru.yandex.misc.db.masterSlave.MasterSlavePolicy;
import ru.yandex.misc.db.masterSlave.WithMasterSlavePolicy;
import ru.yandex.misc.io.http.HttpStatus;
import ru.yandex.misc.lang.StringUtils;
import ru.yandex.misc.web.servlet.HttpServletRequestX;

@ActionContainer
@Slf4j
public class PromoCodeActions {
    private final PromoCodeService promoCodeService;
    private final SetF<String> localAddresses;

    public PromoCodeActions(PromoCodeService promoCodeService) {
        this.promoCodeService = promoCodeService;
        this.localAddresses = resolveLocalAddresses().unmodifiable();
    }

    @Path(value = "/v1/promocode/codes/{promo_code}/activate", methods = HttpMethod.POST)
    @WithMasterSlavePolicy(MasterSlavePolicy.RW_M)
    public PromoCodeActivatedResponsePojo activate(
            @BindWith(Tvm2UidParameterBinder.class) PassportUidOrZero uid,
            @PathParam(value = "promo_code", customConverter = ConverterToSafePromoCode.class) SafePromoCode promoCode,
            @RequestParam(value = "userAgent", required = false) Option<String> userAgent,
            @RequestParam(value = "userIp", required = false) Option<String> userIp,
            @RequestParam(value = "userDeviceId", required = false) Option<String> userDeviceId,
            @RequestParam(value = "userLoginId", required = false) Option<String> userLoginId
    ) {
        log.info(
                "activatePromoCodeByRq: uid {}, promoCode {}, userAgent {}, userIp {}, userDeviceId {}, userLoginId {}",
                uid,
                promoCode,
                userAgent,
                userIp,
                userDeviceId,
                userLoginId
        );

        if (!uid.isAuthenticated()) {
            throw new UnauthorizedException("uid isn't specified");
        }
        PromoCodeActivationResult result = promoCodeService.activatePromoCode(
                promoCode,
                uid.toUid(),
                RequestInfo.cons(userAgent, userIp, userDeviceId, userLoginId)
        );

        return convertToPromoCodeActivatedRsPojo(result);
    }

    //secure option that doesn't log promo code to logs
    @Path(value = "/v1/promocode/activate", methods = HttpMethod.POST)
    @WithMasterSlavePolicy(MasterSlavePolicy.RW_M)
    public PromoCodeActivatedResponsePojo activateFromBody(
            @BindWith(Tvm2UidParameterBinder.class) PassportUidOrZero uid,
            @BoundByJackson ActivatePromoCodePayload rq,
            @RequestParam(value = "userAgent", required = false) Option<String> userAgent,
            @RequestParam(value = "userIp", required = false) Option<String> userIp,
            @RequestParam(value = "userDeviceId", required = false) Option<String> userDeviceId,
            @RequestParam(value = "userLoginId", required = false) Option<String> userLoginId
    ) {
        log.info(
                "activatePromoCode: uid {}, userAgent {}, userIp {}, userDeviceId {}, userLoginId {}, rq {}",
                uid,
                userAgent,
                userIp,
                userDeviceId,
                userLoginId,
                rq
        );

        if (!uid.isAuthenticated()) {
            throw new UnauthorizedException("uid isn't specified");
        }

        boolean isBlankPromoCode = Option.ofNullable(rq)
                .map(ActivatePromoCodePayload::getPromoCode)
                .map(SafePromoCode::getOriginalPromoCode)
                .map(StringUtils::isBlank)
                .orElse(true);

        if (isBlankPromoCode) {
            throw new BadRequestException("promo code is required");
        }

        PromoCodeActivationResult result = promoCodeService.activatePromoCode(
                rq.getPromoCode(),
                uid.toUid(),
                RequestInfo.cons(userAgent, userIp, userDeviceId, userLoginId)
        );

        return convertToPromoCodeActivatedRsPojo(result);
    }

    @Path(value = "/v1/promocode/generate", methods = HttpMethod.POST)
    @WithMasterSlavePolicy(MasterSlavePolicy.RW_M)
    public void generate(
            @SpecialParam HttpServletRequestX reqX,
            @RequestParam(value = "type") PromoCodeTypeApi type,
            @RequestParam(value = "productPeriodCode", required = false) String productPeriodCode,
            @RequestParam(value = "promoTemplateCode", required = false) String promoTemplateCode,
            @RequestParam(value = "prefix", required = false) String prefix,
            @RequestParam(value = "code", required = false, customConverter = ConverterToSafePromoCode.class) SafePromoCode code,
            @RequestParam(value = "numToGenerate") int numToGenerate,
            @RequestParam(value = "numActivations", required = false) Integer numActivations,
            @RequestParam(value = "activeFrom", required = false) Instant activeFrom,
            @RequestParam(value = "activeTo", required = false) Instant activeTo,
            @RequestParam(value = "starTrekTicketId") String starTrekTicketId
    ) throws IOException {
        assertAuthorized(reqX, "GENERATE_PROMOCODE");

        String internalLogin = YateamAuthUtils.getLoginFromAttributeO(reqX).orElse("<unresolved>");
        log.info("generatePromoCodes -> " +
                        "login: {}, " +
                        "type: {}, " +
                        "productPeriodCode: {}, " +
                        "promoTemplateCode: {}, " +
                        "prefix: {}, " +
                        "code: {}, " +
                        "numToGenerate: {}, " +
                        "numActivations: {}, " +
                        "activeFrom: {}, " +
                        "activeTo: {}, " +
                        "starTrekTicketId: {}",
                internalLogin, type, productPeriodCode, promoTemplateCode, prefix, code,
                numToGenerate, numActivations, activeFrom, activeTo, starTrekTicketId
        );

        promoCodeService.scheduleGenerateAndPostPromoCodes(
                GeneratePromoCodeTask.Parameters.builder()
                        .type(type.toCoreEnum())
                        .productPeriodCodeO(Option.ofNullable(productPeriodCode))
                        .promoTemplateCodeO(Option.ofNullable(promoTemplateCode))
                        .prefixO(Option.ofNullable(prefix))
                        .codes(Option.ofNullable(code).map(SafePromoCode::getOriginalPromoCode))
                        .numToGenerate(numToGenerate)
                        .numActivations(Option.ofNullable(numActivations))
                        .activeFromO(Option.ofNullable(activeFrom))
                        .activeToO(Option.ofNullable(activeTo))
                        .starTrekTicketId(starTrekTicketId).build()
        );
    }

    private void assertAuthorized(HttpServletRequestX reqX, String idmRole) {
        if (localAddresses.containsTs(reqX.getRemoteAddr())) {
            return;
        }

        YateamAuthUtils.getLoginFromAttributeO(reqX).orElseThrow(
                () -> new A3ExceptionWithStatus("Failed to resolve internal login", HttpStatus.SC_403_FORBIDDEN)
        );
        if (!IdmAdminRolesService.getRoles(reqX).contains(idmRole)) {
            throw new A3ExceptionWithStatus("Missing IDM role: " + idmRole, HttpStatus.SC_403_FORBIDDEN);
        }
    }

    private SetF<String> resolveLocalAddresses() {
        try {
            SetF<String> localAddressesLocal = Cf.hashSet();
            localAddressesLocal.add(InetAddress.getLocalHost().getHostAddress());
//            skip all hosts ifaces
            Arrays.stream(InetAddress.getAllByName("localhost"))
                    .map(InetAddress::getHostAddress)
                    .forEach(localAddressesLocal::add);
            log.info("skip check for ifaces: {}", localAddressesLocal);
            return localAddressesLocal;
        } catch (IOException e) {
            log.warn("Unable to lookup local addresses");
            return Cf.hashSet();
        }
    }

    //перейти на конвертацию как в ToPromoCodeActivatedRsPojo
    @Deprecated
    private PromoCodeActivatedResponsePojo convertToPromoCodeActivatedRsPojo(PromoCodeActivationResult result) {
        if (result.isActivated()) {
            return result.getActivationType()
                    .map(PromoCodeActivationType::getPromoCodeActivationType)
                    .map(type -> new PromoCodeActivatedResponsePojo(true, Option.of(type), Option.empty()))
                    .orElseThrow(() -> new IllegalStateException("Is active promo code without activation type"));
        } else {
            PromoCodeFailCodeEnum failCodeEnum = result.getFail()
                    .map(PromoCodeActivationFail::getCode)
                    .orElseThrow(() -> new IllegalStateException("Is active promo code without error"));

            if (failCodeEnum == PromoCodeFailCodeEnum.ALREADY_USED) {
                throw new A3ExceptionWithStatus(
                        "promocode-already-activated",
                        "Promo Code already activated",
                        HttpStatus.SC_200_OK
                );
            } else {
                throw new PromoCodeCantBeActivatedException();
            }
        }
    }
}
