package ru.yandex.chemodan.app.psbilling.web.actions.users;

import java.util.Objects;
import java.util.UUID;

import lombok.AllArgsConstructor;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.psbilling.core.billing.users.PaymentInfo;
import ru.yandex.chemodan.app.psbilling.core.billing.users.ReceiptHandler;
import ru.yandex.chemodan.app.psbilling.core.billing.users.ReceiptHandlerV1;
import ru.yandex.chemodan.app.psbilling.core.billing.users.ReceiptHandlerV2;
import ru.yandex.chemodan.app.psbilling.core.billing.users.UserBillingService;
import ru.yandex.chemodan.app.psbilling.core.dao.products.ProductOwnerDao;
import ru.yandex.chemodan.app.psbilling.core.dao.users.OrderDao;
import ru.yandex.chemodan.app.psbilling.core.entities.products.ProductOwner;
import ru.yandex.chemodan.app.psbilling.core.entities.users.Order;
import ru.yandex.chemodan.app.psbilling.core.entities.users.OrderStatus;
import ru.yandex.chemodan.app.psbilling.core.entities.users.ReceiptProcessResult;
import ru.yandex.chemodan.app.psbilling.core.entities.users.Refund;
import ru.yandex.chemodan.app.psbilling.core.products.UserProduct;
import ru.yandex.chemodan.app.psbilling.core.tasks.execution.TaskScheduler;
import ru.yandex.chemodan.app.psbilling.core.users.UserService;
import ru.yandex.chemodan.app.psbilling.core.users.UserServiceManager;
import ru.yandex.chemodan.app.psbilling.web.exceptions.WebActionException;
import ru.yandex.chemodan.app.psbilling.web.model.BuyInappProductResponsePojo;
import ru.yandex.chemodan.app.psbilling.web.model.InappOrderPojo;
import ru.yandex.chemodan.app.psbilling.web.model.InappStoreTypeApi;
import ru.yandex.chemodan.app.psbilling.web.model.OrderPojo;
import ru.yandex.chemodan.app.psbilling.web.model.OrderStatusApi;
import ru.yandex.chemodan.app.psbilling.web.model.PaymentFormTypeApi;
import ru.yandex.chemodan.app.psbilling.web.model.ReceiptRequestPojo;
import ru.yandex.chemodan.app.psbilling.web.model.RefundRequestPojo;
import ru.yandex.chemodan.app.psbilling.web.model.ServiceStatusFilter;
import ru.yandex.chemodan.app.psbilling.web.model.SubscribeResponsePojo;
import ru.yandex.chemodan.app.psbilling.web.model.UserServicesPojo;
import ru.yandex.chemodan.app.psbilling.web.model.UserSubscriptionStatusesPojo;
import ru.yandex.chemodan.util.exception.NotFoundException;
import ru.yandex.chemodan.util.web.Tvm2UidParameterBinder;
import ru.yandex.chemodan.web.EmptyPojo;
import ru.yandex.commune.a3.action.ActionContainer;
import ru.yandex.commune.a3.action.HttpMethod;
import ru.yandex.commune.a3.action.Path;
import ru.yandex.commune.a3.action.parameter.bind.BoundByJackson;
import ru.yandex.commune.a3.action.parameter.bind.annotation.BindWith;
import ru.yandex.commune.a3.action.parameter.bind.annotation.PathParam;
import ru.yandex.commune.a3.action.parameter.bind.annotation.RequestParam;
import ru.yandex.commune.a3.security.UnauthorizedException;
import ru.yandex.commune.dynproperties.DynamicProperty;
import ru.yandex.inside.passport.PassportUid;
import ru.yandex.inside.passport.PassportUidOrZero;
import ru.yandex.misc.db.masterSlave.MasterSlavePolicy;
import ru.yandex.misc.db.masterSlave.WithMasterSlavePolicy;
import ru.yandex.misc.lang.StringUtils;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;

@AllArgsConstructor
@ActionContainer
public class UserBillingActions {
    private static final Logger logger = LoggerFactory.getLogger(UserBillingActions.class);
    private final ProductOwnerDao productOwnerDao;
    private final UserBillingService userBillingService;
    private final OrderDao orderDao;
    private final UserServiceManager userServiceManager;
    private final ReceiptHandlerV1 receiptHandlerV1;
    private final ReceiptHandlerV2 receiptHandlerV2;
    private final UserBillingActionsHelper userBillingActionsHelper;
    private final DynamicProperty<Boolean> processInappReceipt =
            new DynamicProperty<>("ps-billing.process-inapp-receipt", true);
    private final TaskScheduler taskScheduler;

    @Path(value = "/v1/users/subscribe", methods = HttpMethod.POST)
    @WithMasterSlavePolicy(MasterSlavePolicy.R_M)
    public SubscribeResponsePojo subscribe(@BindWith(Tvm2UidParameterBinder.class) PassportUidOrZero uid,
                                           @RequestParam("priceId") String priceCode,
                                           @RequestParam(value = "currency", required = false) Option<String> currency,
                                           @RequestParam("domainSuffix") String domainSuffix,
                                           @RequestParam("returnPath") String returnPath,
                                           @RequestParam("formType") PaymentFormTypeApi formType,
                                           @RequestParam("realUserIp") String realUserIp,
                                           @RequestParam("lang") String language,
                                           @RequestParam(value = "disableTrustHeader", required = false) boolean disableTrustHeader,
                                           @RequestParam(value = "loginId") Option<String> loginId,
                                           @RequestParam(value = "useTemplate", required = false) boolean useTemplate) {
        logger.info(
                "subscribe-> uid={}, priceId={}, currency={}, domainSuffix={}, returnPath={}, formType={}," +
                        " realUserIp={}, lang={}, disableTrustHeader={}, loginId={}, useTemplate={}",
                uid, priceCode, currency, domainSuffix, realUserIp, formType, realUserIp, language,
                disableTrustHeader, loginId, useTemplate);
        if (!uid.isAuthenticated()) {
            throw new UnauthorizedException("uid isn't specified");
        }

        PaymentInfo paymentInfo = userBillingService.initPayment(uid.toPassportUid(), realUserIp, domainSuffix,
                returnPath, formType.getTrustFormTemplate(), priceCode, Option.of(language), currency,
                disableTrustHeader, loginId, useTemplate);

        logger.info("Subscribe result: {}", paymentInfo);
        return new SubscribeResponsePojo(paymentInfo.getPaymentUrl(), paymentInfo.getOrder().getId().toString());
    }

    @Path(value = "/v1/users/unsubscribe", methods = HttpMethod.POST)
    @WithMasterSlavePolicy(MasterSlavePolicy.R_M)
    public void unsubscribe(@BindWith(Tvm2UidParameterBinder.class) PassportUidOrZero uid,
                            @RequestParam("serviceId") String serviceId) {
        logger.info("Unsubscribe-> uid={}, userServiceId={}", uid, serviceId);
        if (!uid.isAuthenticated()) {
            throw new UnauthorizedException("uid isn't specified");
        }

        userServiceManager.stopAutoProlong(uid.toPassportUid(), UUID.fromString(serviceId),
                Option.of("pochta_subscription_autoprolong_cancelled"));
        logger.info("Unsubscribe completed");
    }


    @Path(value = "/v1/users/services", methods = HttpMethod.GET)
    @WithMasterSlavePolicy(MasterSlavePolicy.R_SYNC_SM)
    public UserServicesPojo getUserServices(@BindWith(Tvm2UidParameterBinder.class) PassportUidOrZero uid,
                                            @RequestParam(value = "productOwner", required = false) Option<String> productOwnerStr,
                                            @RequestParam(value = "lang", required = false) String language,
                                            @RequestParam(value = "status", required = false) Option<String> statusO) {
        logger.info("getUserServices-> uid={}, productOwner={}, lang={}, status={}",
                uid, productOwnerStr, language, statusO);

        if (!uid.isAuthenticated()) {
            throw new UnauthorizedException("uid isn't specified");
        }

        Option<ProductOwner> productOwner = productOwnerStr.flatMapO(productOwnerDao::findByCode);

        return userBillingActionsHelper.getUserServices(uid, productOwner, language, Cf.list(), ServiceStatusFilter.parseFilters(statusO, ServiceStatusFilter.ENABLED));
    }

    @Path(value = "/v1/orders/{trustOrderId}/notify", methods = HttpMethod.POST)
    @WithMasterSlavePolicy(MasterSlavePolicy.R_M)
    public EmptyPojo notifyOrder(@PathParam("trustOrderId") String trustOrderId,
                                 @RequestParam(value = "trustRefundId", required = false) String trustRefundId) {
        logger.info("notifyOrder-> trustOrderId={}, trustRefundId={}", trustOrderId, trustRefundId);

        if (StringUtils.isNotBlank(trustRefundId)) {
            userBillingService.checkRefund(trustRefundId);
        } else {
            userBillingService.checkOrder(trustOrderId);
        }

        logger.info("notifyOrder completed-> trustOrderId={}, trustRefundId={}", trustOrderId, trustRefundId);
        return EmptyPojo.INSTANCE;
    }

    @Path(methods = HttpMethod.PUT, value = "/v1/users/inapp/receipt")
    @WithMasterSlavePolicy(MasterSlavePolicy.R_M)
    public BuyInappProductResponsePojo buyInappProduct(@BindWith(Tvm2UidParameterBinder.class) PassportUidOrZero uid,
                                                       @RequestParam(value = "priceId", required = false) Option<String> priceId,
                                                       @RequestParam(value = "currency", required = false) String currency,
                                                       @RequestParam(value = "packageName", required = false) String packageName,
                                                       @RequestParam("storeId") InappStoreTypeApi storeType,
                                                       @BoundByJackson ReceiptRequestPojo receipt) {
        logger.info("buyInappProduct-> uid={}, price_id={}, currency={}, package_name={}, store_id={}, receipt" +
                        ".length={}",
                uid, priceId, currency, packageName, storeType, receipt.getReceipt().map(String::length));

        return processReceiptCore(uid, currency, packageName, storeType, receipt, receiptHandlerV1);
    }

    @Path(methods = HttpMethod.PUT, value = "/v1/users/inapp/payment_initialized")
    @WithMasterSlavePolicy(MasterSlavePolicy.R_M)
    public EmptyPojo inappPaymentInitialized(@BindWith(Tvm2UidParameterBinder.class) PassportUidOrZero uid) {
        logger.info("inappPaymentInitialized-> uid={}", uid);
        if (!uid.isAuthenticated()) {
            throw new UnauthorizedException("uid isn't specified");
        }
        taskScheduler.scheduleAbandonedInappCartEmailTask(uid.toPassportUid());
        return EmptyPojo.INSTANCE;
    }

    // отличается от v1 более подробной обработкой ошибок конфликта данных в чеке и переданного uid
    @Path(methods = HttpMethod.PUT, value = "/v2/users/inapp/receipt")
    @WithMasterSlavePolicy(MasterSlavePolicy.R_M)
    public BuyInappProductResponsePojo processReceipt(@BindWith(Tvm2UidParameterBinder.class) PassportUidOrZero uid,
                                                      @RequestParam(value = "currency", required = false) String currency,
                                                      @RequestParam(value = "packageName", required = false) String packageName,
                                                      @RequestParam("storeId") InappStoreTypeApi storeType,
                                                      @BoundByJackson ReceiptRequestPojo receipt) {
        logger.info("processReceipt-> uid={}, currency={}, package_name={}, store_id={}, receipt.length={}",
                uid, currency, packageName, storeType, receipt.getReceipt().map(String::length));

        return processReceiptCore(uid, currency, packageName, storeType, receipt, receiptHandlerV2);
    }

    private BuyInappProductResponsePojo processReceiptCore(PassportUidOrZero uid,
                                                           String currency,
                                                           String packageName,
                                                           InappStoreTypeApi storeType,
                                                           ReceiptRequestPojo receipt,
                                                           ReceiptHandler receiptHandler) {
        if (!uid.isAuthenticated()) {
            throw new UnauthorizedException("uid isn't specified");
        }
        Option<String> receiptO = receipt.getReceipt();
        if (receiptO.isPresent() && StringUtils.isEmpty(receiptO.get())) {
            throw WebActionException.cons400("invalid-receipt", "Empty receipt");
        }

        if (processInappReceipt.get()) {
            ReceiptProcessResult receiptProcessResult = receiptHandler.processInappReceipt(
                    uid.toPassportUid(), storeType.toCoreEnum(), currency, receiptO, packageName);
            return new BuyInappProductResponsePojo(receiptProcessResult.getOrders().map(InappOrderPojo::new),
                    receiptProcessResult.getTrialUsed());
        }

        return new BuyInappProductResponsePojo();
    }

    @Path(value = "/v1/orders/{orderId}", methods = HttpMethod.GET)
    @WithMasterSlavePolicy(MasterSlavePolicy.R_SYNC_SM)
    public OrderPojo getOrder(@BindWith(Tvm2UidParameterBinder.class) PassportUidOrZero uid,
                              @PathParam("orderId") String orderId) {
        logger.info("getOrder-> uid={}, order_id={}", uid, orderId);
        if (!uid.isAuthenticated()) {
            throw new UnauthorizedException("uid isn't specified");
        }

        Option<Order> orderO = orderDao.findByIdO(UUID.fromString(orderId));
        Order order = orderO.orElseThrow(() -> new NotFoundException("order " + orderId + " not found"));

        if (!Objects.equals(order.getUid(), uid.toString())) {
            throw new UnauthorizedException("unauthorized");
        }
        logger.info("found order: {}", order);
        return new OrderPojo(orderId, OrderStatusApi.fromCoreEnum(order.getStatus()));
    }

    @Path(value = "/v1/users/services/{serviceId}/refund", methods = HttpMethod.PUT)
    @WithMasterSlavePolicy(MasterSlavePolicy.R_M)
    public EmptyPojo startRefund(@BindWith(Tvm2UidParameterBinder.class) PassportUidOrZero uid,
                                 @PathParam("serviceId") String serviceId,
                                 @BoundByJackson RefundRequestPojo refundRequest) {
        logger.info("startRefund-> uid={}, serviceId={}, refundRequest={}", uid, serviceId, refundRequest);

        Option<Refund> refund = userBillingService.initRefund(UUID.fromString(serviceId), refundRequest.getReason());
        logger.info("refund created {} for service {}", refund, serviceId);

        return EmptyPojo.INSTANCE;
    }

    @Path(value = "/v1/users/passport/subscriptions_statuses", methods = HttpMethod.GET)
    @WithMasterSlavePolicy(MasterSlavePolicy.R_SYNC_SM)
    public UserSubscriptionStatusesPojo getUserSubscriptionStatuses(@BindWith(Tvm2UidParameterBinder.class) PassportUidOrZero uid) {
        logger.info("getUserSubscriptionStatuses-> uid={}", uid);
        if (!uid.isAuthenticated()) {
            throw new UnauthorizedException("uid isn't specified");
        }

        boolean hasMail360Products = userServiceManager.findEnabled(String.valueOf(uid.getUid()), Option.empty())
                .map(UserService::getUserProduct)
                .stream().anyMatch(UserProduct::isMail360Product);

        boolean diskProEnable = userBillingService.isDiskProEnable(PassportUid.cons(uid.getUid()));

        return new UserSubscriptionStatusesPojo(hasMail360Products, diskProEnable);
    }

    @Path(value = "/v1/users/orders/refresh", methods = HttpMethod.POST)
    @WithMasterSlavePolicy(MasterSlavePolicy.R_M)
    public void checkOrders(@BindWith(Tvm2UidParameterBinder.class) PassportUidOrZero uid) {
        logger.info("checkOrders-> uid={}", uid);
        if (!uid.isAuthenticated()) {
            throw new UnauthorizedException("uid isn't specified");
        }
        ListF<Order> orders = orderDao.findByUid(uid.toUid());
        orders.stream()
                .filter(order -> order.getStatus().equals(OrderStatus.ON_HOLD))
                .forEach(userBillingService::scheduleCheckOrder);
    }
}
