package ru.yandex.chemodan.app.psbilling.web.actions.users;

import lombok.AllArgsConstructor;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.psbilling.web.model.AbstractUserProductPojo;
import ru.yandex.chemodan.app.psbilling.web.model.ProductSetPojo;
import ru.yandex.chemodan.app.psbilling.web.services.ProductsService;
import ru.yandex.chemodan.util.web.Tvm2UidParameterBinder;
import ru.yandex.commune.a3.action.ActionContainer;
import ru.yandex.commune.a3.action.HttpMethod;
import ru.yandex.commune.a3.action.Path;
import ru.yandex.commune.a3.action.parameter.bind.annotation.BindWith;
import ru.yandex.commune.a3.action.parameter.bind.annotation.PathParam;
import ru.yandex.commune.a3.action.parameter.bind.annotation.RequestParam;
import ru.yandex.inside.passport.PassportUidOrZero;
import ru.yandex.misc.db.masterSlave.MasterSlavePolicy;
import ru.yandex.misc.db.masterSlave.WithMasterSlavePolicy;
import ru.yandex.misc.lang.StringUtils;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;

@ActionContainer
@AllArgsConstructor
public class UserProductActions {
    private static final Logger logger = LoggerFactory.getLogger(UserProductActions.class);

    private final ProductsService productsService;

    // старая ручка используется для получения продуктов для веба, не учитывает конфликта продуктов
    @Path(value = "/v1/productsets/{key}/products", methods = HttpMethod.GET)
    @WithMasterSlavePolicy(MasterSlavePolicy.R_SM)
    //при фиксировании региона пользователя политика перекрывается через аннотацию Transactional
    public ProductSetPojo getProductSet(
            @BindWith(Tvm2UidParameterBinder.class) PassportUidOrZero uid,
            @PathParam(value = "key") String productSetKey,
            @RequestParam(value = "lang", required = false) String language) {
        logger.info("getProductSet -> uid: {}, key: {}, lang: {}", uid, productSetKey, language);
        return productsService.getProductSet(uid.toPassportUidO(), productSetKey,
                StringUtils.notBlankO(language), Option.empty(), false, false, Cf.list());
    }

    // отличие от v1 - в рамках линейки у продуктов возвращаются фичи с флагов disabled (по умолчанию)
    // используется мобилами для покупок inapp-продуктов, а также вебом для получения продуктов (они задизейбленные
    // фичи не получают)
    @Path(value = "/v2/productsets/{key}/products", methods = HttpMethod.GET)
    @WithMasterSlavePolicy(MasterSlavePolicy.R_SM)
    //при фиксировании региона пользователя политика перекрывается через аннотацию Transactional
    public ProductSetPojo getProductSetV2(
            @BindWith(Tvm2UidParameterBinder.class) PassportUidOrZero uid,
            @PathParam(value = "key") String productSetKey,
            @RequestParam(value = "lang", required = false) String language,
            @RequestParam(value = "skipDisabledFeatures", required = false) boolean skipDisabledFeatures,
            @RequestParam(value = "currency", required = false) Option<String> currency) {
        logger.info("getProductSet -> uid: {}, key: {}, lang: {}," +
                        "skipDisabledFeatures: {}, currency: {}",
                uid, productSetKey, language, skipDisabledFeatures, currency);

        ProductSetPojo productSet = productsService.getProductSet(uid.toPassportUidO(), productSetKey,
                StringUtils.notBlankO(language), currency, !skipDisabledFeatures, true, Cf.list());
        log(productSet);
        return productSet;
    }

    // отличие от ручки v2 - добавлена поддержка апгрейдов. если подписка уже есть, то она возвращается в отдельном
    // поле ответа,
    // а также возвращаются только те продукты, на которые можно апгрейдить
    @Path(value = "/v3/productsets/{key}/products", methods = HttpMethod.GET)
    @WithMasterSlavePolicy(MasterSlavePolicy.R_SM)
    //при фиксировании региона пользователя политика перекрывается через аннотацию Transactional
    public ProductSetPojo getProductSetV3(
            @BindWith(Tvm2UidParameterBinder.class) PassportUidOrZero uid,
            @PathParam(value = "key") String productSetKey,
            @RequestParam(value = "lang", required = false) String language,
            @RequestParam(value = "packageName", required = false) String packageName,
            @RequestParam(value = "skipDisabledFeatures", required = false) boolean skipDisabledFeatures,
            @RequestParam(value = "promoActivation", required = false) boolean promoActivation,
            @RequestParam(value = "currency", required = false) Option<String> currency,
            @RequestParam(value = "payloadType", required = false) Option<String> promoPayloadType,
            @RequestParam(value = "payloadVersion", required = false) Option<Integer> promoPayloadVersion) {
        logger.info("getProductSet -> uid: {}, key: {}, lang: {}, skipDisabledFeatures: {}, " +
                        "promoActivation: {}, currency: {}, promoPayloadType: {}," +
                        "promoPayloadVersion: {}",
                uid, productSetKey, language, skipDisabledFeatures, promoActivation, currency,
                promoPayloadType, promoPayloadVersion);

        ProductSetPojo productSet = productsService.getProductSetForUpgrade(
                uid.toPassportUidO(), productSetKey, StringUtils.notBlankO(packageName),
                StringUtils.notBlankO(language), currency,
                !skipDisabledFeatures, promoActivation, promoPayloadType, promoPayloadVersion, Cf.list());

        log(productSet);
        return productSet;
    }

    private void log(ProductSetPojo productSet) {
        logger.info("items to buy: {}",
                String.join(", ", productSet.getItems().stream().map(AbstractUserProductPojo::getProductId)
                        .toArray(String[]::new)));
    }
}
