package ru.yandex.chemodan.app.psbilling.web.converter;

import java.util.function.Function;

import lombok.Data;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.joda.time.Instant;

import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.psbilling.core.converter.ToPromoPayloadConverter;
import ru.yandex.chemodan.app.psbilling.core.entities.groups.Group;
import ru.yandex.chemodan.app.psbilling.core.promocodes.model.PromoCodeData;
import ru.yandex.chemodan.app.psbilling.core.promos.groups.AbstractGroupPromoTemplate;
import ru.yandex.chemodan.app.psbilling.core.texts.TextsManager;
import ru.yandex.chemodan.app.psbilling.web.model.PromoPojo;
import ru.yandex.inside.passport.PassportUid;

@Slf4j
@RequiredArgsConstructor
public class ToGroupPromoConverter implements Function<ToGroupPromoConverter.ConvertData, Option<PromoPojo>> {
    private final Function<ToPromoPayloadConverter.ConvertData, Option<String>> toPayloadConverter;
    private final TextsManager textsManager;

    @Override
    public Option<PromoPojo> apply(ConvertData convertData) {
        return convertData.promoO
                .map(promo -> convertToPojo(promo, convertData));
    }

    private PromoPojo convertToPojo(AbstractGroupPromoTemplate promo, ConvertData convertData) {
        return new PromoPojo(
                promo.getCode(),
                getTitle(promo, convertData),
                getAvailableUntil(promo, convertData),
                toPayloadConverter.apply(
                        ToPromoPayloadConverter.ConvertData.cons(
                                promo.getId(),
                                convertData.payloadType,
                                convertData.payloadVersion,
                                convertData.language
                        )
                )
        );
    }

    private Option<String> getTitle(AbstractGroupPromoTemplate promo, ConvertData convertData) {
        return promo.getPromoNameTankerKey()
                .map(textsManager::findTranslation)
                .flatMapO(t -> t.findLanguageTranslation(convertData.language));
    }

    /**
     * Если группа есть, значит считаем когда закончится акция у группы.
     * Если группы нет, значит пришли из лендинга, то пытаемся узнать время окончания у промокода. Если промо код без
     * даты окончания, то берем время окончания акции.
     */
    private Option<Instant> getAvailableUntil(AbstractGroupPromoTemplate promo, ConvertData convertData) {
        return convertData.groupO.isPresent()
                ? promo.canBeUsedUntilDate(convertData.groupO)
                : convertData.promoCode.flatMapO(PromoCodeData::getToDate).orElse(promo::getToDate);
    }

    @Data(staticConstructor = "cons")
    public static class ConvertData {
        private final Option<PassportUid> uidO;
        private final Option<Group> groupO;
        private final Option<AbstractGroupPromoTemplate> promoO;
        private final String language;
        private final Option<PromoCodeData> promoCode;
        private final Option<String> payloadType;
        private final Option<Integer> payloadVersion;
    }
}
