package ru.yandex.chemodan.app.psbilling.web.model;

import java.math.BigDecimal;

import com.fasterxml.jackson.annotation.JsonProperty;
import lombok.AllArgsConstructor;
import lombok.Getter;
import lombok.NoArgsConstructor;

import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.psbilling.core.entities.CustomPeriod;
import ru.yandex.chemodan.app.psbilling.core.products.UserProductPeriod;
import ru.yandex.chemodan.app.psbilling.core.products.UserProductPrice;
import ru.yandex.chemodan.web.JacksonPojo;
import ru.yandex.misc.lang.DefaultObject;

@AllArgsConstructor
@Getter
@NoArgsConstructor
public class ProductPricePojo extends DefaultObject implements JacksonPojo {
    @JsonProperty("price_id")
    private String priceId;
    @JsonProperty("amount")
    private Option<BigDecimal> amount;
    @JsonProperty("currency")
    private Option<String> currency;
    @JsonProperty("period")
    private CustomPeriodApi period;
    @JsonProperty("period_count")
    private int periodCount;
    @JsonProperty("discount_percent")
    private Option<BigDecimal> discountPercent;
    @JsonProperty("original_amount")
    private Option<BigDecimal> originalAmount;
    @JsonProperty("has_active_service")
    private Option<Boolean> hasActiveService;
    @JsonProperty("start_discount")
    private Option<StartDiscount> startDiscount;

    public ProductPricePojo(UserProductPrice price, Option<Boolean> hasActiveService) {
        //для обратной совместимости. считаем что несколько наших цен мапятся для пользователя всегда в одну
        // и в качестве индетификатора отдаем код периода (мапится с продуктов в трасте)
        UserProductPeriod period = price.getPeriod();
        this.priceId = period.getCode();
        // для инаппных продуктов не показываем цену, потому что она не соответвтует действительности, а мобилы берут
        // ее из стора
        this.amount = isInapp(price) ? Option.empty() : Option.of(price.getPrice());
        this.currency = isInapp(price) ? Option.empty() : Option.of(price.getCurrencyCode());
        this.periodCount = period.getPeriod().getValue();
        this.period = CustomPeriodApi.fromCoreEnum(period.getPeriod().getUnit());
        this.discountPercent = price.getDisplayDiscountPercent();
        this.originalAmount = price.getDisplayOriginalPrice();
        this.hasActiveService = hasActiveService; // используется для inapp при показе продуктсета
        if (period.hasStartPeriod()) {
            CustomPeriod startPeriod = period.getStartPeriodDuration().get();
            startDiscount = Option.of(new StartDiscount(
                    new Period(CustomPeriodApi.fromCoreEnum(startPeriod.getUnit()), startPeriod.getValue()),
                    period.getStartPeriodCount().get(),
                    price.getStartPeriodPrice().get()));
        } else {
            startDiscount = Option.empty();
        }
    }

    public ProductPricePojo(UserProductPrice price) {
        this(price, Option.empty());
    }

    private static boolean isInapp(UserProductPrice price) {
        return price.getPeriod().getUserProduct().getBillingType().isInappProduct();
    }

    @AllArgsConstructor
    @Getter
    public static class StartDiscount {
        @JsonProperty("period")
        private final Period period;
        @JsonProperty("periods_count")
        private final Integer periodsCount;
        @JsonProperty("price")
        private final BigDecimal price;
    }

    //extract to own class if will be needed elsewhere
    @AllArgsConstructor
    @Getter
    public static class Period {
        @JsonProperty("unit")
        CustomPeriodApi period;
        @JsonProperty("length")
        Integer periodLength;
    }
}
