package ru.yandex.chemodan.app.psbilling.web.services;

import java.util.Objects;

import lombok.AllArgsConstructor;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.Tuple2;
import ru.yandex.chemodan.app.psbilling.core.balance.BalanceService;
import ru.yandex.chemodan.app.psbilling.core.balance.PaymentData;
import ru.yandex.chemodan.app.psbilling.core.entities.groups.BalancePaymentInfo;
import ru.yandex.chemodan.app.psbilling.core.entities.groups.Group;
import ru.yandex.chemodan.app.psbilling.core.entities.groups.GroupType;
import ru.yandex.chemodan.app.psbilling.core.entities.groups.billing.BalanceClientType;
import ru.yandex.chemodan.app.psbilling.core.groups.GroupsManager;
import ru.yandex.chemodan.app.psbilling.web.exceptions.ActiveContractWithWrongDataException;
import ru.yandex.chemodan.app.psbilling.web.exceptions.WebActionException;
import ru.yandex.chemodan.app.psbilling.web.model.PaymentDataPojo;
import ru.yandex.chemodan.app.psbilling.web.utils.PaymentDataUtils;
import ru.yandex.chemodan.balanceclient.model.response.GetClientContractsResponseItem;
import ru.yandex.commune.util.RetryUtils;
import ru.yandex.inside.passport.PassportUid;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;

@AllArgsConstructor
public class PaymentDataService {
    private static final Logger logger = LoggerFactory.getLogger(SubscriptionService.class);

    private BalanceService balanceService;
    private GroupsManager groupsManager;
    private int createBalanceOfferRetries;

    public MapF<String, PaymentDataPojo> getByGroupExternalId(GroupType groupType, String groupExternalId) {
        return groupsManager
                .findGroupTrusted(groupType, groupExternalId)
                .flatMapO(Group::getPaymentInfoWithUidBackwardCompatibility)
                .map(this::buildPaymentData)
                .orElseGet(Cf::map);
    }

    public MapF<String, PaymentDataPojo> getByUid(PassportUid uid) {
        return balanceService.findClient(uid).map(clientId -> this.buildPaymentData(new BalancePaymentInfo(clientId, uid))).orElseGet(Cf::map);
    }

    private MapF<String, PaymentDataPojo> buildPaymentData(BalancePaymentInfo paymentInfo) {
        long clientId = paymentInfo.getClientId();
        MapF<Long, PaymentData> paymentData = balanceService.findPaymentData(clientId);
        Option<GetClientContractsResponseItem> activeContract = balanceService.getActiveContract(clientId);

        MapF<String, PaymentDataPojo> result = Cf.hashMap();
        for (Tuple2<Long, PaymentData> entry: paymentData.entries()) {
            boolean hasContract = activeContract.isPresent() && entry.get1().equals(activeContract.get().getPersonId());
            result.put(
                    PaymentDataUtils.buildPaymentDataId(entry.get1(), clientId, paymentInfo.getPassportUid()),
                    PaymentDataPojo.of(entry.get2(), hasContract)
            );
        }

        return result;
    }

    public <T extends PaymentDataPojo> BalancePaymentInfo buildBalancePaymentData(PassportUid uid, Option<String> paymentDataId,
                                                      Option<T> paymentData) {
        if (paymentDataId.isPresent()) {
            return buildBalancePaymentData(paymentDataId.get());
        } else if (paymentData.isPresent()) {
            return buildBalancePaymentData(uid, paymentData.get());
        } else {
            throw WebActionException.cons400("payment_data_id OR body with payment data must be supplied");
        }
    }

    private BalancePaymentInfo buildBalancePaymentData(String paymentDataId) {
        PassportUid uidOfPaymentDataOwner = PaymentDataUtils.uidFromPaymentDataId(paymentDataId);
        Option<Long> clientIdFromPaymentDataId = PaymentDataUtils.clientIdFromPaymentDataId(paymentDataId);

        Option<Long> client = clientIdFromPaymentDataId.isPresent() ? clientIdFromPaymentDataId :
                balanceService.findClient(uidOfPaymentDataOwner);

        if (!client.isPresent()) {
            throw WebActionException.cons500("balance client not found");
        }
        Long clientId = client.get();

        Long personId = PaymentDataUtils.personIdFromPaymentDataId(paymentDataId);
        MapF<Long, PaymentData> balanceData = balanceService.findPaymentData(clientId);
        if (!balanceData.containsKeyTs(personId)) {
            throw WebActionException.cons400("Unknown paymentDataId");
        }

        RetryUtils.retryOrThrow(logger, createBalanceOfferRetries,
                () -> createOffer(uidOfPaymentDataOwner, clientId, personId),
                this::notRetryableErrorForOfferCreation);
        return buildGroupPaymentData(uidOfPaymentDataOwner, clientId, balanceData.getTs(personId).getType());
    }

    private BalancePaymentInfo buildBalancePaymentData(PassportUid uid, PaymentDataPojo paymentData) {
        if (!BalanceService.COUNTRY_RUSSIA.toString().equals(paymentData.getCountryCode())) {
            throw WebActionException.cons400(String.format(
                    "only Russia country code supported (%s). Provided: %s",
                    BalanceService.COUNTRY_RUSSIA, paymentData.getCountryCode()));
        }

        Long clientId = balanceService.findOrCreateClient(uid,
                paymentData.getName(), paymentData.getEmail(), paymentData.getPhone());

        RetryUtils.retryOrThrow(logger, createBalanceOfferRetries,
                () -> createOffer(uid, clientId, paymentData.toCorePaymentData()),
                this::notRetryableErrorForOfferCreation);
        return buildGroupPaymentData(uid, clientId, paymentData.getType());
    }

    private boolean notRetryableErrorForOfferCreation(Throwable throwable) {
        return throwable instanceof ActiveContractWithWrongDataException;
    }

    private Long createOffer(PassportUid uid, Long clientId, PaymentData paymentData) {
        Option<GetClientContractsResponseItem> activeContractO = balanceService.getActiveContract(clientId);
        if (activeContractO.isPresent()) {
            GetClientContractsResponseItem activeContract = activeContractO.get();
            PaymentData activeContractPaymentData = balanceService.findPaymentData(clientId).getTs(activeContract.getPersonId());
            if (paymentData.equals(activeContractPaymentData)) {
                logger.info("Found active contract with equals payment data: {}", activeContract);
                return activeContract.getId();
            }

            logger.error("Active contract payment data: {}, request payment data: {}",
                    activeContractPaymentData, paymentData);
            throw new ActiveContractWithWrongDataException("Found active contract with other payment data");
        }

        Long personId = balanceService.createPerson(uid, clientId, paymentData);
        return balanceService.createOffer(uid, clientId, personId);
    }

    private BalancePaymentInfo buildGroupPaymentData(PassportUid uid, Long clientId, String type) {
        return new BalancePaymentInfo(clientId, uid, BalanceClientType.R.fromValue(type));
    }

    private Long createOffer(PassportUid uid, Long clientId, Long personId) {
        Option<GetClientContractsResponseItem> activeContract = balanceService.getActiveContract(clientId);
        if (!activeContract.isPresent()) {
            return balanceService.createOffer(uid, clientId, personId);
        }

        if (Objects.equals(activeContract.get().getPersonId(), personId)) {
            logger.info("Found active contract, will be contract {}", activeContract.get());
            return activeContract.get().getId();
        } else {
            logger.error("Found active contract {} for person, other than {}", activeContract, personId);
            throw new ActiveContractWithWrongDataException("Found active contract for other person in balance");
        }
    }
}
