package ru.yandex.chemodan.app.psbilling.web.services;

import java.math.BigDecimal;
import java.util.UUID;
import java.util.function.Function;

import lombok.AllArgsConstructor;
import org.joda.time.Duration;
import org.joda.time.Instant;
import org.joda.time.Period;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.CollectionF;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.SetF;
import ru.yandex.chemodan.app.psbilling.core.config.Settings;
import ru.yandex.chemodan.app.psbilling.core.config.featureflags.FeatureFlags;
import ru.yandex.chemodan.app.psbilling.core.converter.ToPromoPayloadConverter;
import ru.yandex.chemodan.app.psbilling.core.dao.users.OrderDao;
import ru.yandex.chemodan.app.psbilling.core.entities.groups.Group;
import ru.yandex.chemodan.app.psbilling.core.entities.products.BillingType;
import ru.yandex.chemodan.app.psbilling.core.entities.users.Order;
import ru.yandex.chemodan.app.psbilling.core.entities.users.OrderStatus;
import ru.yandex.chemodan.app.psbilling.core.entities.users.OrderType;
import ru.yandex.chemodan.app.psbilling.core.entities.users.UserInfo;
import ru.yandex.chemodan.app.psbilling.core.groups.TrialService;
import ru.yandex.chemodan.app.psbilling.core.products.ExperimentalProductFeature;
import ru.yandex.chemodan.app.psbilling.core.products.GroupProduct;
import ru.yandex.chemodan.app.psbilling.core.products.LimitedTimeGroupProducts;
import ru.yandex.chemodan.app.psbilling.core.products.LimitedTimeUserProducts;
import ru.yandex.chemodan.app.psbilling.core.products.UserProduct;
import ru.yandex.chemodan.app.psbilling.core.products.UserProductFeature;
import ru.yandex.chemodan.app.psbilling.core.products.UserProductFeatureRegistry;
import ru.yandex.chemodan.app.psbilling.core.products.UserProductManager;
import ru.yandex.chemodan.app.psbilling.core.products.UserProductPeriod;
import ru.yandex.chemodan.app.psbilling.core.products.UserProductPrice;
import ru.yandex.chemodan.app.psbilling.core.promocodes.model.PromoCodeData;
import ru.yandex.chemodan.app.psbilling.core.promos.PromoTemplate;
import ru.yandex.chemodan.app.psbilling.core.tasks.execution.TaskScheduler;
import ru.yandex.chemodan.app.psbilling.core.texts.PredefinedTankerKey;
import ru.yandex.chemodan.app.psbilling.core.texts.TankerTranslation;
import ru.yandex.chemodan.app.psbilling.core.texts.TextsManager;
import ru.yandex.chemodan.app.psbilling.core.users.UserInfoService;
import ru.yandex.chemodan.app.psbilling.core.users.UserService;
import ru.yandex.chemodan.app.psbilling.core.users.UserServiceManager;
import ru.yandex.chemodan.app.psbilling.core.util.LanguageService;
import ru.yandex.chemodan.app.psbilling.web.converter.ToGroupPromoConverter;
import ru.yandex.chemodan.app.psbilling.web.model.BoughtProductPojo;
import ru.yandex.chemodan.app.psbilling.web.model.FeaturePojo;
import ru.yandex.chemodan.app.psbilling.web.model.GroupProductPojo;
import ru.yandex.chemodan.app.psbilling.web.model.GroupProductSetPojo;
import ru.yandex.chemodan.app.psbilling.web.model.OrderStatusApi;
import ru.yandex.chemodan.app.psbilling.web.model.ProductPricePojo;
import ru.yandex.chemodan.app.psbilling.web.model.ProductSetPojo;
import ru.yandex.chemodan.app.psbilling.web.model.ProductWithPricesPojo;
import ru.yandex.chemodan.app.psbilling.web.model.PromoPojo;
import ru.yandex.chemodan.app.psbilling.web.model.UserServicePojo;
import ru.yandex.inside.passport.PassportUid;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;

@AllArgsConstructor
public class ProductsService {
    private static final Logger logger = LoggerFactory.getLogger(ProductsService.class);
    private final UserInfoService userInfoService;
    private final TrialService trialService;
    private final UserProductManager userProductManager;
    private final UserServiceManager userServiceManager;
    private final OrderDao orderDao;
    private final TextsManager textsManager;
    private final TaskScheduler taskScheduler;
    private final FeatureFlags featureFlags;
    private final Settings settings;
    private final LanguageService languageService;
    private final Function<ToPromoPayloadConverter.ConvertData, Option<String>> toPayloadConverter;
    private final Function<ToGroupPromoConverter.ConvertData, Option<PromoPojo>> toGroupPromoConverter;

    public ProductSetPojo getProductSetForUpgrade(Option<PassportUid> uidO, String productSetKey,
                                                  Option<String> packageName,
                                                  Option<String> languageO,
                                                  Option<String> currency,
                                                  boolean addDisabledFeatures,
                                                  boolean promoActivation,
                                                  Option<String> payloadType,
                                                  Option<Integer> payloadVersion,
                                                  ListF<ExperimentalProductFeature> expFeatures) {
        String language = languageO.orElseGet(() -> languageService.findUserLanguage(uidO));
        LimitedTimeUserProducts productSetProducts = userProductManager.findProductSet(productSetKey, uidO);
        Option<String> region = uidO.map(userInfoService::findOrCreateUserInfo).flatMapO(UserInfo::getRegionId);

        ListF<BillingType> billingTypes =
                productSetProducts.getUserProducts().map(UserProduct::getBillingType).stableUnique();
        if (billingTypes.size() > 1) {
            logger.error("Selector found products {} with different billing types {}",
                    productSetProducts.getUserProducts().map(UserProduct::getCode), billingTypes);
            throw new IllegalStateException("Selector found products with different billing types");
        }
        BillingType typeOfProducts = billingTypes.single();

        ListF<Order> ordersOnHold = getOrdersOnHold(uidO).filter(order -> order.getUserServiceId().isPresent());
        for (Order order : ordersOnHold) {
            UserService userService = userServiceManager.findById(order.getUserServiceId().get());
            if (userService.getUserProduct().getBillingType().isInappProduct() == typeOfProducts.isInappProduct()) {
                logger.info("got hold order");
                Option<UserServicePojo> userServicePojo = mapBoughtProduct(userService, language, expFeatures).
                        map(boughtProduct -> new UserServicePojo(userService, OrderStatusApi.ON_HOLD,
                                boughtProduct, mapNextPayment(userService, order)));
                return new ProductSetPojo(Cf.list(), Option.empty(), userServicePojo,
                        userServicePojo.map(UserServicePojo::getOrderStatus), Option.empty());
            }
        }

        ListF<UserService> boughtServices = uidO
                .map(u -> userServiceManager.findEnabled(u.toString(), Option.empty())).orElse(Cf.list());
        SetF<UUID> boughtProducts = boughtServices.map(UserService::getUserProductId).unique();

        if (uidO.isPresent() && featureFlags.getPreventInappBuyIfTrustExists().isEnabledForUid(uidO.get())) {
            //TODO грязный хак убрать как будет возможность в рамках CHEMODAN-82699
            //если к нам пришли с инапным продукт сетом
            //и у нас куплена веб подписка - возвращаем пустой список
            boolean haveBoughWebProduct = boughtServices
                    .map(UserService::getUserProduct)
                    .map(UserProduct::getBillingType)
                    .filter(BillingType.TRUST::equals)
                    .isNotEmpty();
            if (haveBoughWebProduct && typeOfProducts.isInappProduct()) {
                logger.debug("Have web service while requesting inapps nothing to buy");
                Option<UserService> serviceToUpgrade = findServiceToUpgrade(uidO, packageName, productSetProducts,
                        typeOfProducts);
                Option<UserServicePojo> userServicePojo = serviceToUpgrade.flatMapO(serviceToUpgradeFrom ->
                        mapBoughtProduct(serviceToUpgradeFrom, language, expFeatures)
                                .map(p -> new UserServicePojo(serviceToUpgradeFrom, OrderStatusApi.PAID, p,
                                        mapNextPayment(serviceToUpgradeFrom))));

                return new ProductSetPojo(Cf.list(),
                        Option.empty(),
                        userServicePojo,
                        userServicePojo.map(UserServicePojo::getOrderStatus),
                        Option.empty()
                );
            }
        }

        if (boughtServices.isEmpty() || (productSetProducts.getUserProducts().stream().noneMatch(p -> p.conflictsWithAny(boughtProducts)))) {
            if (featureFlags.getPleaseComeBackPromoOnTuning().isEnabled()) {
                if (promoActivation && uidO.isPresent()) {
                    taskScheduler.schedulePleaseComeBackTask(uidO.get());
                }
            }

            // покупок нет, показываем все продукты
            // или же конфликтов нет, предлагаем покупать все
            return mapProductsToProductSet(uidO,
                    productSetProducts, language, region, currency, addDisabledFeatures, Option.empty(),
                    payloadType, payloadVersion, expFeatures);
        }

        Option<UserService> serviceToUpgrade = findServiceToUpgrade(uidO, packageName, productSetProducts,
                typeOfProducts);
        if (!serviceToUpgrade.isPresent()) {
            logger.info("Exists conflicting services, but unable to upgrade");
            return new ProductSetPojo();
        }
        UserService serviceToUpgradeFrom = serviceToUpgrade.get();
        Option<String> upgradeCurrency = serviceToUpgradeFrom.getPrice().map(UserProductPrice::getCurrencyCode);

        logger.debug("showing products to upgrade from service {}", serviceToUpgradeFrom);
        MapF<UserProduct, ListF<UserProductPeriod>> products =
                filterProductsToUpgrade(productSetProducts.getUserProducts(), serviceToUpgradeFrom, region,
                        upgradeCurrency);

        ListF<FeaturePojo> allFeaturesAsDisabled = products.keys()
                .plus(serviceToUpgradeFrom.getUserProduct())
                .flatMap(UserProduct::getFeatures)
                .filterMap(f -> mapFeature(f, language, false, expFeatures))
                .groupBy(FeaturePojo::getCode)
                .mapValues(f -> f.sortedBy(FeaturePojo::getOrderNum).last())
                .values()
                .sortedBy(FeaturePojo::getOrderNum);


        ListF<ProductWithPricesPojo> productWithPricesPojos = products.entries().filterMap(
                e -> mapToProductPojo(e._1, e._2, Option.empty(), allFeaturesAsDisabled, language, region,
                        upgradeCurrency, addDisabledFeatures, Cf.list()));
        Option<UserServicePojo.PaymentPojo> payment = mapNextPayment(serviceToUpgradeFrom);
        return new ProductSetPojo(productWithPricesPojos,
                getAvailableUntilByProducts(productSetProducts.getAvailableUntil(), productWithPricesPojos),
                mapBoughtProduct(serviceToUpgradeFrom, language, expFeatures).map(p -> new UserServicePojo(serviceToUpgradeFrom, OrderStatusApi.PAID, p, payment)),
                Option.of(OrderStatusApi.PAID),
                mapToPromoPojo(uidO, productSetProducts.getPromo(), language, payloadType, payloadVersion)
        );
    }

    private Option<UserService> findServiceToUpgrade(Option<PassportUid> uidO, Option<String> packageName,
                                                     LimitedTimeUserProducts productSetProducts,
                                                     BillingType typeOfProducts) {
        // у нас есть какие-то покупки и они конфликтуют с тем, что мы предлагаем купить.
        // сложную логику по отфильтровыванию разных конфликтующих продуктов из линейки и подбору того,
        // какой сервис апгрейдить не пишем
        return uidO.flatMapO(u -> userServiceManager.findServiceForUpgrade(u.toString(), typeOfProducts, packageName,
                productSetProducts.getUserProducts().map(UserProduct::getId).toLinkedHashSet()));
    }

    //    Отдаем дату только если остались продукты, в которых эта дата на что-то влияет (триал, скидка)
    private Option<Instant> getAvailableUntilByProducts(Option<Instant> availableUntil,
                                                        ListF<ProductWithPricesPojo> items) {
        if (!availableUntil.isPresent()) {
            return availableUntil;
        }

        boolean hasLimitedProduct = items.stream().anyMatch(item -> item.getTrial().isPresent() ||
                item.getPrices().stream().anyMatch(price -> price.getDiscountPercent().isPresent()));

        if (hasLimitedProduct) {
            return availableUntil;
        }

        return Option.empty();
    }

    private MapF<UserProduct, ListF<UserProductPeriod>> filterProductsToUpgrade(
            ListF<UserProduct> userProducts, UserService upgradeFromService,
            Option<String> region, Option<String> currency) {
        UserProductPrice currentPrice = upgradeFromService.getPrice().get();
        UserProductPeriod currentPeriod = currentPrice.getPeriod();
        UserProduct currentProduct = upgradeFromService.getUserProduct();

        MapF<UserProduct, ListF<UserProductPeriod>> products = userProducts
                .filter(product -> product.conflictsWith(upgradeFromService.getUserProductId()))
                .toMap(Cf.linkedHashMap(), x -> x, UserProduct::getProductPeriods)
                .mapValues(periods -> periods
                        //фильтруем периоды, чтобы не предлагать более короткие
                        .filter(p -> isLongerOrEquals(p.getPeriod().toJodaPeriod(),
                                currentPeriod.getPeriod().toJodaPeriod()))

                        //отображаем только периоды с более высокой ценой
                        .filter(p -> p.selectPriceForRegion(settings, region, currency)
                                .map(price -> price.getPrice().compareTo(currentPrice.getPrice()) > 0).orElse(false))

                        //скрываем текущую подписку, если она есть
                        .filterNot(p -> p.getId().equals(currentPeriod.getId()))
                );

        //если в текущем есть объём, то не предлагать тарифы у которых тоже есть объём, но он меньше
        Option<UserProductFeature> currentSpaceFeature =
                currentProduct.getFeatures()
                        .find(feature -> UserProductFeatureRegistry.isMpfsSpaceFeature(feature.getCode()));
        if (currentSpaceFeature.isPresent()) {
            BigDecimal currentProductSpaceAmount = currentSpaceFeature.get().getAmount();
            products = products.filterKeys(
                    p -> p.getFeatures()
                            .find(feature -> UserProductFeatureRegistry.isMpfsSpaceFeature(feature.getCode()))
                            .map(UserProductFeature::getAmount)
                            .map(amount -> amount.compareTo(currentProductSpaceAmount) >= 0)
                            .orElse(false)
            );

            //при одинаковом объёме мы предлагаем тариф если больше цена и больше период
            products = products.mapValuesWithKey((product, periods) -> {
                if (product.getFeatures()
                        .find(feature -> UserProductFeatureRegistry.isMpfsSpaceFeature(feature.getCode()))
                        .map(UserProductFeature::getAmount)
                        .map(amount -> amount.compareTo(currentProductSpaceAmount) == 0)
                        .orElse(false)) {
                    return periods.filter(p -> isLonger(p.getPeriod().toJodaPeriod(),
                            currentPeriod.getPeriod().toJodaPeriod()));
                } else {
                    return periods;
                }
            });
        }

        return products.filterValues(ListF::isNotEmpty);
    }

    public ProductSetPojo getProductSet(Option<PassportUid> uidO, String productSetKey, Option<String> languageO,
                                        Option<String> currency,
                                        boolean addDisabledFeatures, boolean skipOnConflict,
                                        ListF<ExperimentalProductFeature> experimentalFeatures) {
        if (isGotHoldOrders(uidO)) {
            logger.info("got hold orders");
            return new ProductSetPojo();
        }

        LimitedTimeUserProducts limitedTimeProducts = userProductManager.findProductSet(productSetKey, uidO);
        String language = languageO.orElseGet(() -> languageService.findUserLanguage(uidO));
        Option<String> region = uidO.map(userInfoService::findOrCreateUserInfo).flatMapO(UserInfo::getRegionId);

        ListF<UserService> boughtServices = uidO
                .map(u -> userServiceManager.findEnabled(u.toString(), Option.empty())).orElse(Cf.list());
        SetF<UUID> pricesOfActiveServices = boughtServices.filterMap(UserService::getProductPriceId).unique();
        SetF<UUID> boughtProducts = boughtServices.map(UserService::getUserProductId).unique();

        if (skipOnConflict &&
                limitedTimeProducts.getUserProducts().stream().anyMatch(p -> p.conflictsWithAny(boughtProducts))) {
            logger.info("got conflicts");
            return new ProductSetPojo();
        }

        return mapProductsToProductSet(uidO, limitedTimeProducts, language, region, currency, addDisabledFeatures,
                Option.of(pricesOfActiveServices), Option.empty(), Option.empty(), experimentalFeatures);
    }

    private ProductSetPojo mapProductsToProductSet(Option<PassportUid> uidO,
                                                   LimitedTimeUserProducts limitedTimeProducts, String language,
                                                   Option<String> region,
                                                   Option<String> currency,
                                                   boolean addDisabledFeatures,
                                                   Option<SetF<UUID>> pricesOfActiveServices,
                                                   Option<String> payloadType,
                                                   Option<Integer> payloadVersion,
                                                   ListF<ExperimentalProductFeature> expFeatures) {
        ListF<FeaturePojo> allFeaturesAsDisabled = limitedTimeProducts.getUserProducts()
                .flatMap(UserProduct::getFeatures)
                .filterMap(f -> mapFeature(f, language, false, expFeatures))
                .groupBy(FeaturePojo::getCode)
                .mapValues(f -> f.sortedBy(FeaturePojo::getOrderNum).last())
                .values()
                .sortedBy(FeaturePojo::getOrderNum);

        ListF<ProductWithPricesPojo> productWithPricePojos = limitedTimeProducts.getUserProducts().filterMap(
                p -> mapToProductPojo(p, p.getProductPeriods(), pricesOfActiveServices, allFeaturesAsDisabled, language,
                        region, currency, addDisabledFeatures, expFeatures));

        return new ProductSetPojo(productWithPricePojos,
                getAvailableUntilByProducts(limitedTimeProducts.getAvailableUntil(), productWithPricePojos),
                Option.empty(), Option.empty(), mapToPromoPojo(uidO, limitedTimeProducts.getPromo(), language,
                payloadType, payloadVersion));
    }

    public GroupProductSetPojo buildGroupProductSetPojo(LimitedTimeGroupProducts groupProductsWithPromo,
                                                        Option<PassportUid> uid,
                                                        Option<Group> group,
                                                        Option<String> languageO,
                                                        ListF<ExperimentalProductFeature> expFeatures,
                                                        Option<PromoCodeData> promoCode) {
        String language = languageO.orElseGet(() -> languageService.findUserLanguage(uid));

        ListF<GroupProductPojo> groupProductPojos = mapGroupProductsPojo(
                groupProductsWithPromo.getGroupProducts(),
                uid,
                group,
                language,
                expFeatures,
                true
        )
                .values()
                .toList();

        Option<PromoPojo> promoPojos = toGroupPromoConverter.apply(
                ToGroupPromoConverter.ConvertData.cons(
                        uid,
                        group,
                        groupProductsWithPromo.getPromo(),
                        language,
                        promoCode,
                        Option.empty(),
                        Option.empty())
        );

        return new GroupProductSetPojo(groupProductPojos, promoPojos);
    }

    public MapF<UUID, GroupProductPojo> mapGroupProductsPojo(
            CollectionF<GroupProduct> products,
            Option<PassportUid> uid,
            Option<Group> group,
            String language,
            ListF<ExperimentalProductFeature> expFeatures,
            boolean allowMultipleUsageTrial
    ) {
        return products.toMap(
                        Cf.linkedHashMap(),
                        GroupProduct::getId,
                        p -> mapToGroupProductPojo(p, language, uid, group, expFeatures, allowMultipleUsageTrial)
                )
                .filterValues(Option::isPresent)
                .mapValues(Option::get);
    }

    private Option<PromoPojo> mapToPromoPojo(Option<PassportUid> uidO, Option<PromoTemplate> promoO, String language,
                                             Option<String> payloadType, Option<Integer> payloadVersion) {
        return promoO.map(promo ->
                new PromoPojo(
                        promo.getCode(),
                        promo.getTitle(language),
                        promo.canBeUsedUntilDate(uidO),
                        toPayloadConverter.apply(
                                ToPromoPayloadConverter.ConvertData.cons(promo.getId(), payloadType, payloadVersion,
                                        language)
                        )
                )
        );
    }

    private Option<GroupProductPojo> mapToGroupProductPojo(
            GroupProduct product,
            String language,
            Option<PassportUid> uid,
            Option<Group> group,
            ListF<ExperimentalProductFeature> expFeatures,
            boolean allowMultipleUsageTrial
    ) {
        Option<String> titleO = product.getTitle().flatMapO(t -> t.findByLangOrDefault(language));

        if (!titleO.isPresent()) {
            logger.warn("Unable to find translation for title of product {} to language {}, skipping it",
                    product,
                    language);
            return Option.empty();
        }
        String title = titleO.get();

        ListF<FeaturePojo> features = product.getFeatures().filterMap(f -> mapFeature(f, language, true, expFeatures));

        return Option.of(
                new GroupProductPojo(
                        product,
                        title,
                        features,
                        trialDefinition -> trialService.isTrialAvailable(trialDefinition, group, uid,
                                allowMultipleUsageTrial)
                )
        );
    }

    private Option<ProductWithPricesPojo> mapToProductPojo(
            UserProduct product, ListF<UserProductPeriod> periods,
            Option<SetF<UUID>> pricesOfActiveServices,
            ListF<FeaturePojo> allFeaturesAsDisabled, String language,
            Option<String> region,
            Option<String> currency,
            boolean addDisabledFeatures,
            ListF<ExperimentalProductFeature> experimentalFeatures) {
        Option<String> titleO = product.getTitle().flatMapO(t -> t.findByLangOrDefault(language));
        if (!titleO.isPresent()) {
            logger.warn("Unable to find translation for title of product {} to language {}, skipping it", product,
                    language);
            return Option.empty();
        }
        String title = titleO.get();

        ListF<ProductPricePojo> prices = mapPricesCollection(periods, pricesOfActiveServices, region, currency);
        if (prices.isEmpty()) {
            logger.warn("Collection of prices for product {} is empty, skipping product", product);
            return Option.empty();
        }

        ListF<FeaturePojo> features = mapFeatures(
                allFeaturesAsDisabled, product.getFeatures(), language, addDisabledFeatures, experimentalFeatures);

        return Option.of(new ProductWithPricesPojo(product, title, features, prices));
    }

    private ListF<FeaturePojo> mapFeatures(ListF<FeaturePojo> allFeaturesAsDisabled,
                                           ListF<UserProductFeature> productFeatures, String language,
                                           boolean addDisabledFeatures, ListF<ExperimentalProductFeature> expFeatures) {

        if (addDisabledFeatures) {
            MapF<String, FeaturePojo> productFeaturesPojo = productFeatures
                    .filterMap(f -> mapFeature(f, language, true, expFeatures))
                    .toMapMappingToKey(FeaturePojo::getDescription);
            return allFeaturesAsDisabled.map(f -> productFeaturesPojo.getO(f.getDescription()).orElse(f));
        } else {
            return productFeatures.filterMap(f -> mapFeature(f, language, true, expFeatures));
        }
    }

    private ListF<ProductPricePojo> mapPricesCollection(
            ListF<UserProductPeriod> periods, Option<SetF<UUID>> pricesOfActiveServices,
            Option<String> region, Option<String> currency) {
        if (periods.isEmpty()) {
            return Cf.list();
        }
        return periods.filterMap(
                period -> period.selectPriceForRegion(settings, region, currency).map(
                        price -> new ProductPricePojo(price,
                                pricesOfActiveServices.map(p -> p.containsTs(price.getId())))
                )
        );
    }

    private Option<FeaturePojo> mapFeature(UserProductFeature feature, String language, boolean isFeatureEnabled,
                                           ListF<ExperimentalProductFeature> expFeatures) {
        Option<TankerTranslation> description = feature.getDescription();
        Option<TankerTranslation> group = feature.getGroup();
        Option<TankerTranslation> value = feature.getValue();

        Option<ExperimentalProductFeature> expFeature = ExperimentalProductFeature.find(feature.getCode(),
                feature.getAmount());
        if (expFeature.isPresent() && expFeatures.find(f -> f.code.equals(feature.getCode())).isPresent()) {
            logger.info("replacing description for feature {}", feature);
            Option<PredefinedTankerKey> predefinedTankerKey = expFeature.get().predefinedTankerKey;
            if (predefinedTankerKey.isEmpty()) {
                description = Option.empty();
            } else {
                description = Option.of(textsManager.findPredefinedTankerTranslation(predefinedTankerKey.get()));
            }
        }

        if (!description.isPresent()) {
            return Option.empty();
        }

        return Option.of(new FeaturePojo(
                getTranslate(description, language, feature, "description").orElse(""),
                getTranslate(group, language, feature, "group").orElse(""),
                isFeatureEnabled ? getTranslate(value, language, feature, "value").orElse("") : "",
                isFeatureEnabled, feature.getCode(),
                feature.getOrderNum()
        ));
    }

    private Option<String> getTranslate(Option<TankerTranslation> text, String language, UserProductFeature feature,
                                        String name) {
        if (!text.isPresent()) {
            return Option.empty();
        }

        Option<String> translate = text.get().findByLangOrDefault(language);
        if (!translate.isPresent()) {
            logger.warn("Unable to find translation for {} to {} for productFeature: {}", name, language, feature);
        }

        return translate;
    }

    public Option<BoughtProductPojo> mapBoughtProduct(UserService service, String language,
                                                      ListF<ExperimentalProductFeature> expFeatures) {
        UserProduct product = service.getUserProduct();
        Option<String> titleO = product.getTitle().flatMapO(t -> t.findByLangOrDefault(language));
        if (!titleO.isPresent()) {
            logger.warn("Unable to find translation for title of product {} to language {}", product, language);
            return Option.empty();
        }
        String title = titleO.orElse((String) null);

        Option<ProductPricePojo> pricePojo = service.getPrice().map(ProductPricePojo::new);
        ListF<FeaturePojo> featuresPojo =
                product.getFeatures().filterMap(f -> mapFeature(f, language, true, expFeatures));
        if (featuresPojo.isEmpty()) {
            logger.warn("Features collection for product {} is empty", product);
        }

        return Option.of(new BoughtProductPojo(product, title, featuresPojo,
                pricePojo.orElse((ProductPricePojo) null)));
    }

    public Option<UserServicePojo.PaymentPojo> mapNextPayment(UserService userService) {
        Option<Order> order = userServiceManager.findLastPaymentOrder(userService);
        if (order.isPresent()) {
            return mapNextPayment(userService, order.get());
        } else {
            return Option.empty();
        }
    }

    private Option<UserServicePojo.PaymentPojo> mapNextPayment(UserService userService, Order order) {
        if (order.getType() == OrderType.SUBSCRIPTION) {
            Option<UserProductPrice> priceO = userService.getPrice();
            if (priceO.isEmpty()) {
                return Option.empty();
            }
            UserProductPrice price = priceO.get();
            UserProductPeriod period = price.getPeriod();
            if (period.hasStartPeriod() && period.getStartPeriodCount().get() > order.getSubscriptionsCount()) {
                return Option.of(new UserServicePojo.PaymentPojo(price.getStartPeriodPrice().get(),
                        price.getCurrencyCode()));
            }
            return Option.of(new UserServicePojo.PaymentPojo(price.getPrice(), price.getCurrencyCode()));
        }
        return Option.empty();
    }


    private static boolean isLongerOrEquals(Period p1, Period p2) {
        Instant now = Instant.now();
        Duration d1 = p1.toDurationTo(now);
        Duration d2 = p2.toDurationTo(now);
        return d1.isLongerThan(d2) || d1.isEqual(d2);
    }

    private static boolean isLonger(Period p1, Period p2) {
        Instant now = Instant.now();
        Duration d1 = p1.toDurationTo(now);
        Duration d2 = p2.toDurationTo(now);
        return d1.isLongerThan(d2);
    }

    private boolean isGotHoldOrders(Option<PassportUid> uidO) {
        ListF<Order> ordersOnHold = getOrdersOnHold(uidO);
        if (ordersOnHold.isNotEmpty()) {
            logger.info("Exist orders on hold. Can't buy any product until all orders are not in terminal state");
            return true;
        }
        return false;
    }

    private ListF<Order> getOrdersOnHold(Option<PassportUid> uidO) {
        return uidO.map(u -> orderDao.findByUid(u).filter(x -> x.getStatus() == OrderStatus.ON_HOLD)).orElse(Cf.list());
    }
}
