package ru.yandex.chemodan.app.psbilling.web.services;

import java.math.BigDecimal;
import java.util.Currency;
import java.util.List;
import java.util.UUID;

import lombok.Builder;
import lombok.Getter;
import lombok.RequiredArgsConstructor;
import org.springframework.transaction.annotation.Transactional;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.psbilling.core.balance.BalanceService;
import ru.yandex.chemodan.app.psbilling.core.billing.groups.CheckGroupBillingStatusTask;
import ru.yandex.chemodan.app.psbilling.core.billing.groups.ClientBalanceCalculator;
import ru.yandex.chemodan.app.psbilling.core.billing.groups.GroupBillingService;
import ru.yandex.chemodan.app.psbilling.core.dao.groups.GroupServiceDao;
import ru.yandex.chemodan.app.psbilling.core.dao.groups.billing.ClientBalanceDao;
import ru.yandex.chemodan.app.psbilling.core.entities.groups.BalancePaymentInfo;
import ru.yandex.chemodan.app.psbilling.core.entities.groups.Group;
import ru.yandex.chemodan.app.psbilling.core.entities.groups.GroupService;
import ru.yandex.chemodan.app.psbilling.core.entities.groups.GroupType;
import ru.yandex.chemodan.app.psbilling.core.entities.groups.billing.ClientBalanceEntity;
import ru.yandex.chemodan.app.psbilling.core.entities.products.ProductOwner;
import ru.yandex.chemodan.app.psbilling.core.groups.GroupServicesManager;
import ru.yandex.chemodan.app.psbilling.core.groups.GroupsManager;
import ru.yandex.chemodan.app.psbilling.core.groups.SubscriptionContext;
import ru.yandex.chemodan.app.psbilling.core.products.GroupProduct;
import ru.yandex.chemodan.app.psbilling.core.products.GroupProductManager;
import ru.yandex.chemodan.app.psbilling.core.synchronization.engine.Target;
import ru.yandex.chemodan.app.psbilling.core.tasks.execution.TaskScheduler;
import ru.yandex.chemodan.app.psbilling.core.util.ActionResult;
import ru.yandex.chemodan.app.psbilling.web.exceptions.WebActionException;
import ru.yandex.chemodan.app.psbilling.web.model.PaymentDataPojo;
import ru.yandex.chemodan.balanceclient.model.response.GetClientContractsResponseItem;
import ru.yandex.chemodan.util.exception.AccessForbiddenException;
import ru.yandex.chemodan.util.exception.NotFoundException;
import ru.yandex.inside.passport.PassportUid;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;

@RequiredArgsConstructor
public class SubscriptionService {
    private static final Logger logger = LoggerFactory.getLogger(SubscriptionService.class);

    private final BalanceService balanceService;

    private final GroupsManager groupsManager;
    private final GroupServicesManager groupServicesManager;
    private final GroupBillingService groupBillingService;
    private final GroupServiceDao groupServiceDao;
    private final GroupProductManager groupProductManager;
    private final ClientBalanceDao clientBalanceDao;
    private final ClientBalanceCalculator clientBalanceCalculator;
    private final PaymentDataService paymentDataService;
    private final TaskScheduler taskScheduler;

    public <T extends PaymentDataPojo> GroupService subscribeOrganization(PassportUid uid, String groupExternalId,
                                                                          String productId,
                                                                          Option<String> paymentDataId,
                                                                          Option<T> paymentData,
                                                                          Option<String> deduplicationKey,
                                                                          Option<String> clid,
                                                                          ListF<String> productExperiments) {
        return subscribeOrganization(uid, groupExternalId, productId, paymentDataId, paymentData, deduplicationKey, clid,
                SubscriptionCheckRightConfig.builder().skipValidateClientBalance(false).skipAvailableProductValidation(false).build(), productExperiments);
    }

    public <T extends PaymentDataPojo> GroupService subscribeOrganization(PassportUid uid, String groupExternalId,
                                                                          String productId,
                                                                          Option<String> paymentDataId,
                                                                          Option<T> paymentData,
                                                                          Option<String> deduplicationKey,
                                                                          Option<String> clid,
                                                                          SubscriptionCheckRightConfig checkConfig,
                                                                          ListF<String> productExperiments) {
        GroupsManager.GroupFeatures features = groupsManager.checkOrgFeatures(groupExternalId);

        if ((paymentData.isPresent() && paymentDataId.isPresent())
                || (!paymentData.isPresent() && !paymentDataId.isPresent())) {
            throw WebActionException.cons400("Only payment_data_id OR body with payment data must be supplied");
        }
        GroupProduct product = groupProductManager.findProduct(productId);
        ProductOwner productOwner = product.getUserProduct().getProductOwner();

        BalancePaymentInfo balancePaymentInfo = paymentDataService.buildBalancePaymentData(uid, paymentDataId,
                paymentData);
        Group group = groupsManager.createOrUpdateOrganization(uid, groupExternalId, features.isEdu(), productOwner,
                balancePaymentInfo, clid);

        SubscriptionContext context = new SubscriptionContext(
                product,
                group,
                uid,
                deduplicationKey,
                features.isEdu(),
                new SubscriptionContext.UsedPromoData(productExperiments)
        );

        GroupService groupService = createGroupService(checkConfig, productExperiments, context);
        ActionResult actionResult = groupBillingService.setAutoBillingOn(group.getPaymentInfo().get());
        if (!actionResult.isSuccess()) {
            logger.info(actionResult.getMessage());
        }

        return groupService;
    }

    public GroupService subscribeOrganization(PassportUid uid, String groupExternalId, String productCode,
                                              Option<String> deduplicationKey, ListF<String> productExperiments) {
        return subscribeOrganization(
                uid, groupExternalId, productCode, deduplicationKey,
                SubscriptionCheckRightConfig.builder().skipValidateClientBalance(false).skipAvailableProductValidation(false).build(), productExperiments
        );
    }

    public GroupService subscribeOrganization(PassportUid uid, String groupExternalId, String productCode,
                                              Option<String> deduplicationKey,
                                              SubscriptionCheckRightConfig checkConfig, ListF<String> productExperiments) {
        GroupsManager.GroupFeatures features = groupsManager.checkOrgFeatures(groupExternalId);
        GroupProduct product = groupProductManager.findProduct(productCode);
        ProductOwner productOwner = product.getUserProduct().getProductOwner();
        Group group = findGroupWithValidPaymentData(uid, groupExternalId, productOwner);


        SubscriptionContext context = new SubscriptionContext(
                product,
                group,
                uid,
                deduplicationKey,
                features.isEdu(),
                new SubscriptionContext.UsedPromoData(productExperiments)
        );

        return createGroupService(checkConfig, productExperiments, context);
    }

    @Transactional
    public ListF<GroupService> subscribeOrganizationSubgroup(PassportUid uid, String parentGroupExternalId,
                                                             ListF<String> subgroupExternalIds,
                                                             GroupType subgroupType, String productCode,
                                                             ListF<String> productExperiments) {
        SubscriptionCheckRightConfig checkConfig = SubscriptionCheckRightConfig.builder()
                .skipValidateClientBalance(false)
                .skipAvailableProductValidation(false)
                .build();
        if (!subgroupType.isSubgroup())
            throw new IllegalArgumentException("Illegal subgroup type " + subgroupType);
        GroupsManager.GroupFeatures features = groupsManager.checkOrgFeatures(parentGroupExternalId);
        Group parentGroup = groupsManager.findGroupOrThrow(uid, GroupType.ORGANIZATION, parentGroupExternalId);
        GroupProduct product = groupProductManager.findProduct(productCode);
        List<Group> subgroups = groupsManager.createSubgroupIfNotExists(parentGroup, subgroupType,
                subgroupExternalIds);
        ListF<GroupService> result = Cf.arrayList();
        for (Group subgroup : subgroups) {
            SubscriptionContext context = new SubscriptionContext(
                    product,
                    subgroup,
                    uid,
                    Option.empty(),
                    features.isEdu(),
                    Option.of(new SubscriptionContext.UsedPromoData(productExperiments)),
                    false
            );
            result.add(createGroupService(checkConfig, productExperiments, context));
        }
        return result;
    }

    private GroupService createGroupService(SubscriptionCheckRightConfig checkConfig,
                                            ListF<String> productExperiments,
                                            SubscriptionContext context) {
        GroupProduct product = context.getProduct();
        Group group = context.getGroup();
        groupProductManager.validateAvailableProduct(context.getUid(), group, product, productExperiments);
        if (product.isPrepaid()
                && product.getTrialDefinitionId().isEmpty()
                && !checkConfig.isSkipValidateClientBalance()) {
            checkClientBalanceForPrepaidProduct(group, product);
        }

        return groupServicesManager.createGroupService(context);
    }

    private void checkClientBalanceForPrepaidProduct(Group group, GroupProduct product) {
        BalancePaymentInfo paymentInfo = group.getPaymentInfo().orElseThrow();
        Currency currency = product.getPriceCurrency();

        Option<ClientBalanceEntity> clientBalance = clientBalanceDao.find(paymentInfo.getClientId(), currency);
        if (clientBalance.map(x -> x.getBalanceAmount().compareTo(BigDecimal.ZERO) <= 0).orElse(true)) {
            ClientBalanceEntity newClientBalance = clientBalanceCalculator.updateClientBalance(
                    paymentInfo.getClientId(),
                    currency
            );

            if (newClientBalance.getBalanceAmount().compareTo(BigDecimal.ZERO) <= 0) {
                throw WebActionException.cons400("Client balance not found or less than 0");
            }
        }
    }


    public void unsubscribeOrganization(PassportUid uid, String groupExternalId, UUID serviceId) {
        Group group = groupsManager.findGroupOrThrow(uid, GroupType.ORGANIZATION, groupExternalId);
        GroupService service = checkAffiliation(serviceId, group, true);

        groupServicesManager.disableGroupService(service);

        if (group.getPaymentInfo().isPresent()) {
            taskScheduler.schedule(new CheckGroupBillingStatusTask(group.getPaymentInfo().get().getPassportUid()));
        } else {
            logger.info("Not found payment info for group {}", group);
        }
    }

    private GroupService checkAffiliation(UUID serviceId, Group group) {
        return checkAffiliation(serviceId, group, false);
    }

    private GroupService checkAffiliation(UUID serviceId, Group group, boolean withSubgroups) {
        Option<GroupService> serviceO = groupServiceDao.findByIdO(serviceId);
        if (!serviceO.isPresent()) {
            throw new NotFoundException("Group service '" + serviceId + "' not found");
        }
        GroupService service = serviceO.get();
        if (service.getGroupId().equals(group.getId())) {
            return service;
        }
        if (withSubgroups && group.getChildGroups().exists(g -> g.getId().equals(service.getGroupId()))) {
            return service;
        }
        throw new AccessForbiddenException("you are not authorized for service");
    }

    @Transactional
    public GroupService makeFreeSubscription(String groupExternalId, String productCode) {
        Group group = groupsManager.findGroupTrustedOrThrow(GroupType.ORGANIZATION, groupExternalId);
        GroupService groupService = subscribeOrganization(
                group.getOwnerUid(),
                groupExternalId,
                productCode,
                Option.empty(),
                SubscriptionCheckRightConfig.builder()
                        .skipValidateClientBalance(true)
                        .skipAvailableProductValidation(true).build(),
                Cf.list()
        );
        return groupServiceDao.updateSkipTransactionsExport(groupService.getId(), true);
    }

    public void revokeGroupSubscription(String groupExternalId, Option<UUID> serviceId) {
        Group group = groupsManager.findGroupTrustedOrThrow(GroupType.ORGANIZATION, groupExternalId);

        if (serviceId.isPresent()) {
            GroupService service = checkAffiliation(serviceId.get(), group);
            groupServicesManager.disableGroupService(service);
        }
        else {
            ListF<GroupService> services =
                    groupServiceDao.find(group.getId(), Target.ENABLED);

            for (GroupService activeService: services) {
                UUID productId = activeService.getGroupProductId();
                Option<UUID> defaultProductId = groupProductManager.findById(productId)
                        .getUserProduct().getProductOwner().getDefaultGroupProductId();
                if (defaultProductId.isEmpty() || !defaultProductId.get().equals(productId)) {
                    groupServicesManager.disableGroupService(activeService);
                }
            }
        }

        if (group.getPaymentInfo().isPresent()) {
            taskScheduler.schedule(new CheckGroupBillingStatusTask(group.getPaymentInfo().get().getPassportUid()));
        } else {
            logger.info("Not found payment info for group {}", group);
        }
    }

    private Group findGroupWithValidPaymentData(PassportUid uid, String groupExternalId,
                                                ProductOwner productOwner) {
        GroupsManager.GroupFeatures features = groupsManager.checkOrgFeatures(groupExternalId);
        Group group = groupsManager.findGroupOrThrow(uid, GroupType.ORGANIZATION, groupExternalId);
        Long clientId = group.getPaymentInfo().map(BalancePaymentInfo::getClientId).orElseThrow(
                () -> new NotFoundException(GroupType.ORGANIZATION + " '" + groupExternalId + "' hasn't payment data"));

        // check if this contract has active contract, if not - we should raise error to reaccept offer
        Option<GetClientContractsResponseItem> activeContractO = balanceService.getActiveContract(clientId);
        activeContractO.orElseThrow(() -> new NotFoundException(
                "Active offer for " + GroupType.ORGANIZATION + " '" + groupExternalId + "' not found"));

        //TODO check debts ??
//        GroupBillingService.GroupBillingStatus groupBillingStatus = groupBillingService.calcContractsBalance
//        (contracts);
//        if (groupBillingStatus.getStatus() == GroupStatus.DEBT_EXISTS) {
//            throw WebActionException.cons402("User has debt, no new services available");
//        }
        groupsManager.initGroup(group, features.isEdu(), uid, productOwner);
        return group;
    }

    private static UUID parseGroupServiceId(String id) {
        try {
            return UUID.fromString(id);
        } catch (IllegalArgumentException e) {
            throw new NotFoundException("group service not found by '" + id + "'");
        }
    }

    @Getter
    @Builder
    static class SubscriptionCheckRightConfig {
        private boolean skipValidateClientBalance;
        private boolean skipAvailableProductValidation;
    }
}
