package ru.yandex.chemodan.app.psbilling.worker.billing.users;

import java.util.UUID;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;

import lombok.AllArgsConstructor;

import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.psbilling.core.billing.users.processors.OrderProcessorFacade;
import ru.yandex.chemodan.app.psbilling.core.dao.users.OrderDao;
import ru.yandex.chemodan.app.psbilling.core.entities.users.OrderStatus;
import ru.yandex.chemodan.app.psbilling.core.synchronization.PsBillingQueueNames;
import ru.yandex.commune.bazinga.scheduler.CronTask;
import ru.yandex.commune.bazinga.scheduler.ExecutionContext;
import ru.yandex.commune.bazinga.scheduler.TaskQueueName;
import ru.yandex.commune.bazinga.scheduler.schedule.Schedule;
import ru.yandex.commune.bazinga.scheduler.schedule.ScheduleCron;
import ru.yandex.commune.dynproperties.DynamicProperty;
import ru.yandex.misc.time.TimeUtils;

@AllArgsConstructor
public class CheckOrdersInHoldCrone extends CronTask {
    private final DynamicProperty<Integer> checkOrderInHoldThreadsCount =
            new DynamicProperty<>("ps-billing.check-order-in-hold-threads-count", 10);
    private final DynamicProperty<Integer> checkOrderInHoldSelectBatch =
            new DynamicProperty<>("ps-billing.check-order-in-hold-select-batch", 2000);

    private final OrderDao orderDao;
    private final OrderProcessorFacade orderProcessorFacade;

    @Override
    public Schedule cronExpression() {
        return ScheduleCron.parse("40 * * * *", TimeUtils.EUROPE_MOSCOW_TIME_ZONE);
    }

    @Override
    public void execute(ExecutionContext executionContext) {
        ExecutorService exec = Executors.newFixedThreadPool(checkOrderInHoldThreadsCount.get());
        try {
            Option<UUID> from = Option.empty();
            int lastSize;
            do {
                ListF<UUID> orderIds = orderDao.findInStatus(
                        OrderStatus.ON_HOLD,
                        from,
                        checkOrderInHoldSelectBatch.get());

                for (final UUID orderId : orderIds) {
                    exec.submit(() -> orderProcessorFacade.scheduleCheckOrder(orderId, OrderStatus.ON_HOLD));
                }

                lastSize = orderIds.size();
                from = orderIds.lastO();
            } while (lastSize > 0);
        } finally {
            exec.shutdown();
        }
    }

    @Override
    public TaskQueueName queueName() {
        return PsBillingQueueNames.CRON;
    }
}
