package ru.yandex.chemodan.app.psbilling.worker.monitor;

import org.joda.time.Duration;
import org.joda.time.Instant;

import ru.yandex.chemodan.app.psbilling.core.dao.groups.GroupDao;
import ru.yandex.chemodan.app.psbilling.core.dao.groups.GroupServiceDao;
import ru.yandex.chemodan.app.psbilling.core.dao.groups.GroupTrustPaymentRequestDao;
import ru.yandex.chemodan.app.psbilling.core.dao.groups.billing.ClientBalanceDao;
import ru.yandex.chemodan.app.psbilling.core.entities.groups.billing.PaymentRequestStatus;
import ru.yandex.commune.dynproperties.DynamicProperty;
import ru.yandex.misc.monica.annotation.MonicaMetric;

public class BalanceStatusMonitor extends AbstractMonitor {
    public static final Duration MISTAKENLY_ACTIVE_SERVICE_THRESHOLD = Duration.standardHours(2);
    private final DynamicProperty<Integer> autopaySumMonitorHours =
            new DynamicProperty<>("ps-billing.monitoring.autopay_sum.hours", 1);
    private final DynamicProperty<Integer> autorepaySumMonitorHours =
            new DynamicProperty<>("ps-billing.monitoring.autorepay_sum.hours", 1);

    private final ClientBalanceDao clientBalanceDao;
    private final GroupDao groupDao;
    private final GroupServiceDao groupServiceDao;
    private final GroupTrustPaymentRequestDao groupTrustPaymentRequestDao;

    private volatile int prepaidDebtsCount = 0;
    private volatile int mixpaidClientsCount = 0;
    private volatile int mistakenlyActiveServiceCount = 0;
    private volatile double autoPayChargeSum = 0;
    private volatile double autoRePayChargeSum = 0;

    public BalanceStatusMonitor(ClientBalanceDao clientBalanceDao, GroupDao groupDao, GroupServiceDao groupServiceDao, GroupTrustPaymentRequestDao groupTrustPaymentRequestDao) {
        super("balance_status", "balance warnings");
        this.clientBalanceDao = clientBalanceDao;
        this.groupDao = groupDao;
        this.groupServiceDao = groupServiceDao;
        this.groupTrustPaymentRequestDao = groupTrustPaymentRequestDao;
    }

    @MonicaMetric
    public int getMixpaidClientsCount() {
        return mixpaidClientsCount;
    }

    @MonicaMetric
    public int getPrepaidDebtsCount() {
        return prepaidDebtsCount;
    }

    @MonicaMetric
    public int getMistakenlyActiveServiceCount() {
        return mistakenlyActiveServiceCount;
    }

    @MonicaMetric
    public double getAutoPayRecentChargeSum() {
        return autoPayChargeSum;
    }

    @MonicaMetric
    public double getAutoRePayRecentChargeSum() {
        return autoRePayChargeSum;
    }

    @Override
    public void updateState() {
        prepaidDebtsCount = clientBalanceDao.prepaidDebtsCount();
        mixpaidClientsCount = groupDao.mixpaidClientsCount();
        mistakenlyActiveServiceCount =
                groupServiceDao.getMistakenlyActiveServiceCount(MISTAKENLY_ACTIVE_SERVICE_THRESHOLD);
        Instant autoPaySumFrom = Instant.now().minus(Duration.standardHours(autopaySumMonitorHours.get()));
        autoPayChargeSum = groupTrustPaymentRequestDao.getRecentAutoPaymentsSum(autoPaySumFrom, PaymentRequestStatus.SUCCESS).doubleValue();
        Instant autorepaySumFrom = Instant.now().minus(Duration.standardHours(autorepaySumMonitorHours.get()));
        autoRePayChargeSum = groupTrustPaymentRequestDao.getRecentResurrectPaymentsSum(autorepaySumFrom, PaymentRequestStatus.SUCCESS).doubleValue();
    }

    @Override
    public void resetState() {
        prepaidDebtsCount = 0;
        mixpaidClientsCount = 0;
        mistakenlyActiveServiceCount = 0;
        autoPayChargeSum = 0;
        autoRePayChargeSum = 0;
    }
}
