package ru.yandex.chemodan.app.psbilling.worker.monitor;

import org.joda.time.Duration;

import ru.yandex.chemodan.app.psbilling.core.config.Settings;
import ru.yandex.chemodan.app.psbilling.core.dao.cards.TrustCardBindingDao;
import ru.yandex.chemodan.app.psbilling.core.dao.features.GroupServiceFeatureDao;
import ru.yandex.chemodan.app.psbilling.core.dao.features.UserServiceFeatureDao;
import ru.yandex.chemodan.app.psbilling.core.dao.groups.GroupServiceDao;
import ru.yandex.chemodan.app.psbilling.core.dao.groups.GroupServiceMemberDao;
import ru.yandex.chemodan.app.psbilling.core.dao.groups.billing.ClientBalanceDao;
import ru.yandex.chemodan.app.psbilling.core.dao.users.OrderDao;
import ru.yandex.chemodan.app.psbilling.core.dao.users.UserServiceDao;
import ru.yandex.commune.dynproperties.DynamicProperty;
import ru.yandex.misc.monica.annotation.MonicaMetric;

public class PsBillingStaleRecordsMonitor extends AbstractMonitor {
    public static final Duration TASK_DELAY = Duration.standardMinutes(3);
    public static final Duration STALE_PERIOD = Duration.standardMinutes(30);
    public final DynamicProperty<Long> staleBalanceDelayMinutes =
            new DynamicProperty<>("ps-billing.monitoring.stale.balance.delay.minutes", 150L);

    //add one hour to give time to do tasks to cancel service
    private static final Duration MONITOR_DELAY = Duration.standardHours(1);
    private final UserServiceFeatureDao userServiceFeatureDao;
    private final GroupServiceFeatureDao groupServiceFeatureDao;
    private final UserServiceDao userServiceDao;
    private final GroupServiceMemberDao groupServiceMemberDao;
    private final GroupServiceDao groupServiceDao;
    private final OrderDao orderDao;
    private final ClientBalanceDao clientBalanceDao;
    private final TrustCardBindingDao trustCardBindingDao;
    private final Settings settings;

    private volatile int staleUserServiceFeatures = 0;
    private volatile int staleGroupServiceFeatures = 0;
    private volatile int staleUserServices = 0;
    private volatile int staleGroupServiceMembers = 0;
    private volatile int staleGroupServices = 0;
    private volatile int userServicesWithExpiredBillingCheck = 0;
    private volatile int staleOrders = 0;
    private volatile int staleBalances = 0;
    private volatile int staleCardBindings = 0;

    public PsBillingStaleRecordsMonitor(UserServiceFeatureDao userServiceFeatureDao,
                                        GroupServiceFeatureDao groupServiceFeatureDao, UserServiceDao userServiceDao,
                                        GroupServiceMemberDao groupServiceMemberDao, GroupServiceDao groupServiceDao,
                                        OrderDao orderDao, ClientBalanceDao clientBalanceDao,
                                        TrustCardBindingDao trustCardBindingDao, Settings settings) {
        super("stale-records", "Number of records that not actual for too long");
        this.userServiceFeatureDao = userServiceFeatureDao;
        this.groupServiceFeatureDao = groupServiceFeatureDao;
        this.userServiceDao = userServiceDao;
        this.groupServiceMemberDao = groupServiceMemberDao;
        this.groupServiceDao = groupServiceDao;
        this.orderDao = orderDao;
        this.clientBalanceDao = clientBalanceDao;
        this.trustCardBindingDao = trustCardBindingDao;
        this.settings = settings;
    }

    @MonicaMetric
    public int getStaleUserServiceFeatures() {
        return staleUserServiceFeatures;
    }

    @MonicaMetric
    public int getStaleUserServices() {
        return staleUserServices;
    }

    @MonicaMetric
    public int getStaleGroupServiceMembers() {
        return staleGroupServiceMembers;
    }

    @MonicaMetric
    public int getStaleGroupServices() {
        return staleGroupServices;
    }

    @MonicaMetric
    public int getStaleGroupServiceFeatures() {
        return staleGroupServiceFeatures;
    }

    @MonicaMetric
    public int getUserServicesWithExpiredBillingCheck() {
        return userServicesWithExpiredBillingCheck;
    }

    @MonicaMetric
    public int getStaleOrders() {
        return staleOrders;
    }

    @MonicaMetric
    public int getStaleBalances() {
        return staleBalances;
    }

    @MonicaMetric
    public int getStaleCardBindings() {
        return staleCardBindings;
    }

    @Override
    public void updateState() {
        staleUserServiceFeatures = userServiceFeatureDao.countNotActualUpdatedBefore(STALE_PERIOD);
        staleUserServices = userServiceDao.countNotActualUpdatedBefore(STALE_PERIOD);
        staleGroupServiceMembers = groupServiceMemberDao.countNotActualUpdatedBefore(STALE_PERIOD);
        staleGroupServices = groupServiceDao.countNotActualUpdatedBefore(STALE_PERIOD);
        staleGroupServiceFeatures = groupServiceFeatureDao.countNotActualUpdatedBefore(STALE_PERIOD);
        //add one hour to give time to do tasks to cancel service
        userServicesWithExpiredBillingCheck = userServiceDao.countServicesWithCheckBillingDateBefore(
                settings.getAcceptableUserServiceCheckDateExpirationTime().plus(MONITOR_DELAY));
        staleOrders = orderDao.countInitCreatedBefore(
                settings.getAcceptableInitOrdersCheckDateExpirationTime().plus(MONITOR_DELAY));
        staleBalances = clientBalanceDao.countNotActualUpdatedBefore(
                settings.getBalanceStaleIntervalHours().plus(Duration.standardMinutes(staleBalanceDelayMinutes.get())));
        staleCardBindings = trustCardBindingDao.countStaleRecords(settings.getStaleCardBindingIntervalHours());
    }

    @Override
    public void resetState() {
        staleUserServiceFeatures = 0;
        staleUserServices = 0;
        staleGroupServiceMembers = 0;
        staleGroupServices = 0;
        staleGroupServiceFeatures = 0;
        staleOrders = 0;
        userServicesWithExpiredBillingCheck = 0;
        staleBalances = 0;
        staleCardBindings = 0;
    }
}
