package ru.yandex.chemodan.app.psbilling.worker.monitor;

import org.joda.time.Duration;
import org.joda.time.Instant;

import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.psbilling.core.billing.groups.export.BaseExportService;
import ru.yandex.chemodan.app.psbilling.core.billing.groups.export.YtOperations;
import ru.yandex.chemodan.app.psbilling.core.billing.groups.export.distributionplatform.DistributionPlatformTransactionsExportService;
import ru.yandex.chemodan.app.psbilling.core.billing.groups.export.groupservices.GroupServiceTransactionsExportService;
import ru.yandex.misc.monica.annotation.MonicaMetric;

public class YtJobsStatusMonitor extends AbstractMonitor {
    public static final Duration DP_EXPORTS_STALE_TIME = Duration.standardHours(12);
    public static final Duration GS_EXPORTS_STALE_TIME = Duration.standardHours(36);
    private final DistributionPlatformTransactionsExportService dpExportService;
    private final GroupServiceTransactionsExportService gsExportService;
    private volatile boolean distributionPlatformExportStale = false;
    private volatile boolean groupBillingExportStale = false;


    public YtJobsStatusMonitor(DistributionPlatformTransactionsExportService dpExportService,
                               GroupServiceTransactionsExportService gsExportService) {
        super("yt_job_status", "export staleness metrics");
        this.dpExportService = dpExportService;
        this.gsExportService = gsExportService;
    }

    @MonicaMetric
    // не умеем в boolean https://a.yandex-team.ru/arc_vcgetDistributionPlatformExportStales/disk/support/common/src/main/java/ru/yandex/chemodan/util/yasm/MonicaSnapshotAggregatorForYasm.java?rev=r3803886#L88
    public int getDistributionPlatformExportStale() {
        return distributionPlatformExportStale ? 1 : 0;
    }

    @MonicaMetric
    public int getGroupBillingExportStale() {
        return groupBillingExportStale ? 1 : 0;
    }


    @Override
    public void updateState() {
        distributionPlatformExportStale = isStale(dpExportService, DP_EXPORTS_STALE_TIME);
        groupBillingExportStale = isStale(gsExportService, GS_EXPORTS_STALE_TIME);
    }

    @Override
    public void resetState() {
        distributionPlatformExportStale = false;
        groupBillingExportStale = false;
    }

    private boolean isStale(BaseExportService<?, ?> exportService, Duration staleTime) {
        return isStale(exportService.getPrimaryYtOperations(), staleTime)
                || isStale(exportService.getSecondaryYtOperations(), staleTime);
    }

    private boolean isStale(YtOperations<?> ytOperations, Duration staleTime) {
        if (ytOperations.isEnabled()) {
            Option<Instant> exportTime = ytOperations.getCurrentExportInstant();
            return !exportTime.isPresent() || exportTime.get().isBefore(Instant.now().minus(staleTime));
        }
        return false;
    }
}
