package ru.yandex.chemodan.app.queller.admin.tasks;

import org.joda.time.Duration;
import org.joda.time.format.PeriodFormatter;
import org.joda.time.format.PeriodFormatterBuilder;

import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.function.Function0;
import ru.yandex.chemodan.admin.AdminBenderParameterBinder;
import ru.yandex.chemodan.app.queller.admin.CeleryAutocomplete;
import ru.yandex.chemodan.app.queller.admin.CeleryRegistries;
import ru.yandex.chemodan.queller.celery.job.CeleryTask;
import ru.yandex.commune.a3.action.ActionContainer;
import ru.yandex.commune.a3.action.HttpMethod;
import ru.yandex.commune.a3.action.Path;
import ru.yandex.commune.a3.action.parameter.bind.annotation.BindWith;
import ru.yandex.commune.a3.action.parameter.bind.annotation.RequestParam;
import ru.yandex.commune.a3.action.result.NoContent;
import ru.yandex.commune.admin.z.ZAction;
import ru.yandex.commune.admin.z.ZRedirectException;
import ru.yandex.commune.bazinga.impl.TaskId;
import ru.yandex.commune.bazinga.scheduler.ActiveUidBehavior;
import ru.yandex.commune.bazinga.scheduler.ActiveUidDropType;
import ru.yandex.commune.bazinga.scheduler.ActiveUidDuplicateBehavior;
import ru.yandex.commune.bazinga.scheduler.schedule.CompoundReschedulePolicy;
import ru.yandex.commune.bazinga.scheduler.schedule.RescheduleConstant;
import ru.yandex.commune.bazinga.scheduler.schedule.RescheduleExponential;
import ru.yandex.commune.bazinga.scheduler.schedule.RescheduleLinear;
import ru.yandex.commune.bazinga.scheduler.schedule.RescheduleNever;
import ru.yandex.commune.bazinga.scheduler.schedule.ReschedulePolicy;
import ru.yandex.commune.bazinga.scheduler.schedule.ReschedulePolicyWithAttempts;
import ru.yandex.misc.bender.annotation.BenderBindAllFields;
import ru.yandex.misc.bender.annotation.BenderFlatten;
import ru.yandex.misc.bender.annotation.BenderPart;
import ru.yandex.misc.bender.annotation.XmlRootElement;
import ru.yandex.misc.lang.CamelWords;
import ru.yandex.misc.lang.StringUtils;
import ru.yandex.misc.time.TimeUtils;

/**
 * @author dbrylev
 */
@ActionContainer
public class TasksAdminPage {

    private final CeleryRegistries registries;

    public TasksAdminPage(CeleryRegistries registries) {
        this.registries = registries;
    }

    @ZAction(defaultAction = true)
    @Path("/celery-tasks")
    public TasksPojo index() {
        return new TasksPojo(
                registries.tasks().getDefaults(),
                registries.tasks().getAllNonDefaults(),
                registries.autocomplete());
    }

    @Path("/celery-tasks/task")
    public TaskPojo get(@RequestParam("id") String id) {
        return new TaskPojo(registries.tasks().getO(new TaskId(id)).getOrElse(registries.tasks().getDefaults()));
    }

    @Path(value = "/celery-tasks/task", methods = HttpMethod.POST)
    public NoContent save(@BindWith(AdminBenderParameterBinder.class) TaskPojo pojo) {
        registries.tasks().put(pojo.toCeleryTask());

        return NoContent.cons();
    }

    @Path("/celery-tasks/delete")
    public void delete(@RequestParam("id") String id) {
        registries.tasks().remove(new TaskId(id));

        throw new ZRedirectException("/z/celery-tasks");
    }

    @XmlRootElement(name = "content")
    @BenderBindAllFields
    public static class TasksPojo {
        public final TaskPojo taskDefaults;
        @BenderPart(name = "task", wrapperName = "tasks")
        public final ListF<TaskPojo> tasks;
        @BenderFlatten
        public final CeleryAutocomplete celeryAutocomplete;

        public TasksPojo(
                CeleryTask taskDefaults, ListF<CeleryTask> tasks,
                CeleryAutocomplete celeryAutocomplete)
        {
            this.taskDefaults = new TaskPojo(taskDefaults);
            this.tasks = tasks.map(TaskPojo::new);
            this.celeryAutocomplete = celeryAutocomplete;
        }
    }

    @BenderBindAllFields
    public static class TaskPojo {
        public final String id;
        public final String queue;
        public final int priority;

        public final Option<PrettyDuration> softTimeout;
        public final Option<PrettyDuration> hardTimeout;

        @BenderPart(name = "reschedule", wrapperName = "reschedules")
        public final ListF<ReschedulePojo> reschedules;

        public final ActiveUidDropType activeUidDrop;
        public final ActiveUidDuplicateBehavior activeUidOnDuplicate;

        public TaskPojo(CeleryTask task) {
            this.id = task.id.getId();
            this.queue = task.executionQueue;
            this.priority = task.priority;
            this.softTimeout = task.softTimeout.map(PrettyDuration::new);
            this.hardTimeout = task.hardTimeout.map(PrettyDuration::new);

            this.reschedules = ReschedulePojo.flatten(task.reschedulePolicy);
            this.activeUidDrop = task.activeUidBehavior.getDropType();
            this.activeUidOnDuplicate = task.activeUidBehavior.getDuplicateBehavior();
        }

        public CeleryTask toCeleryTask() {
            return new CeleryTask(
                    new TaskId(id), queue, priority,
                    softTimeout.flatMapO(PrettyDuration::getDuration),
                    hardTimeout.flatMapO(PrettyDuration::getDuration),
                    ReschedulePojo.parse(reschedules),
                    new ActiveUidBehavior(activeUidDrop, activeUidOnDuplicate));
        }
    }

    @BenderBindAllFields
    public static class ReschedulePojo {
        public final String type;
        public final int times;
        public final Option<PrettyDuration> delay;

        public ReschedulePojo(String type, int times, Option<PrettyDuration> delay) {
            this.type = type;
            this.times = times;
            this.delay = delay;
        }

        public static ListF<ReschedulePojo> flatten(ReschedulePolicy policy) {
            if (policy instanceof RescheduleConstant
                || policy instanceof RescheduleLinear
                || policy instanceof RescheduleExponential)
            {
                Duration delay;

                if (policy instanceof RescheduleConstant) {
                    delay = ((RescheduleConstant) policy).getTimeout();
                } else if (policy instanceof RescheduleLinear) {
                    delay = ((RescheduleLinear) policy).getTimeout();
                } else {
                    delay = ((RescheduleExponential) policy).getTimeout();
                }
                int times = ((ReschedulePolicyWithAttempts) policy).getAttemptCount();

                if (delay.getStandardSeconds() > 0) {
                    String type = CamelWords.parse(policy.getClass().getSimpleName()).getWords().last();
                    return Option.of(new ReschedulePojo(type, times, Option.of(new PrettyDuration(delay))));

                } else {
                    return Option.of(new ReschedulePojo("immediately", times, Option.empty()));
                }
            } else if (policy instanceof CompoundReschedulePolicy) {
                return ((CompoundReschedulePolicy) policy).getPolicies().flatMap(ReschedulePojo::flatten);

            } else if (policy instanceof RescheduleNever) {
                return Option.empty();

            } else {
                throw new IllegalArgumentException();
            }
        }

        public static ReschedulePolicy parse(ListF<ReschedulePojo> pojos) {
            ListF<ReschedulePolicyWithAttempts> policies = pojos.map(pojo -> {
                Function0<Duration> timeoutF = () -> pojo.delay.flatMapO(PrettyDuration::getDuration)
                        .getOrThrow("Empty reschedule delay");

                switch (pojo.type) {
                    case "immediately":
                        return new RescheduleConstant(Duration.millis(0), pojo.times);
                    case "constant":
                        return new RescheduleConstant(timeoutF.apply(), pojo.times);
                    case "linear":
                        return new RescheduleLinear(timeoutF.apply(), pojo.times);
                    case "exponential":
                        return new RescheduleExponential(timeoutF.apply(), pojo.times);
                }
                throw new IllegalArgumentException("Unknown reschedule type: " + pojo.type);
            });
            if (policies.isEmpty()) {
                return new RescheduleNever();
            } else if (policies.size() > 1) {
                return new CompoundReschedulePolicy(policies.toArray(ReschedulePolicyWithAttempts.class));
            } else {
                return policies.single();
            }
        }
    }

    @BenderBindAllFields
    public static class PrettyDuration {
        public static final PeriodFormatter formatter = new PeriodFormatterBuilder()
                .printZeroAlways()
                .minimumPrintedDigits(2)
                .appendHours().appendSeparator(":")
                .appendMinutes().appendSeparator(":")
                .appendSeconds()
                .toFormatter();

        public final String value;
        public final Option<String> pretty;

        public PrettyDuration(Duration duration) {
            String pretty;
            if (duration.isShorterThan(Duration.standardMinutes(1))) {
                pretty = duration.getMillis() % 1000 > 0
                        ? TimeUtils.duration.toPrettyString(duration)
                        : Long.toString(duration.getStandardSeconds());
                pretty += " seconds";
            } else {
                pretty = TimeUtils.duration.toPrettyString(duration);
            }
            this.value = formatter.print(duration.toPeriod());
            this.pretty = Option.of(pretty);
        }

        public Option<Duration> getDuration() {
            return StringUtils.notEmptyO(value).map(v -> formatter.parsePeriod(v).toStandardDuration());
        }
    }
}
