package ru.yandex.chemodan.app.queller.celery;

import org.joda.time.Duration;
import org.springframework.amqp.core.AcknowledgeMode;
import org.springframework.amqp.core.BindingBuilder;
import org.springframework.amqp.core.Queue;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.annotation.Import;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.function.Function;
import ru.yandex.chemodan.app.queller.celery.routing.CeleryBazingaJobsConsumer;
import ru.yandex.chemodan.app.queller.celery.routing.CeleryExecutionQueues;
import ru.yandex.chemodan.app.queller.celery.routing.CeleryTasksDirector;
import ru.yandex.chemodan.app.queller.celery.routing.JobsListenerAdapter;
import ru.yandex.chemodan.app.queller.celery.settings.queue.CeleryQueueSettingsRegistry;
import ru.yandex.chemodan.app.queller.celery.settings.task.CeleryTaskRegistry;
import ru.yandex.chemodan.app.queller.celery.settings.worker.CeleryWorkerSettingsContextConfiguration;
import ru.yandex.chemodan.log.TskvNdcUtil;
import ru.yandex.chemodan.queller.celery.QuellerQueues;
import ru.yandex.chemodan.queller.celery.control.CeleryControl;
import ru.yandex.chemodan.queller.celery.job.CeleryJob;
import ru.yandex.chemodan.queller.celery.monitoring.CeleryMetrics;
import ru.yandex.chemodan.queller.celery.monitoring.CeleryMetricsContextConfiguration;
import ru.yandex.chemodan.queller.rabbit.PoolListener;
import ru.yandex.chemodan.queller.rabbit.RabbitPool;
import ru.yandex.chemodan.queller.rabbit.RabbitPoolContextConfiguration;
import ru.yandex.commune.bazinga.impl.worker.BazingaHostPort;
import ru.yandex.commune.bazinga.pg.fetcher.PgBazingaJobsFetcher;
import ru.yandex.commune.bazinga.pg.storage.PgBazingaStorage;
import ru.yandex.commune.bazinga.pg.storage.shard.JobsPartitionShardResolver;
import ru.yandex.commune.zk2.ZkPath;
import ru.yandex.commune.zk2.client.ZkManager;
import ru.yandex.misc.spring.Service;
import ru.yandex.misc.support.tl.ThreadLocalHandle;

/**
 * @author dbrylev
 * @author yashunsky
 */
@Configuration
@Import({
        CeleryWorkerSettingsContextConfiguration.class,
        CeleryMetricsContextConfiguration.class,
        RabbitPoolContextConfiguration.class,
})
public class QuellerCeleryContextConfiguration {

    @Bean
    public CeleryTaskRegistry celeryTaskRegistry(@Qualifier("zkRoot") ZkPath zkRoot, ZkManager zkManager) {
        CeleryTaskRegistry registry = new CeleryTaskRegistry(zkRoot.child("celery-tasks"));

        zkManager.addClient(registry);
        return registry;
    }

    @Bean
    public CeleryQueueSettingsRegistry celeryQueueSettingsRegistry(
            @Qualifier("zkRoot") ZkPath zkRoot, ZkManager zkManager)
    {
        CeleryQueueSettingsRegistry registry = new CeleryQueueSettingsRegistry(zkRoot.child("celery-queues"));

        zkManager.addClient(registry);
        return registry;
    }

    @Bean
    public CeleryBazingaJobsConsumer celeryBazingaJobsConsumer(
            PgBazingaJobsFetcher bazingaJobsFetcher,
            CeleryExecutionQueues celeryExecutionQueues,
            CeleryTasksDirector tasksDirector, CeleryTaskRegistry celeryTaskRegistry)
    {
        return new CeleryBazingaJobsConsumer(
                bazingaJobsFetcher, celeryExecutionQueues, celeryTaskRegistry, tasksDirector);
    }

    @Bean
    public CeleryTasksDirector celeryTasksDirector(
            RabbitPool rabbitPool, CeleryTaskRegistry taskRegistry,
            CeleryExecutionQueues executionQueues, PgBazingaStorage storage, JobsPartitionShardResolver jobsPartitioning,
            @Value("${queller.rabbit.connection.sleepOnNoWorkingRabbits}") Duration sleepOnNoWorkingRabbits,
            @Value("${queller.rabbit.pgRejected.sleepOnProfitlessAttempt}") Duration sleepOnProfitlessPgRejected,
            @Value("${queller.rabbit.started_completed.sleepOnProfitlessAttempt}") Duration sleepOnProfitlessStartCompleted,
            @Value("${queller.rabbit.started_completed.handlerTimeout}") Duration handlerTimeout,
            BazingaHostPort bazingaWorkerId,
            CeleryMetrics celeryMetrics)
    {
        rabbitPool.declareExchange(QuellerQueues.EXECUTE_EXCHANGE);
        rabbitPool.declareExchange(QuellerQueues.SUBMIT_EXCHANGE);

        rabbitPool.declareQueue(QuellerQueues.SUBMIT_QUEUE);
        rabbitPool.declareQueue(QuellerQueues.SECONDARY_SUBMIT_QUEUE);
        rabbitPool.declareQueue(QuellerQueues.STARTED_QUEUE);
        rabbitPool.declareQueue(QuellerQueues.COMPLETED_QUEUE);

        rabbitPool.declareQueue(QuellerQueues.STARTED_REJECTED_QUEUE);
        rabbitPool.declareQueue(QuellerQueues.COMPLETED_REJECTED_QUEUE);

        rabbitPool.declareBinding(BindingBuilder.bind(QuellerQueues.SUBMIT_QUEUE)
                .to(QuellerQueues.SUBMIT_EXCHANGE).withQueueName());
        rabbitPool.declareBinding(BindingBuilder.bind(QuellerQueues.SECONDARY_SUBMIT_QUEUE)
                .to(QuellerQueues.SUBMIT_EXCHANGE).withQueueName());
        rabbitPool.declareBinding(BindingBuilder.bind(QuellerQueues.STARTED_QUEUE)
                .to(QuellerQueues.SUBMIT_EXCHANGE).withQueueName());
        rabbitPool.declareBinding(BindingBuilder.bind(QuellerQueues.COMPLETED_QUEUE)
                .to(QuellerQueues.SUBMIT_EXCHANGE).withQueueName());

        rabbitPool.declareBinding(BindingBuilder.bind(QuellerQueues.STARTED_REJECTED_QUEUE)
                .to(QuellerQueues.SUBMIT_EXCHANGE).withQueueName());
        rabbitPool.declareBinding(BindingBuilder.bind(QuellerQueues.COMPLETED_REJECTED_QUEUE)
                .to(QuellerQueues.SUBMIT_EXCHANGE).withQueueName());

        CeleryTasksDirector.PG_REJECTED_QUEUES.get1().forEach(queue -> {
            rabbitPool.declareQueue(queue);

            rabbitPool.declareBinding(BindingBuilder.bind(queue)
                    .to(QuellerQueues.EXECUTE_EXCHANGE).withQueueName());
        });

        return new CeleryTasksDirector(
                rabbitPool, taskRegistry, executionQueues, storage, jobsPartitioning, bazingaWorkerId,
                sleepOnNoWorkingRabbits, sleepOnProfitlessPgRejected, sleepOnProfitlessStartCompleted,
                handlerTimeout, celeryMetrics);
    }

    @Bean
    public PoolListener celerySubmittedTasksListenerContainer(
            RabbitPool rabbitPool,
            CeleryTasksDirector celeryTaskDirector,
            @Value("${queller.rabbit.submit.prefetchCount}") int prefetchCount,
            @Value("${queller.rabbit.submit.txSize}") int txSize,
            @Value("${queller.rabbit.submit.consumers}") int consumers,
            @Value("${queller.rabbit.submit.maxConsumers}") int maxConsumers,
            @Value("${queller.rabbit.submit.receive.timeout}") Duration receiveTimeout)
    {
        ListenerSettings settings = new ListenerSettings(
                prefetchCount, txSize, consumers, maxConsumers, receiveTimeout);

        return celerySubmittedAndPgRejectedTasksListenerContainer(
                rabbitPool, celeryTaskDirector, settings, QuellerQueues.SUBMIT_QUEUE, Option.empty());
    }

    @Bean
    public PoolListener celerySecondarySubmittedTasksListenerContainer(
            RabbitPool rabbitPool,
            CeleryTasksDirector celeryTaskDirector,
            @Value("${queller.rabbit.secondary.submit.prefetchCount}") int prefetchCount,
            @Value("${queller.rabbit.secondary.submit.txSize}") int txSize,
            @Value("${queller.rabbit.secondary.submit.consumers}") int consumers,
            @Value("${queller.rabbit.secondary.submit.maxConsumers}") int maxConsumers,
            @Value("${queller.rabbit.secondary.submit.receive.timeout}") Duration receiveTimeout)
    {
        ListenerSettings settings = new ListenerSettings(
                prefetchCount, txSize, consumers, maxConsumers, receiveTimeout);

        return celerySubmittedAndPgRejectedTasksListenerContainer(
                rabbitPool, celeryTaskDirector, settings, QuellerQueues.SECONDARY_SUBMIT_QUEUE, Option.empty());
    }

    @Bean
    public Service celeryPgRejectedTasksListeners(
            RabbitPool rabbitPool,
            CeleryTasksDirector celeryTasksDirector,
            @Value("${queller.rabbit.pgRejected.prefetchCount}") int prefetchCount,
            @Value("${queller.rabbit.pgRejected.txSize}") int txSize,
            @Value("${queller.rabbit.pgRejected.consumers}") int consumers,
            @Value("${queller.rabbit.pgRejected.maxConsumers}") int maxConsumers,
            @Value("${queller.rabbit.pgRejected.receive.timeout}") Duration receiveTimeout)
    {
        ListenerSettings settings = new ListenerSettings(
                prefetchCount, txSize, consumers, maxConsumers, receiveTimeout);

        ListF<PoolListener> listeners = CeleryTasksDirector.PG_REJECTED_QUEUES.get1().map(queue ->
                celerySubmittedAndPgRejectedTasksListenerContainer(
                        rabbitPool, celeryTasksDirector, settings, queue, Option.of(queue)));

        return new Service() {
            public void start() { }

            public void stop() { listeners.forEach(PoolListener::stop); }
        };
    }

    private PoolListener celerySubmittedAndPgRejectedTasksListenerContainer(
            RabbitPool rabbitPool, CeleryTasksDirector tasksDirector,
            ListenerSettings settings, Queue listeningQueue, Option<Queue> rejectedQueue)
    {
        return celeryTasksListenerContainer(
                rabbitPool, settings, listeningQueue,
                jobs -> tasksDirector.handleSubmit(jobs, rejectedQueue, false));
    }

    @Bean
    public Service celeryStartedCompletedTasksListeners(
        RabbitPool rabbitPool,
        CeleryTasksDirector tasksDirector,
        @Value("${queller.rabbit.started_completed.prefetchCount}") int prefetchCount,
        @Value("${queller.rabbit.started_completed.txSize}") int txSize,
        @Value("${queller.rabbit.started_completed.consumers}") int consumers,
        @Value("${queller.rabbit.started_completed.maxConsumers}") int maxConsumers,
        @Value("${queller.rabbit.started_completed.receive.timeout}") Duration receiveTimeout,
        @Value("${queller.rabbit.pgRejected.started_completed.prefetchCount}") int rejectedPrefetchCount,
        @Value("${queller.rabbit.pgRejected.started_completed.txSize}") int rejectedTxSize,
        @Value("${queller.rabbit.pgRejected.started_completed.consumers}") int rejectedConsumers,
        @Value("${queller.rabbit.pgRejected.started_completed.maxConsumers}") int rejectedMaxConsumers,
        @Value("${queller.rabbit.pgRejected.started_completed.receive.timeout}") Duration rejectedReceiveTimeout)
    {
        ListenerSettings mainSettings = new ListenerSettings(
                prefetchCount, txSize, consumers, maxConsumers, receiveTimeout);
        ListenerSettings rejectedSettings = new ListenerSettings(
                rejectedPrefetchCount, rejectedTxSize, rejectedConsumers, rejectedMaxConsumers, rejectedReceiveTimeout);

        ListF<PoolListener> main = StartedOrCompleted.getMain().map(
                soc -> startedOrCompletedPoolListenerWithReject(rabbitPool, tasksDirector, mainSettings, soc));
        ListF<PoolListener> rejected = StartedOrCompleted.getRejected().map(
                soc -> startedOrCompletedPoolListenerWithReject(rabbitPool, tasksDirector, rejectedSettings, soc));

        ListF<PoolListener> listeners = main.plus(rejected);

        return new Service() {
            public void start() { }

            public void stop() { listeners.forEach(PoolListener::stop); }
        };

    }

    private PoolListener startedOrCompletedPoolListenerWithReject(
            RabbitPool rabbitPool,
            CeleryTasksDirector tasksDirector,
            ListenerSettings listenerSettings, StartedOrCompleted startedOrCompleted)
    {
        Queue mainQueue = startedOrCompleted.inputQueue;
        Queue rejectQueue = startedOrCompleted.rejectQueue;
        Function<CeleryJob, Boolean> handler = startedOrCompleted.handlerExtractor.apply(tasksDirector);

        return celeryTasksListenerContainer(
                rabbitPool, listenerSettings, mainQueue,
                jobs -> tasksDirector.handleStartOrComplete(jobs, handler, rejectQueue));
    }

    private PoolListener celeryTasksListenerContainer(
            RabbitPool rabbitPool, ListenerSettings listenerSettings,
            Queue queue, Function<ListF<CeleryJob>, ListF<Boolean>> listener)
    {
        PoolListener container = rabbitPool.createListener();
        container.setExclusive(false);
        container.setQueues(queue);

        container.setConcurrentConsumers(listenerSettings.consumers);
        container.setMaxConcurrentConsumers(listenerSettings.maxConsumers);
        container.setPrefetchCount(listenerSettings.prefetchCount);
        container.setTxSize(listenerSettings.txSize);
        container.setReceiveTimeout(listenerSettings.receiveTimeout.getMillis());

        container.setMessageListener(new JobsListenerAdapter(jobs -> {
            ThreadLocalHandle h = TskvNdcUtil.pushToNdc("queue", queue.getName());
            try {
                return listener.apply(jobs);
            } finally {
                h.popSafely();
            }
        }));

        container.setAcknowledgeMode(AcknowledgeMode.MANUAL);

        container.start();

        return container;
    }

    @Bean
    public CeleryExecutionQueues celeryExecutionQueues(
            RabbitPool rabbitPool,
            CeleryQueueSettingsRegistry queueSettingsRegistry,
            CeleryMetrics celeryMetrics)
    {
        return new CeleryExecutionQueues(
                QuellerQueues.EXECUTE_EXCHANGE, rabbitPool,
                queueSettingsRegistry, celeryMetrics);
    }

    @Bean
    public CeleryControl celeryControl(RabbitPool rabbitPool,
            @Value("${queller.celery.outputExchangeName}") String outputExchangeName,
            @Value("${queller.celery.inputExchangeName}") String inputExchangeName,
            @Value("${queller.celery.controlMessageTtl}") Duration controlMessageTtl,
            @Value("${queller.celery.workerReplyTtl}") Duration workerReplyTtl,
            @Value("${queller.celery.serviceQueuesXExpires}") Duration serviceQueuesXExpires)
    {
        return new CeleryControl(rabbitPool, outputExchangeName, inputExchangeName,
                controlMessageTtl, workerReplyTtl, serviceQueuesXExpires);
    }

    private static class ListenerSettings {
        final int prefetchCount;
        final int txSize;
        final int consumers;
        final int maxConsumers;
        final Duration receiveTimeout;

        public ListenerSettings(int prefetchCount, int txSize, int consumers, int maxConsumers, Duration receiveTimeout) {
            this.prefetchCount = prefetchCount;
            this.txSize = txSize;
            this.consumers = consumers;
            this.maxConsumers = maxConsumers;
            this.receiveTimeout = receiveTimeout;
        }
    }

    private enum StartedOrCompleted {
        STARTED(QuellerQueues.STARTED_QUEUE,
                QuellerQueues.STARTED_REJECTED_QUEUE,
                director -> director::handleStarted),
        STARTED_REJECTED(QuellerQueues.STARTED_REJECTED_QUEUE,
                QuellerQueues.STARTED_REJECTED_QUEUE,
                director -> director::handleStarted),

        COMPLETED(QuellerQueues.COMPLETED_QUEUE,
                QuellerQueues.COMPLETED_REJECTED_QUEUE,
                director -> director::handleCompleted),
        COMPLETED_REJECTED(QuellerQueues.COMPLETED_REJECTED_QUEUE,
                  QuellerQueues.COMPLETED_REJECTED_QUEUE,
                  director -> director::handleCompleted);

        static ListF<StartedOrCompleted> getMain() {
            return Cf.list(STARTED, COMPLETED);
        }

        static ListF<StartedOrCompleted> getRejected() {
            return Cf.list(STARTED_REJECTED, COMPLETED_REJECTED);
        }

        final Queue inputQueue;
        final Queue rejectQueue;
        Function<CeleryTasksDirector, Function<CeleryJob, Boolean>> handlerExtractor;

        StartedOrCompleted(Queue inputQueue, Queue rejectQueue,
                Function<CeleryTasksDirector, Function<CeleryJob, Boolean>> handlerExtractor)
        {
            this.inputQueue = inputQueue;
            this.rejectQueue = rejectQueue;
            this.handlerExtractor = handlerExtractor;
        }
    }
}
